/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef HTML_FONT_HH
#define HTML_FONT_HH

#include "html/Attrib.hh"
#include <string>

namespace html {
  class writer;


  /**  Text font attribute class.
    *  @brief Text font attribute class.
    *  @author John Zweizig
    *  @version 1.1; Last modified March 4, 2008
    *  @ingroup IO_html
    */
  class font : public attrib {
  public:

    /**  Construct a font attribute object that specifies a default 
      *  font.
      *  @brief Default constructor.
      */
    font(void);

    /**  Construct a font attribute object that specifies a font by name.
      *  @brief Named font constructor.
      *  @param face Font name
      */
    font(const std::string& face);

    /**  Destroy an font object.
      *  @brief Destructor.
      */
    ~font(void);

    /**  Make an identical copy of the alignment object.
      *  @brief Clone alignment.
      *  @return Pointer to the cloned alignment object.
      */
    font* clone(void) const;

    /**  Test whether the font is the default value.
      *  @return true if default font.
      */
    bool isDefault(void) const;

    /**  Write the font attribute to the specified html writer.
      *  @brief Write the font.
      *  @param w Html writer reference.
      */
    void putAttr(writer& w) const;

    /**  Test whether this font is equal to the specified.
      *  @brief Compare operator.
      *  @param x Alignment object to be compared
      *  @return True if this font is equal to the argument.
      */
     bool operator==(const font& x) const;

    /**  Set the font to the default.
      *  @brief Set default font.
      */
    void setDefault(void);

    /**  Set the font face to the specified name.
      *  @brief Set the font face.
      *  @param face Font face name.
      */
    void setFace(const std::string& face);

    /**  Get the font face name.
      *  @brief Get the face name.
      *  @return Constant character pointer to null-terminated face name string.
      */
    const char* getFace(void) const;

  private:
    std::string mFace;
  };

}

inline bool
html::font::operator==(const html::font& x) const {
    return mFace == x.mFace;
}

inline const char*
html::font::getFace(void) const {
    return mFace.c_str();
}

#endif //  HTML_FONT_HH
