/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: framefast						*/
/*                                                         		*/
/* Module Description: Frame reader using TOC for fast access		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 26Sep00  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: framefast.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_FRAMEFAST_H
#define _LIGO_FRAMEFAST_H


#include <iosfwd>
#include <deque>
#include <vector>
#include "Time.hh"
#include "Interval.hh"
#include "framefast/frametype.hh"
#include "framefast/frameio.hh"

#if defined(__SUNPRO_CC) && defined(G__DICTIONARY)
   namespace framefast {}
   using namespace framefast;
#endif



namespace framefast {

/** @page framefast
    This header exports a frame reader for fast reading and a frame
    writer for fast writing.
   
    @memo Fast frame reader
    @author Written September 2000 by Daniel Sigg
    @version 1.0
    @ingroup IO_framefast
 ************************************************************************/

/** Fast frame reader. This frame reader uses the table of content (TOC)
    of a frame to extract data directly from the file or memory object.
    It uses memory mapped files for fast IO throughput.Performance
    has been optimized for reading small number of channels from large
    frame files which are stored on disk. Only frames version 4.0 or 
    later which include a TOC are supported.
    @memo Fast frame reader.
    @author Written September 2000 by Daniel Sigg
    @version 1.0
    @ingroup IO_framefast
 ************************************************************************/
   class framereader {
   private:
      framereader (const framereader&);
      framereader& operator= (const framereader&);
   
   public:
      /** Constructs a fast frame reader.
          @memo Default constructor.
       ******************************************************************/
      framereader ();
      /** Destructs a fast frame reader.
          @memo Destructor.
       ******************************************************************/
      virtual ~framereader();
   
      /** Loads a file into the fast frame reader (the file is not 
          actually read, just mapped).
          @memo Load a file.
          @param filename Name of file
          @param map Map the file into memory rather than read it
          @return true if successful
       ******************************************************************/
      bool loadFile (const char* filename, bool map = true);
      /** Loads a memory object into the fast frame reader.
          @memo Load a memory object.
          @param fdata Pointer to frame data
          @param len Length of data array (in bytes)
          @param ownit The framereader will own the data
          @return true if successful
       ******************************************************************/
      bool loadFrame (const void* fdata, int len, bool ownit = false);
      /** Loads a frame into the fast frame reader. The frame storage
          object is adopted.
          @memo Load a frame object.
          @param frame frame object
          @return true if successful
       ******************************************************************/
      bool loadFrame (basic_frame_storage* frame) {
         frame_storage_ptr ptr (frame); 
         return loadFrame (ptr); }
      /** Loads a frame into the fast frame reader. The frame storage
          object is adopted.
          @memo Load a frame object.
          @param frame frame object pointer
          @return true if successful
       ******************************************************************/
      bool loadFrame (frame_storage_ptr& frame) {
         fFrame = frame;
         return (fFrame.data() != 0); }
      /** Releases a previously loaded frame form the fast frame reader. 
          A frame storage pointer object is returned.
          @memo Release a frame object.
          @param frame frame object pointer (return)
          @return true if successful
       ******************************************************************/
      bool releaseFrame (frame_storage_ptr& frame);
   
      /** Unloads a frame form the fast frame reader.
          @memo Unload a frame.
          @return void
       ******************************************************************/
      void unload ();
   
      /** Reads the frame header and checks for frame format.
          @memo Checks frame.
          @return true if frame file, false otherwise
       ******************************************************************/
      bool isFrame ();
   
      /** Get the dictionary.
          @memo Get dictionary.
          @param dict Dictionary (return)
          @return true if successful
       ******************************************************************/
      bool getDict (dict_t& dict);
   
      /** Get a frame header.
          @memo Get a frame header.
          @param h Frame header
          @param fnum Frame number within file
          @return true if successful
       ******************************************************************/
      bool getFrameHeader (frameheader_t& h, int fnum = 0);
   
      /** Get detector information.
          @memo Get detector information.
          @param det Detector information
          @param fnum Frame number within file
          @param dnum Detector number
          @return true if successful
       ******************************************************************/
      bool getDetectorInfo (detector_t& det, int fnum = 0, int dnum = 0);
      /** Get number of detector information.
          @memo Get number of detectors.
          @return Number of detectors
       ******************************************************************/
      int getDetectorInfoNum();
   
      /** Get history information.
          @memo Get history information.
          @param hist History information
          @param fnum Frame number within file
          @return true if successful
       ******************************************************************/
      bool getHistoryInfo (hist_t& hist, int fnum = 0);
   
      /** Get raw data information.
          @memo Get raw data information.
          @param raw Raw data information
          @param fnum Frame number within file
          @return true if successful
       ******************************************************************/
      bool getRawDataInfo (rawdata_t& raw, int fnum = 0);
   
      /** Get data from the specified position.
          @memo Get data.
          @param dat Data record
          @param pos Position within frame
          @param dtype Data type
          @param cpy Copy data?
          @return true if successful
       ******************************************************************/
      bool getData (data_t& dat, int_8u_t pos, datatype_t dtype,
                   frvect_t::datacopy cpy = frvect_t::fv_copy);
   
      /** Get data for the specified channel and frame number.
          @memo Get data.
          @param dat Data record
          @param channel Channel name
          @param nframe Frame number
          @param cpy Copy data?
          @return true if successful
       ******************************************************************/
      bool getData (data_t& dat, const char* channel, int nframe = 0,  
                   frvect_t::datacopy cpy = frvect_t::fv_copy);
   
      /** Copies channel data into a float array. A return value of    
          -1 indicates that the frame does not contain a TOC; a return
          value of 0 indicates that the channel name was not found; a
          positive return value repesents the number of data points
          associated with this channel. The returned number of points
          can be larger than the maximum specified, however, only
          max points are actually copied into the array. If the array
          pointer is 0, just the number of points is returned.
          @memo Get float data.
          @param chnname Name of channel
          @param x data array (destination)
          @param max maximum length of data array
          @return number of data points, or <=0 on error
       ******************************************************************/
      int copy (const char* chnname, real_4_t* x, int max);

      /** Copies channel data into a double array. A return value of    
          -1 indicates that the frame does not contain a TOC; a return
          value of 0 indicates that the channel name was not found; a
          positive return value repesents the number of data points
          associated with this channel. The returned number of points
          can be larger than the maximum specified, however, only
          max points are actually copied into the array. If the array
          pointer is 0, just the number of points is returned.
          @memo Get double data.
          @param chnname Name of channel
          @param x data array (destination)
          @param max maximum length of data array
          @return number of data points, or <=0 on error
       ******************************************************************/
      int copy (const char* chnname, real_8_t* x, int max);

      /** Copies channel data into a short array. A return value of    
          -1 indicates that the frame does not contain a TOC; a return
          value of 0 indicates that the channel name was not found; a
          positive return value repesents the number of data points
          associated with this channel. The returned number of points
          can be larger than the maximum specified, however, only
          max points are actually copied into the array. If the array
          pointer is 0, just the number of points is returned.
          @memo Get short data.
          @param chnname Name of channel
          @param x data array (destination)
          @param max maximum length of data array
          @return number of data points, or <=0 on error
       ******************************************************************/
      int copy (const char* chnname, int_2s_t* x, int max);

      /** Copies channel data into an integer array. A return value of    
          -1 indicates that the frame does not contain a TOC; a return
          value of 0 indicates that the channel name was not found; a
          positive return value repesents the number of data points
          associated with this channel. The returned number of points
          can be larger than the maximum specified, however, only
          max points are actually copied into the array. If the array
          pointer is 0, just the number of points is returned.
          @memo Get int data.
          @param chnname Name of channel
          @param x data array (destination)
          @param max maximum length of data array
          @return number of data points, or <=0 on error
       ******************************************************************/
      int copy (const char* chnname, int_4s_t* x, int max);
   
      /** Writes the TOC to an output stream.
          @memo Write TOC.
          @param os Output stream
          @return Output stream
       ******************************************************************/
      std::ostream& writeTOC (std::ostream& os);
   
      /** Writes the file header to an output stream.
          @memo Write file header.
          @param os Output stream
          @return Output stream
       ******************************************************************/
      std::ostream& writeFileHeader (std::ostream& os);
   
      /** Gets the length of the frame.
          @memo Length of frame.
          @return Length of frame
       ******************************************************************/
      int length() const {
         return fFrame.size(); }
      /** Gets the pointer to the frame.
          @memo Pointer of frame.
          @return Frame pointer
       ******************************************************************/
      const char* frame() const {
         return fFrame.data(); }
      /** Gets the pointer to the end of the frame.
          @memo Pointer of frame end.
          @return End of frame pointer
       ******************************************************************/
      const char* frameend() const {
         return frame() + length(); }
      /** Gets the original name of the frame storage object.
          @memo Origianl name of frame.
          @return Origianl name of frame
       ******************************************************************/
      const char* fname() const {
         return fFrame.name(); }
      /** Guess the basic filename from the detector information and
          the GPS time (doesn't include directory nor extension).
          Uses the original name if set.
          @memo Guess filename.
          @return Filename
       ******************************************************************/
      std::string guessFilename();
   
      /** Gets the number of frames in the file.
          @memo Number of frames.
          @return Number of frames
       ******************************************************************/
      int nframe();
      /** Gets the start time of a frame.
          @memo Start time of a frame.
          @param framenum Frame number
          @return Start time
       ******************************************************************/
      Time starttime (int framenum = 0);
      /** Gets the duration of a frame.
          @memo Duration of a frame.
          @param framenum Frame number
          @return Duration
       ******************************************************************/
      Interval duration (int framenum = 0);
      /** Gets the time of the next frame (in the next file).
          @memo time of next frame.
          @return Next time
       ******************************************************************/
      Time nexttime ();
      /** Gets the TOC.
          @memo Get the TOC.
          @return TOC
       ******************************************************************/
      const toc_t* getTOC();
      /** Enforce a rescan of the TOC.
          @memo Rescan the TOC.
          @return void
       ******************************************************************/
      void setScanTOC (bool set = true) {
         fForceScanToc = set; }
   
   protected:
      /// TOC status
      enum toc_status_t {
      /// TOC not yet checked
      kTOCUndefined = 0,
      /// no TOC
      kTOCNo,
      /// has TOC
      kTOCYes
      };
      /// file header status
      enum header_status_t {
      /// file header not yet checked
      kFileHeaderUndefined = 0,
      /// invalid file header (frame)
      kFileHeaderInvalid,
      /// valid file header
      kFileHeaderValid
      };
   
      /// Debug flag
      bool 		fDebug;
      /// Frame data
      frame_storage_ptr fFrame;
      /// Pointer to toc
      toc_t*		fTOC;
      /// Frame TOC status
      toc_status_t	fHasTOC;
      /// Force to rescan TOC
      bool		fForceScanToc;
      /// Pointer to header
      fileheader_t*	fFileHeader;
      /// Frame header status
      header_status_t	fHasFileHeader;
      /// Pointer to frameheader
      frameheader_t*	fFrameHeader;
      /// Need to swap? (only valid after header was read successful)
      bool		fSwap;
   
      /// new toc structure
      bool newTOC();
      /// free TOC structure
      bool freeTOC();
      /// read TOC structure
      bool readTOC();
      /// regenerate TOC from raw frame file
      bool scanTOC();
   
      /// new header structure
      bool newFileHeader();
      /// free header
      bool freeFileHeader();
      /// read header
      bool readFileHeader();
   
      /// look up channel in TOC
      int lookup (const char* chnname, int_8u_t*& ofs, datatype_t& dtype);
   };


/** Output operator for writing a list of frame structures.
    @memo Output operator.
    @param os Output stream
    @param fr Frame reader
    @return Output stream
 ************************************************************************/
   std::ostream& operator<< (std::ostream& os, const framereader& fr);


/** Fast frame writer.
    @memo Fast frame reader.
    @author Written September 2000 by Daniel Sigg
    @version 1.0
    @ingroup IO_framefast
 ************************************************************************/
   class framewriter {
   private:
      framewriter (const framewriter&);
      framewriter& operator= (const framewriter&);
   
   public:
      /// frame position
      enum framepos {
      /// beginning of frame
      framebegin = 0,
      /// while writting a frame
      framebusy,
      /// between frames
      framebreak,
      /// end of frame
      frameend
      };
   
      /// Frame buffer (data container with smart pointer)
      class framebuffer_t {
      public:
         /// Initializes the frame buffer
         explicit framebuffer_t (int size = 0);
         /// Copy constructor
         framebuffer_t (const framebuffer_t& buf);
         /// Destructor
         ~framebuffer_t ();
         /// Assignment operator
         framebuffer_t& operator= (const framebuffer_t& buf);
      	 /// Allocate an empty memory buffer
         bool allocate (int size);
      	 /// Deallocate memory buffer
         void deallocate ();
      
         /// data pointer
         const char* data() const {
            return fData; }
      	 /// position
         char* pos() const {
            return fData ? fData + fLen : 0; }
         /// length of memory buffer
         int len() const {
            return fLen; }
         /// Size of memory buffer
         int size() const {
            return fSize; }
         /// increase buffer length
         void expand (int len) {
            fLen += len; }
      protected:
         /// Length of frame buffer
         int		fLen;
      	 /// Allocated size of frame buffer
         int		fSize;
         /// Pointer to frame buffer
         char*		fData;
         /// Own data
         mutable bool 	fOwn;
      };
      /// Set of frame buffers
      typedef std::deque<framebuffer_t> framebufferlist;
      /// Set of detector information
      typedef std::vector<detector_t> detector_array_t;
   
   
      /** Constructs a fast frame writer.
          @param nFrameLen frame length in sec
          @param nFrame Number of frames per file
          @param compress Compress data vectors
          @param version Frame specification version
          @memo Default constructor.
       ******************************************************************/
      explicit framewriter (int nFrameLen = 1, int nFrame = 1,
                        int compress = 0, 
                        int version = kDefaultFrameVersion);
      /** Destructs a fast frame writer.
          @memo Destructor.
       ******************************************************************/
      ~framewriter ();
   
      /** Returns the status of the frame writer.
          @memo Status of frame writer.
          @return status (frame position)
       ******************************************************************/
      framepos status() {
         return fFramePos; }
      /** Set the frame format.
          @memo Set the frame format.
          @param nFrame Number of frames per file
          @param nFrameLen Length of each frames (in sec)
          @param compress Compress data vectors
          @param version Frame specification version
          @return true if successful
       ******************************************************************/
      bool setFormat (int nFrameLen = 1, int nFrame = 1, 
                     int compress = 0,
                     int version = kDefaultFrameVersion);
      /** Returns the length of frames.
          @memo Length of frames.
          @return Length of frames in sec
       ******************************************************************/
      int frameLength () const {
         return fFrameLen; }
      /** Set the time of the next frame.
          @memo Set the frame start time.
          @param time Frame start time
          @return true if successful
       ******************************************************************/
      bool setTime (const Time& time);
   
      /** Set the detector information.
          @memo Set the detector information.
          @param det Detector information
          @return true if successful
       ******************************************************************/
      bool setDetectorInfo (const detector_t& det);
      /** Add a detector information.
          @memo Add a detector information.
          @param det Detector information
          @return true if successful
       ******************************************************************/
      bool addDetectorInfo (const detector_t& det);
      /** Get the detector information.
          @memo Get the detector information.
          @param dnum Detector number
          @return pointer to detector information if successful
       ******************************************************************/
      const detector_t* getDetectorInfo (int dnum = 0) const {
         return ((dnum >= 0) && (dnum < (int)fDetector.size())) ?
            &fDetector[0] : 0; }
      /** Get the table of contents.
          @memo Get the table of contents.
          @return pointer to TOC information if successful
       ******************************************************************/
      const toc_t* getTOC () const {
         return &fTOC; }
      /** Gets the number of frames in the file.
          @memo Number of frames.
          @return Number of frames
       ******************************************************************/
      int nframe() const;
      /** Gets the start time of a frame.
          @memo Start time of a frame.
          @param framenum Frame number
          @return Start time
       ******************************************************************/
      Time starttime (int framenum = 0) const;
      /** Gets the duration of a frame.
          @memo Duration of a frame.
          @param framenum Frame number
          @return Duration
       ******************************************************************/
      Interval duration (int framenum = 0) const;
      /** Gets the time of the next frame (in the next file)
          @memo time of next frame.
          @return Next time
       ******************************************************************/
      Time nexttime () const;
   
      /** Add an ADC data vector to the frame.
          @memo Add ADC data.
          @param adc ADC structure
          @param vect Vector structure containing the data
          @return true if successful
       ******************************************************************/
      bool addData (const adcdata_t& adc, const frvect_t& vect);
   
      /** Moves to the next frame. This will write the current frame
          into a memory buffer and add it to the frame buffers. During
          the first call the frame file header gets written first,
          whereas during the last call the table of contents and the 
          end-of-file marker is appended. The time is increased by the 
	  frame length.
          @memo Move to the next frame.
          @return true if successful
       ******************************************************************/
      bool next();

      /** Moves to the next frame. This will write the current frame
          into a memory buffer and add it to the frame buffers. During
          the first call the frame file header gets written first,
          whereas during the last call the table of contents and the 
          end-of-file marker is appended. The argument specifies for the 
	  start time of the next frame.
          @memo Move to the next frame.
          @param nextframe Start time of next frame
          @return true if successful
       ******************************************************************/
      bool next (const Time& nextframe);
   
      /** Writes the frame buffers to an output. This will purge the 
          buffers from the frame writer.
          @memo Write a frame.
          @return true if successful
       ******************************************************************/
      bool write (basic_frameout* out);
      /** Get the frame buffers. This will automatically purge
          the buffers from the frame writer and transfer ownership
          of the buffers to the caller.
          @memo Get the frame buffers.
          @return true if successful
       ******************************************************************/
      bool getBuffers (framebufferlist& bufs) {
         bufs = fFrameBuffer; purge(); 
         return true; }
   
      /** Get the number of frame buffers already written.
          @memo Get the number of buffers.
          @return Number of buffers
       ******************************************************************/
      int buffers() const {
         return fFrameBuffer.size(); }
      /** Get a pointer to the frame buffer.
          @memo Get the buffer pointer.
          @param buffer Buffer number
          @return Buffer size
       ******************************************************************/
      const char* framebuffer (int buffer) const {
         return ((buffer >= 0) && (buffer < (int)fFrameBuffer.size())) ? 
            fFrameBuffer[buffer].data() : 0; }
      /** Get the length of the frame buffer.
          @memo Get the frame buffer length.
          @param buffer Buffer number
          @return Buffer size
       ******************************************************************/
      int bufferlen (int buffer) const {
         return ((buffer >= 0) && (buffer < (int)fFrameBuffer.size())) ? 
            fFrameBuffer[buffer].len() : 0; }
      /** Get the total number of bytes written so far.
          @memo Total frame size.
          @return Number of bytes written
       ******************************************************************/
      int total() const {
         return fTotal; }
      /** Purge the current frame buffers.
          @memo Purge frame buffers.
       ******************************************************************/
      void purge () {
         fFrameBuffer.clear(); }
   
      /** Set the output file name.
          @memo Set filename.
          @param filename Name of file
       ******************************************************************/
      void setFilename (const char* filename) {
         fFilename = filename; }
      /** Get the output file name.
          @memo Get filename.
          @return Name of file
       ******************************************************************/
      const char* getFilename () const {
         return fFilename.c_str(); }
      /** Guess the basic filename from the detector information and
          the GPS time (doesn't include directory nor extension).
          @memo Guess filename.
          @return Filename
       ******************************************************************/
      std::string guessFilename() const;
      /** Set the frame number.
          @memo Set the frame number.
          @param num frame number
       ******************************************************************/
      void setFrameNum (int num) {
         fFrameNum = num; }
      /** Get the frame number.
          @memo Get frame number.
          @return frame number
       ******************************************************************/
      int getFrameNum () const {
         return fFrameNum; }
      /** Set the run number.
          @memo Set the run number.
          @param num run number
       ******************************************************************/
      void setRunNum (int num) {
         fRunNum = num; }
      /** Get the run number.
          @memo Get run number.
          @return run number
       ******************************************************************/
      int getRunNum () const {
         return fRunNum; }
   
   protected:
      /// Swap byte order?
      bool		fSwap;
      /// Version
      int		fVersion;
      /// Frame position
      framepos		fFramePos;
      /// Start time of frame
      Time		fTime;
      /// Time of next frame
      Time		fNext;
      /// Run number
      int		fRunNum;
      /// Frame number
      int		fFrameNum;
      /// Frame length in seconds
      int		fFrameLen;
      /// Number of frames per file
      int		fNFrame;
      /// compression scheme?
      int		fCompress;
      /// Current frame counter
      int		fN;
      /// Total byte count written into the frame buffers so far
      int		fTotal;
      /// Data buffer
      framebufferlist	fFrameBuffer;
      /// Detector information
      detector_array_t	fDetector;
      /// Table of contents
      toc_t		fTOC;
      /// File name
      std::string	fFilename;
   
      /** Pointers to Frame header, detector info, history info, 
          raw data, ADC data, data vector, End of frame, TOC, 
          End of file */
      ptr_struct 	fPtr[9];
   
      /// get the current buffer 
      framebuffer_t* currentBuffer();
      /// append data to output buffers
      bool appendBuffer (const char* dat, int len);
      /// write frame file header
      bool beginFile();
      /// write frame file trailer (TOC)
      bool endFile();
      /// write frame header
      bool beginFrame();
      /// write frame trailer
      bool endFrame();
   };

}

#endif /* _LIGO_FRAMEFAST_H */
