/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef NDS2SOCKET_HH
#define NDS2SOCKET_HH
//
//    This is a C++ NDS2 client interface.
//
#include "DAQC_api.hh"

namespace sends {

/** @page Secure Network Data Access (NDS2) API (C++)
    This API provides a client interface for the NDS2 server. It allows 
    the user to get channel data over the network. Usage is generally 
    by way of the abstract DAQC_api class. Several server specific methods
    methods are provided.

    The protocol rules are similar to the classic nds server protocol. 
    * Each request (including the authentication message) result in a 4 
      hex digit response code that indicates the status of the request.
      The 

    The syntax implemented by this interface is:
    <table>
    <tr>
    <td>Function</td>
    <td>NDS2Socket method</td>
    <td>NDS2 command</td>
    </tr>
    <tr>
    <td>Get data from specified time</td>
    <td>RequestData</td>
    <td>get-data &lt;start&gt; &lt;duration&gt; [&lt;stride&gt;] {<channel-desc> ...};</td>
    </tr>
    <tr>
    <td>Get available channel list</td>
    <td>Available</td>
    <td>get-channels [&lt;gps&gt;] [&lt;chan-type&gt;] ;</td>
    </tr>
    <tr>
    <td>Get continuous stream of online data.</td>
    <td>RequestOnlineData</td>
    <td>get-online-data [[&lt;duration&gt;] &lt;stride&gt;] {"channel"...};</td>
    </tr>
    <tr>
    <td>Quit session</td>
    <td>StopWriter</td>
    <td>quit;</td>
    </tr>
    </table>

    A typical online NDS2 client would do the following:
    \verbatim
    //---------  Construct a DAQD socket 
    NDS2Socket nds;

    //---------  Open a socket to the specified server port.
    const char* servername = "nds-server";
    int port(31200);
    nds.open(servername, port);
    if (!nds.isOpen()) fail("Open failed!");

    //---------  Specify the channel to be read.
    const char* chan = "channel name";
    if (!nds.AddChannel(chan, 0.0, _undefined)) fail("Add channel failed!");
    if (nds.RequestData(gpsStart, gpsEnd, 2.0)) fail("Data Request failed");

    //---------  Specify the channel to be read.
    float* Samples = new float[data_len];
    while (1) {
    int nData = nds.GetChannelData(chan, Samples, data_len);
    if (nData <= 0) break;
    ...  Process data ...
    }
    \endverbatim
    @memo Access channel data through the network data server
    @author John Zweizig
    @version 0.1; Last modified March 5, 2008
    @ingroup IO_daqs
************************************************************************/

/*@{*/

/**  NDS2Socket provides a client interface to the Network Data Server.
  *  The server provides data in the CDS proprietary format or as standard
  *  frames. The interface may also be used to list channel names, or to
  *  specify the channels to be read in.
  *  @brief The DAQD socket class.
  *  @author John Zweizig and Daniel Sigg
  *  @version 1.2; last modified March 5, 2008
  *  @ingroup IO_daqs
  */
class NDS2Socket :public DAQC_api {
public:
    using DAQC_api::GetData;

    /**  Construct an unopened socket.
      *  \brief Default (idle) constructor.
      */
    NDS2Socket(void);
   
    /**  Create a socket and connect it to the specified NDS2 server.
      *  The network address argument has the same syntax as that passed to
      *  open().
      *  \brief Construct and open an nds2 client socket.
      *  \param ipaddr  Server IP address
      *  \param ipport  Server port ID.
      *  \param RcvBufferLen System receive buffer length.
      */
    explicit NDS2Socket(const std::string& ipaddr, int ipport = NDS2_PORT,
                        long RcvBufferLen = 1048576);
   
    /**  Disconnect and close a socket.
      *  \brief Destructor.
      */
    ~NDS2Socket(void);
    
    /**  Open an existing socket and connect it to a server.
      *  The argument, \a ipaddr, specifies the IP address of the node on which 
      *  the network data server is running. It may be specified either as 
      *  a symbolic name or as four numeric fields separated by dots. open()
      *  returns zero if successful, a positive non-zero error code if one was 
      *  returned by DAQD or -1.
      *  \brief Open connection to the server.
      *  \param ipaddr  Server IP address
      *  \param ipport  Server port ID.
      *  \param bufsize System receive buffer size.
      *  \return Zero on success, Server error response or -1.
      */
    int  open (const std::string& ipaddr, int ipport = NDS2_PORT, 
	       long bufsize = 1048576);

    /**  Disconnect and close a socket.
      *  \brief Close socket
      */
    void close(void);
   
    /**  Flush any pending input data from the socket.
      *  \brief Flush the socket data.
      */
    void flush(void);

    /**  The names, sample rates, etc. of all channels known by the server are 
      *  appended into the channel vector. addAvailable() returns the number of 
      *  entries found or -1 if an error occurred.
      *  @brief List all known channels.
      *  \param typ  %Channel type.
      *  \param gps  Time.
      *  \param list Vector to receive the list of channels.
      *  \param timeout Maximum time to wait for a response.
      *  \return Number of channels added or -1 if an error occurred.
      */
    int  addAvailable (chantype typ, long gps, std::vector<DAQDChannel>& list, 
		       wait_time timeout=-1);
   
    /**  Receive block of data in the CDS proprietary format.
      *  A single block of data (including the header) is received and stored
      *  in an internal buffer. If a reconfigure block precedes the block,
      *  the new configuration information is stored in the nds block.
      *  \brief Get a data block from the server.
      *  \param timeout Number of seconds to time out or -1 to wait forever.
      *  \return data length, or failure.
      */
    int GetData(wait_time timeout = -1);
   
    /**  The network data server is requested to start a net-writer task.
      *  Only channels explicitly specified by AddChannel() will be written.
      *  \brief Start reading online data.
      *  \param stride  time stride.
      *  \param timeout Maximum time to wait for server response.
      *  \return Server response code.
      */
    int  RequestOnlineData (double stride=1.0, wait_time timeout=-1);
   
    /**  The network data server is requested to start a net-writer task
      *  for past data. Start time and duration are given in GPS seconds. 
      *  Only channels explicitly specified by AddChannel() will be written.
      *  \brief Request data.
      *  \param start    Start time of data to be read.
      *  \param duration Duration of data.
      *  \param timeout  Maximum time to wait for server response.
      *  \return Server response code.
      */
    int  RequestData (unsigned long start, unsigned long duration, 
		      wait_time timeout=-1);

    /**  The specified epoch of interest is used to constrain channel 
      *  requests for the remainder of this transaction.
      *  \brief Set an epoch of interest.
      *  \param epochName Epoch name or gps range.
      *  \return Zero or -1 on error.
      */
    int SetEpoch(const std::string& epoch);

    /**  The specified epoch of interest is used to constrain channel 
      *  requests for the remainder of this transaction.
      *  \brief Set an epoch of interest.
      *  \param gps range.
      *  \return Zero or -1 on error.
      */
    int SetEpoch(unsigned long start, unsigned long stop);

    /**  The list of pre-defined epochs is returned by the server
      *  in a space-separated format where each epoch is defined as
      *  <epochName>=<startGPS>-<stopGPS>
      *  \brief Get epoch list
      *  \param epochList string which will contain the list if successful
      *  \return server response code if 0 or negative, otherwise the string length.
      */
    int GetEpochList(std::string &epochList) ;

    /**  This function effectively countermands the RequestXXX() functions.
      *  StopWriter returns either the server response code or -1 if no 
      *  writer has been requested.
      *  \brief Stop a data writer.
      *  \return Response code or -1 on error.
      */
    int  StopWriter(void);
   
    /**  The network data server is requested to return start time and 
      *  duration of the data stored on disk.
      *  \brief Get known time intervals.
      *  \param typ      %Channel type to query.
      *  \param start    Variable to receive start gps time.
      *  \param duration Variable to receive duration.
      *  \param timeout  Maximum wait time.
      *  \return Zero on success or error code.
      */
    int  Times (chantype typ, unsigned long& start, unsigned long& duration, 
		wait_time timeout = -1);
   
    /**  Execution is blocked until data are available to the socket. This
      *  can be used to wait for data after a request has been made. The
      *  calling function can then e.g. allocate a buffer before calling 
      *  GetData(), GetName() or GetFrame(). If poll is true the function
      *  returns immediately with 1 if data is ready, or 0 if not.
      *  \brief Wait for data to arrive.
      *  \param poll Polling flag
      *  \return 1: Data ready, 0: no data or negative: error.
      */
    int WaitforData (bool poll);
      
private:   
    /**  Perform sasl based authorization 
      *  @param server Server ip name
      *  @return 0 on success or sasl error code.
      */
    int authenticate(const char* server);

    /**  Get a character string from the server. The server data are encoded 
      *   in radix 64.
      */
    int gets(char* buf, int buflen);

    /**  Send a character string to the server. The server data are encoded 
      *   in radix 64.
      */
    int puts(const char* buf, int buflen);


    /**  Receive a data header and data block.
      *  A memory buffer of correct length is allocated automatically. The 
      *  header record will be put at the beginning of the buffer. 
      *  @return The number of data bytes, or <0 on error.
      */
    int RecvData(wait_time timeout = -1);

    /**  Receive data from the socket.
     *  Up to 'len' bytes are read from the socket into '*buf'. If readall
     *  is true, RecvRec will perform as many reads as is necessary to 
     *  fill 'buf'.
     */
    int RecvRec(char *buf, long len, bool readall=false, 
		wait_time maxwait=-1);
    
    /**  Receive a reconfigure block.
      *  A reconfigure structure of channel metadata.
      */
    int RecvReconfig(unsigned long len, wait_time maxwait=-1);
   
    /**  Send a record data to the socket.
     *  Up to 'len' bytes are written to the socket from '*buf'.
     */
    int SendRec(const char *buffer, long length, wait_time maxwait=-1);

    /**  Send a request.
     *  SendRequest sends a request specified by 'text' to the server and
     *  reads the reply status and optionally the reply text. 'reply' is 
     *  a preallocated buffer into which the reply is copied and 'length' 
     *  is the size of the reply buffer. If 'reply' is omitted or coded 
     *  as 0, the status will not be read. The reply status is returned by 
     *  SendRequest and the length of the reply message is returned in 
     *  *Size if Size is non-zero. Only the first message is read into 
     *  reply, so in general *Size != length.
     */
    int SendRequest(const std::string& text, char *reply=0, long length=0, 
		    long *Size=0, wait_time maxwait=-1);

private:
    ///  Socket number.
    int   mSocket;
   
    ///  Offline flag sent when the writer is started.
    int   mOffline;

    ///  Authentication context
    void* mAuthContxt;

private:
    static bool sasl_init;
};

/*@}*/

} // namespace sends

#endif  //  NDS2SOCKET_HH
