
#ifndef TRENDCHAN_HH
#define TRENDCHAN_HH

#include "TSeries.hh"
#include "TrendAcc.hh"

class FrWriter;
class Dacc;

/**  Accumulate and write trend data. The trend is given  minor and a 
  *  major interval. The trend averages over all samples within the minor
  *  interval and produces a frame over the major interval.
  *  @memo Make trend frames.
  *  @author John Zweizig.
  *  @version 1.0; Modified February 28, 2000.
  */
class TrendChan {
public:
  /**  Data type for data counts.
    */
  typedef TrendAcc::count_t count_t;

  /**  Arithmetic data type.
    */
  typedef TrendAcc::math_t math_t;

  /**  TSeries data type.
    */
  typedef float trend_t;

public:
  /**  Check that the specified string is a valid DMT trend channel name.
    *  \brief Check for valid name.
    *  \param chan %Channel name string to be tested
    *  \param monid Monitor ID string.
    *  \return true if name is valid.
    */
  static bool validName(const std::string& chan, std::string& monid);

public:
  /**  Default Constructor
   */
  TrendChan(void);

  /**  Construct a %TrendChan identical to the argument instance.
    *  \brief Copy constructor
    *  \param x %TrendChan instance to be copied.
    */
  TrendChan(const TrendChan& x);

  /**  Construct a new channel with the given channel base name and 
    *  accumulation interval.
    *  \brief Named constructor.
    *  \param Name Base name string for the trend channel.
    *  \param nAvg Accumulation interval.
    */
  explicit TrendChan(const char* Name, Interval nAvg);

  /**  Destructor
   */
  ~TrendChan();

  /**  Assign the contents of the argument %TrendChan to this instance.
    *  \brief Assignment operator
    *  \param x %TrendChan to be copied to this instance.
    *  \return Reference to this instance.
    */
  TrendChan& operator=(const TrendChan& x);

  /**  Add contents of the argument channel to the current instance.
    *  \brief Addition operator
    *  \param x %TrendChan to be added to this instance.
    *  \return Reference to this instance.
    */
  TrendChan& operator+=(const TrendChan& x);

  /**  Dump the channel status to an STL output stream.
    *  \brief Print trend contents.
    *  \param os Reference to the output stream.
    *  \return Reference to output stream.
    */
  std::ostream& dump(std::ostream& os) const;

  /**  Return the base name of the trend channel time series to be generated.
    *  \brief Get channel base name.
    *  \return Base name string for this channel.
    */
  std::string getName(void) const;

  /**  Return the end of the trend time series.
    *  \brief Current trend end time.
    *  \return End time.
    */
  Time getEndTime(void) const;

  /**  Get the number of samples in the specified time interval.
    *  \brief Get the number of samples
    *  \param tStart Start of time interval
    *  \param tStop  End of time interval
    *  \return Number of samples
    */
  long getNSample(const Time& tStart, const Time& tStop) const;

  /**  Test whether any data has been trended.
    *  @memo Test for trended data.
    *  @return True if trend data exist.
    */
  bool isEmpty(void) const;

  /**  Add each time-series element as a data point to the trend running 
    *  average.
    *  @memo Accumulate trend points from a time series.
    *  @param ts The value of the next element of a series.
    */
  void addData(const TSeries& ts);

  /**  Add a data point to the trend running average. If the time-stamp is 
    *  outside the currently accumulated point, the previous point is saved to
    *  the trend series and the accumulator is initialized to the new point.
    *  \brief Accumulate a single point.
    *  \param t     The time of the next element to be added.
    *  \param point The value of the next element of a series.
    */
  void addData(const Time& t, math_t point);

  /**  Clear all data.
    */
  void clear(void);

  /**  Delete data up to (but not including) the specified time.
    *  Data are discarded from all time series in the trend channel.
    *  @memo remove data from front of trend.
    *  @param t new trend start-time.
    */
  void discard(const Time& t);

  /**  Extend all the trend series to the specified time.
    *  @memo Extend time series.
    *  @param t new trend end-time.
    */
  void extend(const Time& t);

  /**  Get a reference to the current average value series
    *  \brief Average series reference.
    *  \return Reference to the average time series.
    */
  const TSeries& refAvgSeries(void) const;

  /**  Get a reference to the current maximum entry series
    *  \brief Maximum series reference.
    *  \return Reference to the maximum time series.
    */
  const TSeries& refMaxSeries(void) const;

  /**  Get a reference to the current minimum entry series
    *  \brief Minimum series reference.
    *  \return Reference to the minimum time series.
    */
  const TSeries& refMinSeries(void) const;

  /**  Get a reference to the current entry count series
    *  \brief Count series reference.
    *  \return Reference to the count time series.
    */
  const TSeries& refNSeries(void) const;

  /**  Get a constant reference to the current RMS series.
    *  \brief Rms series reference.
    *  \return Reference to the RMS time series.
    */
  const TSeries& refRmsSeries(void) const;

  /**  Set the portion of the current trend covered by the argument series
    *  to the argument data. All data (mean, rms, min, max) are set to the
    *  same values and the number of averages is set to 1 over the same range.
    *  \brief Set the trend data.
    *  \param Avg Average time series to be used as a trend.
    */
  void setData(const TSeries& Avg);

  /**  Set the portion of the current trend covered by the argument series
    *  to the argument data.
    *  \brief Set the trend data.
    *  \param Avg Average time series.
    *  \param Rms RMS time series.
    *  \param Min Minimum time series.
    *  \param Max Maximum time series.
    *  \param Num Number time series.
    */
  void setData(const TSeries& Avg, const TSeries& Rms, const TSeries& Min, 
	       const TSeries& Max, const TSeries& Num);

  /**  Set the trend name and assign names to the constituent time series.
    *  When the trend is written these derived names will be used as the
    *  "ADC" names in the frame file.
    *  @memo Set the trend name.
    *  @param Name Name for the trend data.
    */
  void setName(const char* Name);

  /**  Request the data sub-channels for this trend channel from the specified 
    *  Dacc instance. Note that no data are transferred by this method.
    *  \brief Request trend channels.
    *  \param In Dacc instance to be used to read in trend data.
    */
  void setReadout(Dacc& In);

  /**  Copy the trend data for this channel from the specified data accessor
    *  instance. The requested data are assumed to have been already read
    *  by the calling function.
    *  \brief Copy trend data data.
    *  \param In Data accessor from which the trend data are read.
    */
  void setReadData(Dacc& In);

  /**  Set the units for the mean, rms, min and max channels.
    *  \brief Set trend value units.
    *  \param units Unit name.
    */
  void setUnits(const std::string& units);

  /**  Start accumulating a trend series. Any data already accumulated before 
    *  the frame start time is discarded. The frame start time is saved.
    *  @memo Start a new trend frame.
    *  \param start Frame start time.
    */
  void startFrame(const Time& start);

  /**  Force current point into trend series, and pad the time series to 
    *  the specified time.
    *  @memo Synchronize trend series to current data.
    *  @param t Time to which the data will be padded.
    */
  void synch(const Time& t=Time(0));

  /**  Write channel data to a trend frame.
    *  \brief Write the data channels.
    *  \param writer Frame writer to receive channel data.
    */
  void write(FrWriter& writer);

private:
  /**  Reset the running average.
    *  @memo add a single point.
    *  @param point The value of the next element of a series.
    */
  void reset(void);

  /**  Start accumulating a new trend point at the specified time. The 
    *  accumulation start time is moved forward to the nearest point time
    *  boundary.
    *  @memo Start to accumulate a new trend point.
    *  @param start a new trend point.
    */
  void startAcc(const Time& start);

  /**  Add the current running average to the time series.
    *  @memo add a single point.
    *  @param point The value of the next element of a series.
    */
  void trendPoint(void);

  /**  Append the current running average to the end of the time series.
    *  @memo add a single point.
    *  @param point The value of the next element of a series.
    */
  void appendPoint(void);

  /**  Add the current running average to an appropriate point in the
    *  time series.
    *  @memo add a single point.
    *  @param point The value of the next element of a series.
    */
  void incrementPoint(void);

private:
  /**  Trend channel name.
    */
  std::string mName;

  /**  Running total of accumulated points.
    */
  TrendAcc mAccum;

  /**  Start time for the current frame.
    */
  Time mStartFrame;

  /**  Start time for current accumulation period.
    */
  Time mStartAcc;

  /**  Accumulation interval.
    */
  Interval mAccIntvl;

  /**  Trend data - Number of samples.
    */
  TSeries mNSampTS;

  /**  Trend data - Average value.
    */
  TSeries  mAvgTS;

  /**  Trend data - RMS value.
    */
  TSeries  mRmsTS;

  /**  Trend data - Minimum Value.
    */
  TSeries  mMinTS;

  /**  Trend data - Maximum value.
    */
  TSeries  mMaxTS;

  /**  Units for the mean, rms, min and max trends. The n trend is unitless
    *  and is left blank.
    */
  std::string mUnits;
};

//======================================  Inline functions
#ifndef __CINT__

inline bool
TrendChan::isEmpty(void) const {
    return !mStartAcc && mNSampTS.isEmpty();
}

inline std::string 
TrendChan::getName(void) const {
    return mName;
}

inline const TSeries& 
TrendChan::refAvgSeries(void) const {
    return mAvgTS;
}

inline const TSeries& 
TrendChan::refMaxSeries(void) const {
    return mMaxTS;
}

inline const TSeries& 
TrendChan::refMinSeries(void) const {
    return mMinTS;
}

inline const TSeries& 
TrendChan::refNSeries(void) const {
    return mNSampTS;
}

inline const TSeries& 
TrendChan::refRmsSeries(void) const {
    return mRmsTS;
}
#endif  // def(__CINT__)

#endif  // TRENDCHAN_HH
