/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef FRSTATDATAREF_HH
#define FRSTATDATAREF_HH

#include <string>
#include "FramePointerTypes.hh"

class FrVectRef;
class Time;

/**  Class FrStatDataRef contains a wrapped reference to an FrStatData and 
  *  supplies various data accessor methods.
  *  @memo Lib independent FrStatData reference
  *  @author J. Zweizig
  *  @version 1.0; last modified Jul 30, 2006
  */
class FrStatDataRef {

//-------------------------------------  Lib-dependent types
public:
    /// Size data type definition.
    typedef unsigned long    size_type;

    /// FrStatData structure data typ definition.
    typedef FrameCPP::FrStatData stat_type;

    /// FrStatData structure data typ definition.
    typedef frstatdata_pointer stat_ptr_type;

//-------------------------------------  Class methods
public:
    /**  Null static data reference constructor.
      *  \brief Default (null) FrStat reference constructor.
      */
    FrStatDataRef(void);

    /**  Construct a FrStatRef from a pointer to an existing structure.
      *  \brief Adoption constructor.
      *  \param x Pointer to an existing FrStat structure.
      */
    FrStatDataRef(const stat_ptr_type x);

    /**  Construct a new FrStatData structure.
      *  \brief FrStatData constructor.
      *  \param name    Static data name.
      *  \param comment Static data comment string.
      *  \param repr    Data representation string.
      *  \param vsn     version number.
      *  \param tStart  Data validity start time.
      *  \param tEnd    Data validity end time.
      *  \param det     Detector data structure pointer.
      *  \param v       Static data vector.
      */
    FrStatDataRef(const std::string& name, const std::string& comment, 
		  const std::string& repr, long vsn, const Time& tStart, 
		  const Time& tEnd, void* det, FrVectRef& v);

    /**  Destroy the reference. Delete the references structure if it is
      *  owned exclusively by this reference.
      *  \brief Destructor.
      */
    ~FrStatDataRef(void);

    /**  Reset the pointer and release the referenced instance if owned.
      *  \brief Clear pointer.
      */
    void clear(void);

    /**  Get comment string.
      *  \brief Comment string.
      *  \return Comment string pointer.
      */
    const char* getComment(void) const;

    /**  Get end time.
      *  \brief End time.
      *  \return End time.
      */
    Time        getEndTime(void) const;

    /**  Get data vector reference.
      *  \brief Data vector reference
      *  \return Data vector reference
      */
    FrVectRef   getFrVect(void) const;

    /**  Get static data name string.
      *  \brief Name string
      *  \return Pointer to name string.
      */
    const char* getName(void) const;

    /**  Get static data representation string.
      *  \brief Representation string
      *  \return Pointer to representation string.
      */
    const char* getRepresentation(void) const;

    /**  Get start time.
      *  \brief Start time.
      *  \return Start time.
      */
    Time        getStartTime(void) const;

    /**  Static data version number.
      *  \brief Version number.
      *  \return Version number.
      */
    long        getVersion(void) const;

    /**  Test whether static reference is null.
      *  \brief test for null reference.
      *  \return True if reference is null.
      */
    bool        null(void) const;

    /**  Release oewnership of a static reference. The pointer is released 
      *  and the reference is set to null. The references data structure is 
      *  not deleted.
      *  \brief release FrStatData reference.
      *  \return pointerr to the referenced FrStatData structure.
      */
    stat_ptr_type release(void);

    /**  Create a copy of the referecned FrStatData structure.
      *  \brief Clone the references FrStatData structure.
      *  \return Pointer to the referenced structure.
      */
    stat_ptr_type replicate(void) const;

    /**  Construct an FrStatData instance with the specified data fields
      *  and save the reference.
      *  \brief Construct a FrStatData instance.
      *  \param name    Static data name.
      *  \param comment Static data comment string.
      *  \param repr    Data representation string.
      *  \param vsn     version number.
      *  \param tStart  Data validity start time.
      *  \param tEnd    Data validity end time.
      *  \param det     Detector data structure pointer.
      *  \param v       Static data vector.
      */
    void setStat(const std::string& name, const std::string& comment, 
		 const std::string& repr, long vsn, const Time& tStart,
		 const Time& tEnd, void* det, FrVectRef& v);

    /**  Set the reference to the specified FrStatData pointer. Set the
      *  ownership flag as indicated.
      *  \brief Adopt an FrStatData anstance.
      *  \param v     Pointer to FrStatData structure.
      *  \param owner Mark reference as the owner if true.
      */
    void setStat(stat_ptr_type v, bool owner=false);

private:
  bool       mOwner;
  stat_ptr_type mData;
};

//======================================  Inline methods.
inline void
FrStatDataRef::clear(void) {
#ifdef FCPP_SHARED_PTRS
    mData.reset();
#else
    if (mOwner) delete mData;
    mOwner = false;
    mData = 0;
#endif
}

inline FrStatDataRef::stat_ptr_type
FrStatDataRef::release(void) {
#ifdef FCPP_SHARED_PTRS
    return mData;
#else
    if (!mOwner) return replicate();
    mOwner = false;
    return mData;
#endif
}

inline bool
FrStatDataRef::null(void) const {
    return !mData;
}

#endif // FRSTATDATAREF_HH
