/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef CHANNELINDEX_HH
#define CHANNELINDEX_HH

#include <string>

/**  The %ChannelIndex class implements a cross-reference between the 
  *  channel name and the index of data accessor from which it is available.
  *  \brief Channel / Stream cross reference.
  *  \author John Zweizig (john.zweizig@ligo.org)
  *  \version $Id: ChannelIndex.hh 7688 2016-08-15 17:18:20Z ed.maros@LIGO.ORG $
  */
class ChannelIndex
{
public:
    ///  Index type
    typedef unsigned int index_type;

    //------------------------------------  Constructor/Destructor.
public:
    /**  Construct an empty index.
     *  \brief  Default Constructor.
     */
    ChannelIndex(void);

    /**  Construct an object given a channel name and accessor index.
     *  \brief Construct a channel index entry for the specified channel name.
     *  \param name Channel name.
     *  \param index Accessor index.
     */
    ChannelIndex(const std::string& name, int index=-1);

    /**  Destroy a channel entry.
     *  \brief Destructor.
     */
    ~ChannelIndex();

    //------------------------------------  Accessors
    /**  Compare the channel name to an argument string.
      *  \brief Compare the channel name to an argument string.
      *  \param name String to which the channel name is compared.
      *  \return true if the channel name is the same as the argument.
      */
    bool EqName(const std::string& name) const;

    /** Get the debug flag for this channel.
     *  \brief Debug flag.
     *  \return Debug flag value.
     */
    bool getDebug(void) const;

    /**  Return the accessor index for the  specified channel.
      *  \brief Get the Accessor index.
      *  \return Accessor list index.
      */
    index_type getIndex(void) const;

    /**  Returns a pointer to the channel name string.
     *  \brief Get the channel name.
     *  \return Pointer to the name string for this channel.
     */
    const std::string& getName(void) const;

    /**  The debug flag is set as specified.
     *  \brief Set or clear the debug flag.
     *  \param yorn New debug flag setting.
     */
    void setDebug(bool yorn) {mDebug = yorn;}

    /**  Set the accessor index for this channel.
      *  \brief Set index value.
      *  \param index New index value.
      */
    void setIndex(index_type index) {mIndex = index;}

private:                                // Private data members.
    /**  The name of the channel to be processed.
      *  \brief Channel name.
      */
    std::string mName;

    /**  Index of associated accessor in the accessor list.
      *  \brief Accessor list index.
      */
    index_type mIndex;

    /**  If set, copious error and status messages are printed to cerr.
      *  \brief Debug flag
      */
    bool mDebug;
};


//======================================  Inline methods.
inline bool 
ChannelIndex::EqName(const std::string& name) const {
    return (mName == name);
}

inline bool 
ChannelIndex::getDebug(void) const {
    return mDebug;
}

inline ChannelIndex::index_type 
ChannelIndex::getIndex(void) const {
    return mIndex;
}

inline const std::string& 
ChannelIndex::getName(void) const {
    return mName;
}

#endif // CHANNELINDEX_HH
