//-----------------
// class psdCompare
//-----------------

//Author: Sharif Siddiqui
//        Graduate Student, Penn State University
//Date  : 06/06/00
//Version: 1.01
//
//Last Modified: 06/09/00
//
//Original Code written in Matlab by: Somya Mohanty      
//
//
//NOTE: This is the implementation of the first part of the algortihm,
//      which only detects points at which Psd change crosses threshold.
//      Second part will be available soon........ 

#ifndef __CINT__
#include "FSpectrum.hh"
#include "FSeries.hh"
#include "TSeries.hh"
#include "Interval.hh"
#include "Dacc.hh"
#include "DVector.hh"
#include <iostream>

#include "TNtuple.h"
#include "TCanvas.h"
#include "TROOT.h"
#include "TFrame.h"
Dacc In;
#endif

/**  This is the implementation of the first part of the algortihm,
  *  which only detects points at which Psd change crosses threshold.
  *  Second part will be available soon.
  *
  *  Further documentation is available on-line from 
  *  \URL{http://www.personal.psu.edu/~sms35/ligo/demo.html#psd_change}.
  *
  *  @author Sharif Siddiqui &  Somya Mohanty.
  *  @version 1.01; Last Modified 6/9/00
  */
class PsdCompare{

private:
  //Variables
  size_t DataLength;   //Length of Input data in number of samples
  size_t PsdLength;    //Length of data segment in number of samples 
  size_t Nfft;         //Number of (FFT) points in smaller subsegments 
  size_t FreqLength;   //Nfft/2 = number of frequency steps 
 
  double TimePeriod;  //Sampling Interval 
  double Threshold;   //Threshold
  double TimeSegment; //Duration of data segment (of PsdLength) in seconds
  double FreqStep;    //Frequency step between PSD data. 
   
  unsigned int Lag;   //Gap between reference and compared data segments   
  unsigned int N;     //Number of smaller subsegments in one data segments
  unsigned int Total_Iteration;
  
  float SampSecs;  //Sampling time while filling Data in channel.  
  float *mean_ref; //Array of Means of the Reference data seg. for diff. freqs 
  float *var_ref; //Array of variance of the Ref. data seg for diff. freqs
  float *mean_comp; //Array of Mean of the data block to compare
  float *var_comp; //Array of Variances of the data block  to compare.
     
  unsigned char *OutputData; //Output binary image data. (255 = 1, 0 = 0)
  
  const TSeries* X;       //Pointer to Input Time Series data to be analysed 
  Interval IntrVal; //Interval Object containing sampling interval of inp data
  
  //Member Functions
  void GetMeanVar(DVector *curDVect, float *mean, float *var);//computes mean 
                                                             //and variance 
  void SaveImage(unsigned char *Data, char *filename); //Save the output data
                                                        //to a raw binary image
 
public:  
  ///Set of contructors...
  PsdCompare();  //Default Constructor         
  PsdCompare(TSeries* ts, unsigned int nfft=64, unsigned int lag=3, 
	     double timeSegment=0.5, float threshold=0.5);
  ///Destructor
  ~PsdCompare();

  //Other Methods..
  /// Sets the input TSeries object.
  void SetTSeries(TSeries* ts);
  
  ///Changes/Sets Parameters
  int  SetParams(unsigned int nfft, unsigned int lag, double timeSegment,
		 float threshold);

  ///Calculates Mean/Variance and constructs OutPut.
  int  GetImage(); 
  ///Constructs a binary data file (overloaded)
  int  GetImage(char *OutputFile); 
  ///Gets the OutputData in 2D 'Data' array and returns its row/column length.
  int  GetData(size_t &freqLength, size_t &iteration, int Data[]); 
  ///Plots the OutputData vector
  void GetPlot(); 
  ///Plot according to coordinate limited by user
  void GetPlot(int x, int y); 
  ///Reports different parameters
  void GetInfo(); 
  ///Dumps the content of output on stream
  ostream& Dump(ostream& out) const; 
};




PsdCompare::PsdCompare()
{
  X = NULL;            //Initializing the pointers as NULL
  OutputData = NULL;
  //So far nothing is planned to initialize here...
  //Have to use SetTSeries and SetParams later to initialize the object.
}



PsdCompare::PsdCompare(TSeries* ts, unsigned int nfft, unsigned int lag, 
		       double timeSegment, float threshold)
{
  X = new TSeries(*ts);  //Using copy constructor to Initialize X. 
  SetParams(nfft, lag, timeSegment, threshold);
}   





PsdCompare::~PsdCompare()
{
  free(OutputData);  //Free allocated memory
  delete X;
}




void PsdCompare::SetTSeries(TSeries* ts)
{
  if(X){  
    delete X;
    cout<<"Previous Time Series Deleted.."<<endl;
  }
  X = new TSeries(*ts);
  SetParams(Nfft, Lag, TimeSegment, Threshold);
}



int PsdCompare::SetParams(unsigned int nfft, unsigned int lag, 
			   double timeSegment, float threshold)
{
  //Checking if this method is called before setting a TSeries input
  if(!X){
    cout<<"Error: Possibly no input time series data\n";
    cout<<"       Please use SetTSeries(..) to load the input data first\n\n";
    cout<<"Sorry... object not initialized.\n";
    return 0;
  }

  //Initializing all private variables...
  Lag = lag;
  Threshold = threshold;
  Nfft = (unsigned int)pow(2, (unsigned int)(log(double(nfft))/log(2.0))); 
  FreqLength = Nfft/2;
  TimeSegment = timeSegment;
  DataLength = X->getNSample();  
  IntrVal = X->getTStep();
  TimePeriod = IntrVal.GetSecs();
  SampSecs = (float)(DataLength*TimePeriod);
  PsdLength = (unsigned long)(TimeSegment/TimePeriod);
  
  if(floor(DataLength/PsdLength) < Lag){
    cout<<"Error: Too few Data Samples... adjust Lag or increase data.."<<endl;
    cout<<"       Sometimes one possible solution is restarting ROOT"<<endl;
    return 0;
  }

  Total_Iteration = (unsigned int)floor(DataLength/PsdLength) - Lag;
  // N = floor(PsdLength/Nfft);
  N = PsdLength/Nfft;

  if(OutputData)free(OutputData);

  if((OutputData = (unsigned char *)calloc(FreqLength, Total_Iteration))==NULL)
  {
    cout<<"Error: Couldnot allocate memrory"<<endl;
    return 0;
  }
  cout<<"Object Initialized and meomory allocated.."<<endl;
  return 1;
}
 


int PsdCompare::GetImage(char *OutputFile)
{
  if(!GetImage())return 0;
  
  cout<<"\n\nSaving binary raw image to file "<<OutputFile<<"..."<<endl;  
  SaveImage(OutputData, OutputFile);
  return 1;
}



int PsdCompare:: GetImage()
{
  unsigned int i,j;
  unsigned int start_point = 0;
  double comp_data; 
    
  for(i=0;i<Total_Iteration;i++)for(j=0;j<FreqLength;j++)
          OutputData[i*FreqLength + j] = 0;
 
  for(i = 0; i < Total_Iteration; i++){ // i = Iteration count
    /* Allocating memory for variables */  
   if(   ((mean_ref = new float[FreqLength]) == NULL) 
      || ((var_ref =  new float[FreqLength]) == NULL)
      || ((mean_comp = new float[FreqLength])== NULL)
      || ((var_comp = new float[FreqLength]) == NULL)){
        cout<<"Error: Couldnot Allocate memory...."<<endl;
        return 0;
     }
   unsigned int cur_start;    

   cur_start = start_point;
   // Extract Data Vector of reference  data segment 
   DVector* dv_ref = X->refDVect()->Extract(cur_start, PsdLength); 
   // Calculate mean, var etc for current data chunk  
   GetMeanVar(dv_ref, mean_ref, var_ref); 
   
   cur_start = start_point+(PsdLength*Lag);
   // Extract Data Vector of compared data segment.
   DVector* dv_comp = X->refDVect()->Extract(cur_start, PsdLength);
   GetMeanVar(dv_comp, mean_comp, var_comp);

   //cout<<"Completed iteration #"<<i<<endl;   
 
   // Comparing and loading output data in OutputData[][].. 
   for(j = 0; j < FreqLength; j++){ // j = frequency step number 
     comp_data = sqrt(var_ref[j] + var_comp[j]);
     if(comp_data!=0)    
       comp_data = sqrt(N)*(mean_ref[j] - mean_comp[j])/comp_data; 

     if(abs(comp_data) > Threshold) OutputData[i*FreqLength + j] = 255;
     else OutputData[i*FreqLength + j] = 0;      
   } 
   start_point+= PsdLength; //Increment the start point.
   
   free(mean_ref);
   free(var_ref);
   free(mean_comp);
   free(var_comp); 
  }// i loop ends  
  return 1;  
}



void PsdCompare::GetMeanVar(DVector *curDVect, float *mean, float *var)
{
  unsigned int i,j; //Loop Counters.
  
  float *tData; //Extracted data array of length - Nfft (smaller subsegment)
  float *fData; //Frequency Spectrum data array. 
  double *sumdata; //Sum of psd for different frequencies.
  double *sum_sqrdata;//Sum of square of psd for different frequencies.
 
  TSeries *ts_nfft;  
  FSeries *fs;
  FSpectrum *fSpect;
  
  // Allocating Memory
  if( ((tData = new float[Nfft]) == NULL) ||
      ((fData = new float[FreqLength]) == NULL) ||
      ((sumdata = new double[FreqLength]) == NULL)||
      ((sum_sqrdata= new double[FreqLength]) == NULL))
  {
    cout<<"Error: Couldnot Allocate Memory..."<<endl;
    exit(1);
  }  
   
  for(i = 0; i < FreqLength; i++){
    sumdata[i] = 0;
    sum_sqrdata[i] = 0;  
  }
  
  for(i = 0; i < PsdLength; i+= Nfft){ 
     //Data of length nfft is extracted for smaller subsegment.   
     curDVect->getData(i, Nfft, tData); 
     ts_nfft = new TSeries(Time(0), IntrVal, Nfft, tData);
     fs = new FSeries(*ts_nfft); 
     //Obtain the FSeries object from TSeries ts_nfft
     fSpect = new FSpectrum(*fs);
     //Obtain Freq. Spectrum for the subsegment 
     fSpect->getData(FreqLength, fData); //Get frequency data pointer 
     //eff_lenf = fSpect.getNStep();
     
    for(j = 0; j < FreqLength; j++){ //j = frequency
       sumdata[j]+= fData[j];
       sum_sqrdata[j]+= fData[j]*fData[j];
    } 
     delete ts_nfft;
     delete fs;
     delete fSpect;
   } //i loop ends

  //Calculate Mean and Variance
    for(i = 0; i < FreqLength; i++){ // i = Frequency
      mean[i] = (float)(sumdata[i]/N);
      var[i]  = (float)((sum_sqrdata[i]/N) - mean[i]*mean[i]);
    }

  delete[] tData;
  delete[] fData;
  delete[] sumdata;
  delete[] sum_sqrdata;
}



int PsdCompare::GetData(size_t &freqLength, size_t &iteration, int Data[])
{
  unsigned int i,j;

  if(!OutputData)return 0;
  freqLength = FreqLength;
  iteration  = Total_Iteration;
  
  for(i = 0; i < Total_Iteration; i++)
    for(j = 0; j < FreqLength; j++)
      Data[i*FreqLength + j] = (int)(OutputData[i*FreqLength + j])/255;
  //Converting character to integer 1 or 0.     
  return 1;
} 




void PsdCompare::SaveImage(unsigned char *Data, char *filename)
{
  FILE *output;
  unsigned int size;
  
  if((output = fopen(filename, "w")) == NULL){
    cout<<"Error: Could not open output file..";
    exit(1);
  }
  cout<<"Generating image of size "<<FreqLength<<"x"<<Total_Iteration<<endl;  
  size = Total_Iteration * FreqLength;
  //Writing data to file
  fwrite(Data, size, 1, output);
  fclose(output);
}



void PsdCompare::GetInfo()
{
  cout<<"--------Information of current PsdCompare Class----------\n\n";
  cout<<"Sample Time    = "<<SampSecs<<" Secs"<<endl;
  cout<<"Data Length    = "<<DataLength<<endl;
  cout<<"Time Segment   = "<<TimeSegment<<endl;
  if(TimePeriod)
  cout<<"Sampling Freq  = "<<1/TimePeriod<<endl;
  else cout<<"Sampling Freq  = Not Set"<<endl;
  cout<<"PSD length     = "<<PsdLength<<" (TimeSegment x Sampling Freq)"<<endl;
  cout<<"Lag            = "<<Lag<<endl;
  cout<<"Nfft           = "<<Nfft<<endl;
  cout<<"No. of Freq    = "<<FreqLength<<endl;
  cout<<"No. of nfft seg= "<<N<<endl;
  cout<<"Total Iteration= "<<Total_Iteration<<endl;
  cout<<"Threshold      = "<<Threshold<<endl<<endl;
}      


ostream& PsdCompare::Dump(ostream& out) const 
{
  unsigned int i,j;
  int data;
   
  for(i = 0; i < Total_Iteration; i++){
    for(j = 0; j < FreqLength; j++){
      data = (int)(OutputData[i*FreqLength + j])/255;
      out << "Output[" << j << "][" << i << "] = " << data << endl;
    }
  }
  return out;
}
  


void PsdCompare::GetPlot()
{
  GetPlot(FreqLength, Total_Iteration);
}


void PsdCompare::GetPlot(int x, int y)
{
  int i,j;
  
  float Xax, Yax, Yax0;
  float *Data;

  double fStep;

  const DVector* curDVect;
  
  if( (Data = (float *)malloc(Nfft*sizeof(float))) == NULL ){
    cout<<"Error: Couldnot allocate memory.."<<endl;
    exit(1);
  }
  
  curDVect = X->refDVect(); 
  curDVect->getData(0, Nfft, Data); 
  TSeries* Ts = new TSeries(Time(0), Interval(TimePeriod), Nfft, Data); 
 
  FSeries* Fs = new FSeries(*Ts); 
  FSpectrum* fSpect = new FSpectrum(*Fs);    
  fStep = fSpect->getFStep();  
  
  //gROOT->Reset();
   TCanvas* canv1 = new TCanvas("canv1","PSD Change Detection Plot", 200, 10, 840, 680);
   canv1->SetFillColor(42);
   canv1->GetFrame()->SetFillColor(21);
   canv1->GetFrame()->SetBorderSize(6);
   canv1->GetFrame()->SetBorderMode(-1); 
   
   TNtuple* ntuple = new TNtuple("ntuple", "Psd Change Detection", "x:y");

   Yax0 = Lag*TimeSegment;    
   for(i = 0; i < y; i++){
     Yax = Yax0 + i*TimeSegment;
     for(j = 0; j < x; j++){ 
       Xax = j*fStep;
       if(OutputData[i*FreqLength + j])
              ntuple->Fill(Yax, Xax); 
     }
   }

   ntuple->SetMarkerColor(4);
   ntuple->SetMarkerStyle(21);
   ntuple->SetFillColor(45);
   ntuple->Draw("y:x");

   delete Ts;
   delete Fs;
   delete ntuple;
}










