//-------------------------------------  Make a Bode blot from a specified 
//                                       frequency series.
void Bode(const FSeries* fs, const char *opts=0) {
    const double r2d = 180.0/3.1415926535;

    //---------------------------------  Check the argument
    if (!fs || !fs->getNStep()) {
        printf("Bode: Empty series specified.");
	return;
    }

    //---------------------------------  Set up the canvas
    TCanvas* c;
    if (gPad) {
        //  cout << "Reset canvas" << endl;
        c = gPad->GetCanvas();
	c->cd();
	c->Clear();
    }
    gStyle->SetOptStat(0);
    gStyle->SetOptTitle(0);

    int nbin     = fs->getNStep();
    double step  = fs->getFStep();
    double fLow  = fs->getLowFreq();
    bool  logP   = (fLow == 0.0);
    double sfreq = fLow - 0.5*step;
    if (logP) {
        sfreq += step;
	nbin--;
    }
    double efreq = fs->getHighFreq()-0.5*step;

    TString Name=uniqueName("Magn");
    TH1F* histM= new TH1F(Name.Data(), "Magnitude", nbin, sfreq, efreq);
    Name=uniqueName("Arg");
    TH1F* histA = new TH1F(Name.Data(), "Argument", nbin, sfreq, efreq);

    //----------------------------------  Calculate the magnitude and phase
    //                                    Note that in the Sun, a Log10(0)
    //                                    will break the function. I avoid 
    //                                    this by adding 10^(-30), but this
    //                                    will distort the magnitude around
    //                                    -300 dB (hopefully not physical).
    fComplex Cf;
    double fArg = fLow;
    FSeries fRef(*fs);
    for (int i=0 ; i<nbin ; i++) {
        Cf = fRef(fArg);
        histM->Fill(fArg, 10.0*TMath::Log10(Cf.MagSq()+1e-30));
        histA->Fill(fArg, Cf.Arg()*r2d);
	fArg += step;
    }

    //----------------------------------  Display the title.
    TPaveLabel* title = new TPaveLabel(0.30, 0.92, 0.70, 0.98, fs->getName());
    title->Draw();

    //----------------------------------  Draw the magnitude (histM)
    Name = uniqueName("padM");
    TPad* padM = new TPad(Name.Data(), "Amplitude pad", 0.0, 0.45, 1.0, 0.90);
    padM->Draw();
    padM->cd();
    padM->SetTopMargin(0.025);
    padM->SetBottomMargin(0.02);
    padM->SetTicks(1,1);
    if (logP) padM->SetLogx(1);
    TAxis* yax = histM->GetYaxis();
    yax->SetLabelSize(0.08);
    histM->Draw(opts);

    TText* l1 = new TText(0.04, 0.45, "Amplitude (dB)");
    l1->SetNDC(true);
    l1->SetTextAlign(22);
    l1->SetTextSize(0.08);
    l1->SetTextAngle(90.);
    l1->Draw("");

    //------------------------------  Draw the Phase histogram
    c = padM->GetCanvas();
    c->cd();
    Name = uniqueName("padA");
    TPad* padA = new TPad(Name.Data(), "Phase pad", 0.0, 0.00, 1.0, 0.45);
    padA->Draw();

    padA->cd();
    padA->SetTopMargin(0.01);
    padA->SetBottomMargin(0.25);
    padA->SetTicks(1,1);
    if (logP) padA->SetLogx(1);
    yax = histA->GetYaxis();
    yax->SetLabelSize(0.08);
    TAxis* ax = histA->GetXaxis();
    ax->SetLabelSize(0.08);
    ax->SetLabelOffset(-.03);
    histA->Draw(opts);

    padA->cd();
    TText* l2 = new TText(0.5, 0.06, "Frequency (Hz)");
    l2->SetNDC(true);
    l2->SetTextAlign(22);
    l2->SetTextSize(0.08);
    l2->Draw("");

    TText* l3 = new TText(0.04, 0.625, "Phase");
    l3->SetNDC(true);
    l3->SetTextAlign(22);
    l3->SetTextSize(0.08);
    l3->SetTextAngle(90.);
    l3->Draw("");

    //-------------------------> Update the canvas.
    c->cd();
    c->Modified();
    c->Update();
}

//-------------------------------------  Make a bode plot from a plot reference.
void Bode(const FSeries& fs, const char *opts=0) {
    Bode(&fs, opts);
}
