/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "DaqErrFilter.hh"
#include "DVector.hh"
#include "rndm.hh"
#include <iostream>

using namespace std;

//======================================  Daq error constructor
DaqErrFilter::DaqErrFilter(const char* subtype) 
  : mSubTypes(0), mClustWidth(0), mAvgWidth(1.0), mErrRate(1.0), 
    mLsbErrTime(0), mStartTime(0), mCurrentTime(0)
{
    setSubTypes(subtype);
}

//======================================  Daq error destructor
DaqErrFilter::~DaqErrFilter(void) {
}

//======================================  Apply errors to data
TSeries 
DaqErrFilter::apply(const TSeries& ts) {
    TSeries r(ts);

    //----------------------------------  Initialize at start.
    if (!mStartTime) {
        mStartTime = ts.getStartTime();
	roll();
    }
    mCurrentTime = r.getStartTime();
    Time tEnd    = r.getEndTime();
    TSeries::size_type nByt = (r.refDVect())->getSize();
    TSeries::size_type N    = r.getNSample();

    //----------------------------------  Emulate swap errors
    if (testType(tSwap) && nByt == 2) {
	short* p = reinterpret_cast<short*>(r.refData());
	for (TSeries::size_type i=0 ; i<N; i+=2) {
	    short p0 = *p;
	    short p1 = p[1];
	    *p++ = p1;
	    *p++ = p0;
	}
    }

    //----------------------------------  Emulate swap errors
    if (testType(tLSB)) {
        int test(1);
	bool lEnd = *reinterpret_cast<const char*>(&test);
        char* p = reinterpret_cast<char*>(r.refData());
        while (mLsbErrTime <= tEnd) {
	    if (mLsbErrTime > mCurrentTime) {
	        TSeries::size_type iBin = r.getBin(mLsbErrTime);
		TSeries::size_type iByt = iBin * nByt;
		iByt -= (iByt & 15);
		for (int i=0; i<mClustWidth; i++) {
		    TSeries::size_type inx = iByt + 16*i;
		    if (lEnd) inx += 8;   // little endian
		    else      inx +=11;   // big endian
		    if (inx < N*nByt) p[inx] |= 0xff;
		}
	        mCurrentTime = mLsbErrTime;
	    }
	    roll();
	}
    }

    //----------------------------------  Done, go away
    mCurrentTime = tEnd;
    return r;
}

//======================================  List a type string
DaqErrFilter*
DaqErrFilter::clone(void) const {
    return new DaqErrFilter(*this);
}

//======================================  List a type string
string 
DaqErrFilter::getTypeString(void) const {
    string s;
    for (int i=0; i<tMax; i++) {
	if ((mSubTypes & (1<<i)) != 0) {
	    if (!s.empty()) s += ",";
	    switch (i) {
	    case tSwap:
	        s += "swap";
	        break;
	    case tLSB:  
	        s += "lsb";
	        break;
	    default:
	        break;
	    }
	}
    }
    return s;
}

//======================================  Reset errors
void 
DaqErrFilter::reset(void) {
    mStartTime = Time(0);
}

//======================================  Generate a random lsb error event
void 
DaqErrFilter::roll(void) {
    if (!testType(tLSB) || !mErrRate) return;
    mLsbErrTime = mCurrentTime;
    do {
        //------------------------------  Generate a random width.
        if (mAvgWidth != 0.0) {
	    mClustWidth = PoissonRndm(mAvgWidth);
	}

	//------------------------------  Throw  random time.
	//                                Force 1/16th second granularity
	double dT = -log(Rndm())/mErrRate;
	dT -= dT - long(16.0*dT) / 16.0;
	mLsbErrTime += Interval(dT);
    } while (!mClustWidth && mAvgWidth);
}

//======================================  Enable an error type
bool 
DaqErrFilter::selectType(const std::string& s) {
    if (s == "swap") {
	mSubTypes |= (1<<tSwap);
    }
    else if (s == "lsb") {
	mSubTypes |= (1<<tLSB);
    }
    else {
        cout << "Unknown DAQ error type: " << s << endl;
	return false;
    }
    return true;
}

//======================================  Set the average error cluster width
void 
DaqErrFilter::setAvgWidth(double avgWid) {
    mAvgWidth = avgWid;
}

//======================================  Set a fixed error cluster width
void 
DaqErrFilter::setClusterWidth(long wid) {
    mClustWidth = wid;
}

//======================================  Set a error cluster rate
void 
DaqErrFilter::setErrRate(double f) {
    mErrRate = f;
}

//======================================  Set the subtypes
void 
DaqErrFilter::setSubTypes(const std::string& s) {
    mSubTypes = 0;
    for (string::size_type i=0; i < s.size(); ) {
        string::size_type l = s.find(",", i);
	selectType(s.substr(i,l-i));
	if (l == s.npos) break;
	i = l+1;
    }
}
