/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "Generator.hh"
#include "FrWriter.hh"
#include "SigFlag.hh"
#include "TrigPar.hh"
#include "frame_name.hh"
#include <string>
#include <fstream>
#include <map>

/**  The DMTGen class contains all the driver functionality for the DMT data
  *  generation program \ref DMTGen .
  *  \brief DMT data simulation driver program.
  *  \author J. Zweizig
  *  \version 1.4; Last Modified October 30, 2010
  */
class DMTGen {
public:
    /**  DMT signal generation class constructor.
      *  \brief Construct generation driver.
      *  \param argc Number of command line argument words.
      *  \param argv Vector of command line argument strings.
      */
    DMTGen(int argc, const char* argv[]);

    /** Destroy driver program
      * \brief Destructor
      */
    ~DMTGen(void);

    /**  Read a configuration file.
      *  \brief Read configuration.
      *  \param file Configuration file path.
      *  \return True if error occurred in parsing file.
      */
    bool readConfig(const std::string& file);

    /**  Data generation entry
      *  \brief Generate data.
      *  \return Generation failed if return code is non-zero.
      */
    int generate(void);

    /**  Write a frame of generated data.
      *  \brief Write generated data.
      *  \param t0 Start time.
      *  \param dT Lengt of frame to be written.
      */
    void writeFrame(const Time& t0, Interval dT);

    /**  Test whether configuration was successful
      *  \brief Test configuration.
      *  \return True if successful configuration.
      */
    bool configured(void) const;

private:
    typedef std::map<std::string, const Pipe*> FilterMap;
    typedef FilterMap::const_iterator          const_filt_iter;
    typedef FilterMap::iterator                filt_iter;

private:
    bool                 mConfigured;
    FilterMap            mFilterMap;
    generator::SourceDict mSourceDict;
    std::string          mIdent;
    TrigPar              mDict;
    generator::Generator mGener;
    long                 mRunID;
    frame_name           mFrameName;
    enum {kFile, kSMP}   mWriteMode;
    FrWriter             mWriter;
    int                  mVerbStat;
    std::ofstream        mVerbStream;
    Time                 mStartTime;
    Time                 mEndTime;
    Interval             mDt;
    int                  mDebug;
    int                  mNFramesGen;
    int                  mNFramesOut;
    int                  mRealTime;
    SigFlag              mTerm;
};

//======================================  DMTGen inline functions
inline bool 
DMTGen::configured(void) const {
    return mConfigured;
}
