//
//  Class FDEasyCalibrate: High-level class for calibrating ASQ power spectra
//  into units of physical displacement (nanometers or strain).
//    version:  0.3
//       date:  2004.09.24   
//     author:  Patrick J. Sutton (psutton@ligo.caltech.edu)
//
///////////////////////////////////////////////////////////////////////////


/* Procedure for using FDEasyCalibrate:
 * 
 * 1. CALL ONCE: Constructor.  
 * 2. CALL MANY: Update response function.  
 * 3. CALL MANY: Apply.  
 *    Inherited from FDFilter::Apply - no need for different method. 
 * 
 */


#ifndef FDEasyCalibrate_HH
#define FDEasyCalibrate_HH

#include "FDCalibrate.hh"

#ifndef __CINT__
//#include "FSpectrum.hh"
//#include "LineFilter.hh"
#include "Time.hh"
//#include "TSeries.hh"
#include "DaccAPI.hh"
#endif  //----- __CINT__


/**  The FDEasyCalibrate class is a high-level class to perform  
  *  frequency-domain calibration of FSpectrum (PSD) and FSeries (fSeries) 
  *  objects.  It drives the FDCalibrate class for the typical user, which 
  *  in turn converts FSpectrum (PSD) data from units of raw channel counts 
  *  to physical displacement in nm.
  *  @memo Class for calibrating ASQ FSpectrum data.
  */
class FDEasyCalibrate : public FDCalibrate {


public:


  /**  Constructor based on an uber reference-calibration file.
    *  The frequency-dependent reference
    *  calibration data is linearly interpolated to the specified
    *  frequencies.  The response function is initialized to its
    *  default value (alpha=1, beta=1).
    *  Alpha and beta values stored in the reference
    *  file can be used instead of generating new calibrations by
    *  setting the bool argument \c GenerateCalParam to \c false.
    *  @memo Constructor using reference calibration file.
    *  @param access  Data accessor.
    *  @param refcal_file File containing reference calibration data.
    *  @param GenerateCalParam Generate new alpha, beta parameters from data if true.
    *  @param Fmin  Minimum frequency at which calibration is to be evaluated.
    *  @param Fstep Spacing of frequencies at which calibration is to be evaluated.
    *  @param npoint Number of frequencies at which calibration is to be evaluated.
    */
  FDEasyCalibrate(DaccAPI *access, const char *refcal_file, 
		  bool GenerateCalParam,
		  double Fmin, double Fstep, int npoint);

  /** Destructor.
   */
  ~FDEasyCalibrate(void);

  
  /**  Update response function by estimating calibration parameters 
    *  from current data.  Returns zero if new calibration is physically 
    *  valid.  If the new calibration is not physically valid as defined 
    *  by IsCalibrationGood(), then the  
    *  response function is reset to its reference value (its value for 
    *  alpha=1=beta).
    *  Once the response function is set, an FSpectrum in units AS_Q^2/Hz 
    *  can be calibrated to nm^2/Hz by simply calling one of the Apply() 
    *  methods inherited from FDFilter.
    *  \brief Update response function with estimated calibration parameters 
    *  \return Zero if response function is valid.
    */
  int UpdateResponseFunction(void);


  /**  Update response function using stored calibration parameters. 
    *  Returns zero if requested calibration is available and physically valid.
    *  If the new calibration is not physically valid as defined 
    *  by IsCalibrationGood(), then the  
    *  response function is reset to its reference value (its value for 
    *  alpha=1=beta).
    *  Once the response function is set, an FSpectrum in units AS_Q^2/Hz 
    *  can be calibrated to nm^2/Hz by simply calling one of the Apply() 
    *  methods inherited from FDFilter.
    *  \brief Update response function using stored calibration parameters. 
    *  \param t Time for which alpha, beta values are to be fetches
    *  \return Zero if response function is valid.
    */
  int UpdateResponseFunction(Time t);

};


#endif  //----- FDEasyCalibrate_HH
