/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef FIXEDLENTS_HH
#define FIXEDLENTS_HH

#include "TSeries.hh"
#include <stdexcept>

/**  The fixed length time series is defined in
  *
  *  <tt> \#include "FixedLenTS.hh" </tt>
  *
  *  You set the maximum time either in the constructor, e.g.
  *
  *  <tt>FixedLenTS mHistory(7200.0);</tt>
  *
  *  Or explicitly 
  *
     <tt>FixedLenTS mHistory; <br>
     mHistory.setMaxLen(t);</tt>
  *
  *  You then append data like....
  *  
  *  <tt>
  *  //----------------------------------  Keep the history <br>
  *  Time t0 = fs.getStartTime(); <br>
  *  Interval dT = fs.getEndTime()-t0; <br>
  *  mTimeTot += dT; <br>
  *  float bandP = mBand; <br>
  *  mHistory.fixedAppend(t0, dT, &bandP); </tt>
  *
  *  When it reaches the maximum it starts to delete data from the start,
  *  rather than increasing the length. The FixedLenTS is based on a TSeries 
  *  so you can reference it as you would a TSeries.
  *  @brief Fixed length Time series
  *  @author J. Zweizig
  *  @version 1.2; Last modified February 17, 2006
  */
class FixedLenTS : public TSeries {
public:

  /**  Default constructor creates a fixed length TSeries with zero length.
    *  A <tt>FixedLenTS</tt> created with this constructor is unusable until
    *  the maximum length has been set with <tt>setMaxLen()</tt>
    *  @brief Default constructor.
    */
  FixedLenTS(void);

  /**  Construct a  <tt>FixedLengthTS</tt> with a specified maximum length.
    *  @brief Data constructor.
    *  @param max maximum length for time series.
    */ 
  explicit FixedLenTS(Interval max);

  /**  Create a FixedLenTS that is an identical copy of the argument series.
    *  \brief Copy constructor.
    *  \param x %FixedLenTS series to be copied
    */
  FixedLenTS(const FixedLenTS& x);

  /**  Destroy the Fixed length time series.
    *  @brief Destructor.
    */
  inline virtual ~FixedLenTS() {}

  /**  Append one or more float data words to the series. If the series 
    *  with the appended data is longer than the specified maximum length,
    *  data are removed from the beginning of the series and the start
    *  time is updated as appropriate.
    *  @brief Append float data.
    *  @param t0 Time of first datum.
    *  @param dT Time step between data.
    *  @param x  Pointer to one or more data.
    *  @param n  Number of data to be appended to the time series.
    *  @return
    *   <table>
    *    <tr><td>  0 </td><td> successful</td></tr>
    *    <tr><td> -1 </td><td> ts start time isn't end+N*dt </td></tr>
    *    <tr><td> -2 </td><td> sample intervals aren't equal.</td></tr>
    *   </table>
    */
  int fixedAppend(const Time& t0, Interval dT, float* x, int n=1);

  /**  Append a TSeries to the fixed length series. If the resulting series 
    *  would be longer than the specified maximum length, data are removed 
    *  from the start of the current series and/or the start of the data to 
    *  be appended and the current series start time is updated as appropriate.
    *  @brief Append a TSeries.
    *  @param ts TSeries to be appended.
    *  @return
    *   <table>
    *    <tr><td>  0 </td><td> successful</td></tr>
    *    <tr><td> -1 </td><td> ts start time isn't end+N*dt </td></tr>
    *    <tr><td> -2 </td><td> sample intervals aren't equal.</td></tr>
    *   </table>
    */
  int fixedAppend(const TSeries& ts);

  /**  Append zeros to the series until the specified end-time.
    *  @brief Append zeroes until the specified time.
    *  @param t0 Time of first datum.
    *  @param dT Time step between data.
    */
  void padUntil(const Time& t0, Interval dT);

  /**  Set the maximum length for the fixed length time series.
    *  @brief specify the maximum length.
    *  @param dt Maximum length
    */
  void setMaxLen(Interval dt);

  /**  Get the Maximum length specified.
    *  @brief  Get the Maximum length.
    *  @return Maximum length.
    */
  Interval getMaxLen(void) const;

private:
  Interval maxLen;
};

//======================================  Inline functions
inline
FixedLenTS::FixedLenTS(void)
  : maxLen(0) 
{}

inline
FixedLenTS::FixedLenTS(Interval max)
  : maxLen(max) 
{}

inline
FixedLenTS::FixedLenTS(const FixedLenTS& x)
  : TSeries(x), maxLen(x.maxLen)
{}

inline void 
FixedLenTS::setMaxLen(Interval dt){
    maxLen = dt;
}

inline Interval
FixedLenTS::getMaxLen(void) const {
    return maxLen;
}

inline int
FixedLenTS::fixedAppend(const TSeries& ts) {
    int rc = 0;
    if (ts.empty()) return rc;
    Time tStart = ts.getEndTime() - maxLen;
    if (!empty() && getEndTime() > tStart) {
        if (tStart > getStartTime()) eraseStart(tStart - getStartTime());
	if (getEndTime() < ts.getStartTime()) extend(ts.getStartTime());
	rc = TSeries::Append(ts);
    } else if (ts.getStartTime() < tStart) {
        TSeries::operator=(ts.extract(tStart, maxLen));
    } else {
        TSeries::operator=(ts);
    }
    return rc;
}

inline int
FixedLenTS::fixedAppend(const Time& t0, Interval dT, float* x, int n) {
    return fixedAppend(TSeries(t0, dT, n, x));
}

inline void
FixedLenTS::padUntil(const Time& t0, Interval dT) {
    if (!dT) throw std::runtime_error("FixedLenTS::padUntil invalid dT");
    Time tStart = t0 - maxLen;
    if (empty() || getEndTime() <= tStart) {
	float data(0.0);
        setData(tStart, dT, &data, 1);
    } else if (tStart > getStartTime()) {
	eraseStart(tStart - getStartTime());
    }
    if (dT != getTStep()) 
        throw std::runtime_error("FixedLenTS::padUntil invalid dT");
    extend(t0);
}

#endif // FIXEDLENTS
