/** @name when
  *  The #when# command converts a GPS time to a string. The command line 
  *  syntax is:
  * \begin{verbatim}
   when [-u | -l] [<gps>] [<fmt>]
    \end{verbatim}
  *  Where: \\
  * \begin{tabular}{ll} \\
  * #-u#    & requests printout of UTC time. \\
  * #-l#    & requests printout of local time (default). \\
  * #<gps># & is the time to be printed. If #<gps># is 0 or not specified 
  *           the current time is printed. \\
  * #<fmt># & specifies the output format using 
  *           \URL[TimeStr]{../Base/TimeStr.html} escape characters.
  * \end{tabular}
  * Example command:
  * \begin{verbatim}
    $ when 830880189 "%M %d, %Y %H:%N:%S"
    May 5, 2006 9:02:55
    \end{verbatim}
  * @memo GPS time conversion program.
  * @author J. Zweizig
  * @version 1.2; Last modified May 5, 2005
  */
//@{
//@}

//   when -  Print out the specified UTC time in a desired format.
//
//
#include "Time.hh"
#include <stdlib.h>
#include <iostream>
using namespace std;

int
main(int argc, const char* argv[]) {
    Time::ulong_t t=0;
    bool syntax(false), utc(false);

    //----------------------------------  Get local/utc options
    int inx=1;
    while (inx < argc && *argv[inx] == '-') {
        if (*(argv[inx]+1) == 'u') {
	    utc = true;
	} else if (*(argv[inx]+1) == 'l') {
	    utc = false;
	} else {
	    cerr << "Unrecognized option: " << argv[inx] << endl;
	    syntax = true;
	}
	inx++;
    }

    //----------------------------------  Get time
    if (inx < argc) {
        char* ptr;
        t = strtol(argv[inx++], &ptr, 0);
	if (*ptr != 0) syntax = true;
    }

    //----------------------------------  Get format
    const char* fmt = "%M %d, %Y %H:%N:%S";
    if (inx < argc) fmt = argv[inx++];
    if (syntax) {
        cerr << "Error in command. Syntax:" << endl;
	cerr << argv[0] << " [-u | -l] [<GPS-time>] [<format>]" << endl;
	return 1;
    }

    //----------------------------------  Print it out
    Time time(t);
    if (!time) time = Now();
    char line[80];
    if (utc) TimeStr (time, line, fmt);
    else     LocalStr(time, line, fmt);
    cout << line << endl;

    //----------------------------------  Done
    return 0;
}
