/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TABLE_TABLE_HH
#define TABLE_TABLE_HH

#include "column.hh"

//======================================  Table class definition

/**  The table namespace contains the table definition and manipulation 
  *  classes used by the tablepgm table manipuation program.
  *  @memo  Namespace containing classes used by tablepgm.
  *  @author John Zweizig (john.zweizig@ligo.org).
  *  @version 1.0; Last Modified September 15, 2010
  */
namespace table {

    /**  Table data are are organized as a vector of table columns. table 
      *  class methods allow easy access ann manipulation of the stored data.
      *  @memo Table storage and manipulation class.
      *  @author John Zweizig (john.zweizig@ligo.org).
      *  @version 1.12; Last Modified November 18, 2009
      */
    class Table {
    public:
	/** Column index data type.
	 */
	typedef unsigned int col_index;

	/**  Row index data type.
	 */
	typedef Column::row_index row_index;

	/** Sort algorithm.
	 */
	enum sort_enum {
	    kAuto,    ///< Use internal rule to decide
	    kMerge,   ///< Use a merge sort (order N*log(N) )
	    kBubble   ///< Use a bubble sore (order N^2)
	};
    public:
	/**  Null table constructor.
	  *  \brief default contrictor.
	  */
	Table(void) : mRow(0), mDebug(0) {}

	/**  Named table constructor.
	  *  \brief Contruct and fill a named table.
	  *  \param title Table title string.
	  *  \param in    Input data file name.
	  */
	Table(const std::string& title, const std::string& in);

	/**  Add an empty row to each column.
	  *  \brief Add a row.
	  *  \param b4   Row number.
	  *  \param init Contents of cell to be added t each row
	  */
	void addRow(row_index b4=(~0), const Cell& init=Cell(""));

	/**  Check if table is defined.
	  *  @memo Is table defined
	  *  @return true if table structure is defined.
	  */
	bool defined(void) const;

	/**  Check if table has any rows
	  *  @memo Does table have rows?
	  *  @return true if columns are empty.
	  */
	bool empty(void) const;

	/**  Check if column is defined.
	  *  @memo Does column exist
	  *  @param name Column name.
	  *  @return true if column is defined.
	  */
	bool exists(const std::string& name) const;

	/**  Get the index of the row with the specified table. If the 
	  *  specified column name is not found, the value returned is 
	  *  equal to the table size (i.e. the number of columns).
	  *  @memo Get column index
	  *  @param name name of column to be found
	  *  @return Column index
	  */
	col_index find(const std::string& name) const;

	/**  Return a reference to the named column.
	  *  \brief Reference a named column
	  *  \param name Column name.
	  *  \return Reference to the named column.
	  */
	Column& findColumn(const std::string& name);

	/**  Return a constant reference to the named column.
	  *  \brief Reference a named column
	  *  \param name Column name.
	  *  \return Constant reference to the named column.
	  */
	const Column& findColumn(const std::string& name) const;

	/**  Get a constant pointer to the table title string.
	  *  \brief Get te table title.
	  *  \return Constnt pointer to the table title string.
	  */
	const char* getTitle(void) const;

	/**  Get the number of rows in each column.
	  *  \brief Get the number of rows.
	  *  \return The number of rows.
	  */
	Column::row_index getNRows(void) const;

	/**  Add a column in before the specified column index. The number
	  *  of rows in the new column will be the same as for the existing
	  *  columns, and theinserted cells will be empty.
	  *  \brief Add a column
	  *  \param b4 Column before which the new column will be inserted.
	  *  \param title Name of the new column.
	  */
	void insertColumn(col_index b4, const std::string& title);

	/**  Copy/insert rows from a specified table into the current 
	  *  table. The rows are copied to columns of the same name. 
	  *  The argument table must have at least the same column
	  *  names as the current table, and may contain additional 
	  *  columns.
	  *  \brief Add rows from another table.
	  *  \param b4 Row before which the new data will be inserted.
	  *  \param tab Table containing data to beinserted.
	  */
	void insertRows(Column::row_index b4, const Table& tab);

	/**  Merge the contents of two columns. The string representations 
	  *  are concatinated with the join string.
	  *  \brief Merge two columns
	  *  \param iCol Column to be merged/modified
	  *  \param jCol Column to merge with column \c iCol.
	  *  \param join String to insert between the two original strings.
	  */
	void merge(col_index iCol, col_index jCol, const std::string& join);

	/**  Move the row specified by the "from" index to the position 
	  *  specified by the "to" index. 
	  *  \brief Move a row inside a table.
	  *  \param from Row number.
	  *  \param to   Row number if front of whic the row is to be moved.
	  */
	void move_row(Column::row_index from, Column::row_index to);

	/**  Read data from the specified file into the table. The file 
	  *  format is inferred from the file name extension as xml ("*.xml") 
	  *  or text.
	  *  \brief Read in a table.
	  *  \param in Input file name.
	  */
	void read(const std::string& in);

	/**  Read data from a text file into the table.
	  *  \brief Read a txt table.
	  *  \param in Input file name.
	  */
	void readtxt(const std::string& in);

	/**  Read data from an xml file into the table.
	  *  \brief Read an xml table.
	  *  \param in Input file name.
	  */
	void readxml(const std::string& in);

	/**  Remove a specified column.
	  *  \brief Remove a column.
	  *  \param col Index of column to be removed.
	  */
	void removeColumn(col_index col);

	/**  Copy all data from specified rows to the start of the table
	  *  and delete empty rows.
	  *  \brief Select rows.
	  *  \param mask Boolean vector indicating which rows to retain.
	  */
	void selectRows(const std::vector<bool>& mask);

	/**  Set the debug printout level
	  *  \brief Set the debug level. 
	  *  \param i Debug printout level.
	  */
	void setDebug(int i);

	/**  Return the number of columns currently defined.
	  *  \brief Number of columns
	  *  \return Number of columns.
	  */
	col_index size(void) const;

	/**  Reorder the rows to put the cell in the specified column in 
	  *  increasing (decreasing) order.
	  *  \brief Sort the table
	  *  \param col Column to be put into increasing (decreasing) order.
	  *  \param decrease If true, rows of specified column are sorted into
	  *                  decreasing order.
	  *  \param t        sort type
	  */
	void sort(col_index col, bool decrease=false, sort_enum t=kAuto);

	/**  Print status to cout
	 */
	void status(void) const;

	/**  Exchange the cell contents of the two specified rows in all
	  *  columns of the table.
	  *  \brief Swap row contents.
	  *  \param i One column.
	  *  \param j The other column
	  */
	void swap_rows(Column::row_index i, Column::row_index j);

	/**  Write out the table to the specified file name. The output format
	  *  is detemined by the file name extension. Text files contain 
	  *  information specified by the flag word: a header (1), the table 
	  *  body (2) and a summary line (4).
	  *  @memo write the table to a specified file.
	  *  @param f The output file name.
	  *  @param flg Word containing Ored flags.
	  *  @param t title?
	  */
	void write(const std::string& f, int flg=7, const std::string& t=0) const;

	/**  Write the table to an html file.
	  *  @memo Write an html file.
	  *  @param out Output file stream
	  *  @return Reference to the output file stream.
	  */
	std::ostream& writehtml(std::ostream& out) const;

	/**  Write the table to a text file.
	  *  @memo Write a text file.
	  *  @param out Output file stream
	  *  @param flags Bit mask indicating which segments of the table to
	  *               write to the file: 1=header, 2=body, 3=summary.
	  *  @return Reference to the output file stream.
	  */
	std::ostream& writetxt(std::ostream& out, int flags=7) const;

	/**  Write the table to an xml file.
	  *  @memo Write an xml file.
	  *  @param out Output file stream
	  *  @return Reference to the output file stream.
	  */
	std::ostream& writexml(std::ostream& out) const;

	/**  Return a reference to the indexed column.
	  *  \brief Reference a column.
	  *  \param col Index of column to be referenced.
	  *  \return Reference to the indexed column.
	  */
	Column& operator[](col_index col);

	/**  Return a constant reference to the indexed column.
	  *  \brief Reference a column.
	  *  \param col Index of column to be referenced.
	  *  \return Constant reference to the indexed column.
	  */
	const Column& operator[](col_index col) const;

	/**  Return a reference to the named column.
	  *  \brief Reference a column.
	  *  \param col Name of column to be referenced.
	  *  \return Reference to the named column.
	  */
	Column& operator[](const std::string& col);

	/**  Return a constant reference to the named column.
	  *  \brief Reference a column.
	  *  \param col Name of column to be referenced.
	  *  \return Constant reference to the named column.
	  */
	const Column& operator[](const std::string& col) const;

    private:
	void bsort(col_index col, bool decrease);
	void bsort(std::vector<double> index, bool decrease);
	void msort(col_index col, bool decrease);
	void msort(std::vector<double> index, bool decrease);
    private:
	Column::row_index mRow;
	std::string mTitle;
	std::vector<Column> mTable;
	int mDebug;
    };

    //==================================  Inline methods.
    inline bool
    Table::defined(void) const {
	return !mTable.empty();
    }

    inline bool
    Table::empty(void) const {
	return !mRow;
    }

    inline Table::col_index
    Table::size(void) const {
	return mTable.size();
    }

    inline bool
    Table::exists(const std::string& name) const {
	return find(name) != mTable.size();
    }

    inline Column::row_index 
    Table::getNRows(void) const {
	return mRow;
    }

    inline const char* 
    Table::getTitle(void) const {
	return mTitle.c_str();
    }

    inline Column& 
    Table::operator[](col_index col) {
	return mTable[col];
    }

    inline const Column& 
    Table::operator[](col_index col) const {
	return mTable[col];
    }

    inline Column& 
    Table::operator[](const std::string& col) {
	return findColumn(col);
    }

    inline const Column& 
    Table::operator[](const std::string& col) const {
	return findColumn(col);
    }

}
#endif // !defined(TABLE_TABLE_HH)
