/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "lsmp_con.hh"
#include "framexmit/framesend.hh"
#include "SigFlag.hh"
#include "Time.hh"
#include <iostream>
#include <fstream>
#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <string>
#include <stdexcept>
#include <fcntl.h>
#include <unistd.h>

struct buffer {
    static int bsize;
    buffer(void)
	: inUse(false), data(new char[bsize])
    {}

    bool           inUse;
    char*          data;
};

int buffer::bsize = 1000000;

inline std::ostream&
operator<<(std::ostream& out, const framexmit::frameSend::frsend_stats& s) {
    out << "send requests:      " << s.send_requests      << std::endl;
    out << "send_packets:       " << s.send_packets       << std::endl;
    out << "resend_requests:    " << s.resend_requests    << std::endl;
    out << "resend_nodata:      " << s.resend_nodata      << std::endl;
    out << "resend_toomanypkts: " << s.resend_toomanypkts << std::endl;
    out << "resend_invalidpkt:  " << s.resend_invalidpkt  << std::endl;
    out << "resend_puterror:    " << s.resend_puterror    << std::endl;
    out << "resend_packets:     " << s.resend_packets     << std::endl;
    out << "total_sent:         " << s.total_sent         << std::endl;
    out << "total_bytes:        " << s.total_bytes        << std::endl;
    return out;
}

//======================================  Program class
class lsmp_xmit {
public:
    lsmp_xmit(int argc, const char* argv[]);
    ~lsmp_xmit(void);
    int process(void);
    bool daemon_me(void) const;
    const char* log_file(void) const;

private:
    bool        demonize;
    const char* logfile;
    const char* verbose;
    const char* bcast_net;
    std::string mcast_addr;
    const char* partition;
    int         mcast_port;
    unsigned int dt;
    useconds_t  delay;  /// number of useconds to wait before a multicast.
};

inline bool
lsmp_xmit::daemon_me(void) const {
    return demonize;
}

inline const char* 
lsmp_xmit::log_file(void) const {
    return logfile;
}

using namespace std;

//======================================  Program
int
main(int argc, const char* argv[]) {
    int rc = 1;
    try {
	lsmp_xmit mon(argc, argv);

	//-------------------------------------  daemonize
	if (mon.daemon_me()) {
	    switch (fork()) {
	    case -1:
		return 1;
	    case 0:
		break;
	    default:
		return 0;
	    }
	}

	//------------------------------  Optionally switch to a log file
	const char* logfile=mon.log_file();
	if (logfile) {
	    int flags = O_WRONLY | O_CREAT;
	    int fd = open(logfile, flags, 0664);
	    close(1); // stdout
	    if (dup(fd) != 1) {
		cerr << "stdout not diverted!" << endl;
	    }
	    close(2); // stderr
	    if (dup(fd) != 2) {
		cout << "stderr not diverted!" << endl;
	    }
	    close(fd);
	}

	//------------------------------  Munchy crunchy, munchy crunchy
	rc = mon.process();
    }

    //----------------------------------  Catch standard exceptions
    catch (exception& e) {
	cerr << "Caught exception: " << e.what() << endl;
	rc = 1;
    }
    catch (...) {
	cerr << "Caught unidentified exception." << endl;
	rc = 1;
    }
    return rc;
}

//======================================  Construct program class
lsmp_xmit::lsmp_xmit(int argc, const char* argv[]) 
     : demonize(false), logfile(0), verbose(0), bcast_net("localhost"), 
       mcast_addr("226.1.2.3"), partition("Fast_Data"), 
       mcast_port(framexmit::frameXmitPort), dt(1), delay(0)
{
    bool syntax = false;
    //----------------------------------  Parse optional arguments
    int keyarg = 1;
    for ( ; keyarg<argc && *argv[keyarg] == '-'; keyarg++) {
	string argi = argv[keyarg];
	if (argi == "--daemonize" || argi == "-d") {
	    demonize = true;
	}
	else if (argi == "--delay") {
	    double rdelay = strtod(argv[++keyarg], 0);
	    if (rdelay <= 0) delay = 0;
	    else             delay = uint32_t(1000000.0*rdelay+0.5);
	}
	else if (argi == "--help") {
	    syntax = true;
	}
	else if (argi == "--log" || argi == "-l") {
	    logfile = argv[++keyarg];
	}
	else if (argi == "--verbose" || argi == "-v") {
	    verbose = argv[++keyarg];
	}
	else {
	    cerr << "Unrecognized argument: " << argi << endl;
	    syntax = true;
	}
    }

    //----------------------------------  Parse positional arguments
    if (argc>keyarg)   partition = argv[keyarg];
    if (argc>keyarg+1) bcast_net = argv[keyarg+1];
    if (argc>keyarg+2) mcast_addr = argv[keyarg+2];
    if (argc>keyarg+3) dt = strtoul(argv[keyarg+3], 0, 0);

    //----------------------------------  Dig out the port number.
    string::size_type pinx = mcast_addr.find(":");
    if (pinx != string::npos) {
	mcast_port = strtol(mcast_addr.c_str()+pinx+1, 0, 0);
	mcast_addr.erase(pinx);
    }

    //----------------------------------  See if arguments
    if (syntax) {
	cout << "Frame multicast server command syntax:" << endl << endl;
	cout << "lsmp_xmit [--daemonize] [--log <logfile>] [--help] " << endl;
	cout << "          [--delay <delay>] [--verbose <vfile>] \\" << endl;
	cout << "          [<partition> [<mcast-net> [<mcast-ip> [<dt>]]]]"
	     << endl << endl;
	cout << "where:" << endl;
	cout << "<logfile>   Name of file to receive log output" << endl;
	cout << "<delay>     delay (in seconds) before multicasting a frame."
	     << endl;
	cout << "<vfile>     Name of file to receive verbose output" << endl;
	cout << "<partition> Name of partition with frames to be multicast"
	     << endl;
	cout << "<mcast-net> Address on network interface receive multicast"
	     << endl;
	cout << "<mcast-ip>  Multicast ip:port" << endl;
	cout << "<dt>        Frame length (in seconds)" << endl;
	throw runtime_error("lsmp_xmit command syntax error");
    }
}

//======================================  Destroy program class
lsmp_xmit::~lsmp_xmit(void) {
}

//======================================  Processing loop
int
lsmp_xmit::process(void) {
    const int   bufnum  = 8;

    //----------------------------------  Print configuration info
    cout << "Starting lsmp_xmit" << endl
	 << "  lsmp_xmit version: " << "$Id: lsmp_xmit.cc 7236 2014-11-30 12:32:02Z john.zweizig@LIGO.ORG $" << endl
	 << "  Compilation date:  " << __DATE__ << " " << __TIME__ << endl
	 << "  partition name:    " << partition << endl
	 << "  multicast network: " << bcast_net << endl
	 << "  multicast ip-addr: " << mcast_addr << endl
	 << "  multicast ip-port: " << mcast_port << endl
	 << "  frame duration:    " << dt << endl
	 << "  multicast delay:   " << delay/1000000.0 << endl;

    //----------------------------------  Open the input partition
    LSMP_CON smpin(partition);
    buffer::bsize = smpin.getBufferLength();

    //----------------------------------  Slip in number and size of buffers
    cout << "  number of buffers: " << bufnum << endl
	 << "  buffer size:       " << buffer::bsize << endl;

    //----------------------------------  Allocate a bunch of buffers.
    buffer* bufs = new buffer[bufnum];

    //----------------------------------  Optionally open verbose stream
    ofstream vstream;
    if (verbose) {
	vstream.open(verbose, ofstream::out | ofstream::app);
	if (!vstream.is_open()) {
	    cerr << "Unable to open verbose file: " << verbose << endl;
	    verbose = 0;
	}
    }

    //----------------------------------  Diddle the framexmit parameters
    framexmit::frameSend fx;
    framexmit::par.set_parameter("sndMaxPacketRebroadcast", 3);
    framexmit::par.set_parameter("sndRebroadcastEpoch", 250000);
    framexmit::par.init();
    framexmit::par.display(cout);

    //----------------------------------  Open a multicaster
    cout << "Opened multicast to: " << mcast_addr << ":" << mcast_port 
	 << " over net with: " << bcast_net << endl;
    if (!fx.open(mcast_addr.c_str(), bcast_net, mcast_port)) {
	cerr << fx.last_message() << ": " << strerror(fx.last_errno()) << endl;
	return 1;
    }

    SigFlag sf(SIGTERM);
    sf.add(SIGINT);

    //----------------------------------  Loop over frames until signal.
    while (!sf) {

	//------------------------------  Get an smp frame buffer.
	const char* inbuff = smpin.get_buffer();
	if (!inbuff) {
	    cerr << "Unable to get buffer from: " << partition << endl;
	    perror("Error in get_buffer");
	    break;
	}

	//------------------------------  Get frame parameters
	Time now(Now());
	int datalen = smpin.getLength();
	unsigned long evtID = smpin.getEvtID();
	if (verbose) {
	    vstream << now.getS() << " get " << evtID << " " << datalen
		    << " OK." << endl;
	}

	//------------------------------  Add a fixed delay
	if (delay) {
	    int sleep_rc = usleep(delay);
	    if (sleep_rc < 0) {
		perror("error in usleep");
	    }
	}

	//------------------------------  Get a framexmit buffer.    
	now = Now();
	if (verbose) {
	    vstream << now.getS() << " put " << evtID << " " << datalen;
	    vstream.flush();
	}
	buffer* buf = 0;
	for (int i=0; i<bufnum; ++i) {
	    if (!fx.isUsed(bufs[i].inUse)) {
		buf = bufs + i;
		break;
	    }
	}
 
	if (!buf) {
	    if (verbose) {
		vstream << " can't get unused buffer." << endl;
		vstream.flush();
	    }
	    cerr << "Cannot find unused buffer." << endl;
	    break;
	}

	//------------------------------  Dump some statistics
	if ((evtID % 10800) == 0) {
	    cout << "framexmit statistics at gps " << evtID << endl;
	    cout << fx.statistics() << endl;
	}

	//------------------------------  Copy the data
	if (datalen <= buffer::bsize) {
	    memcpy(buf->data, inbuff, datalen);
	    smpin.free_buffer();
	    //cout << "Sending buffer: " << evtID << " length: " << datalen 
	    //     << endl;
	    bool ok = fx.send(buf->data, datalen, &(buf->inUse), false, 
			      evtID, dt);

	    if (!ok) {
		cerr << "frameSend failed, ID=" << evtID << " length: " 
		     << datalen << endl;
		cerr << "framesend error: " << fx.last_message() << ": "
		     << strerror(fx.last_errno()) << endl;
		if (verbose) {
		    vstream << " frameSend failed. " << fx.last_message() 
			    << ": " << strerror(fx.last_errno()) << endl;
		    vstream.flush();
		}
		break;
	    }
	    if (verbose) {
		vstream << " OK." << endl;
		vstream.flush();
	    }
	} else {
	    cerr << "Data for event " << evtID << " too long to transmit ("
		 << datalen << " bytes)." << endl;
	    smpin.free_buffer();
	}
   }
    if (sf) cerr << "lsmp caught a TERM signal." << endl;
    if (verbose) {
	vstream << " -------- lsmp_xmit completed GPS=" << Now().getS() << endl;
	vstream.flush();
    }

    fx.close();
    cout << "lsmp_xmit has finished." << endl;
    //smpin.close();
    return 0;
}
