/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "NDS2Socket.hh"
#include "FrWriter.hh"
#include "TSeries.hh"
#include <vector>
#include <stdexcept>
#include <string>
#include <iostream>
#include <cstdlib>

//========================================  NDS2Frame class
class NDS2Frame {
public:
    typedef unsigned long gps_type;
public:
    NDS2Frame(int argc, const char* argv[]);
    ~NDS2Frame(void);
    int main_loop(void);
    void syntax(void) const;
private:
    typedef std::pair<std::string, TSeries*> chan_node;
    typedef std::vector<chan_node> chan_list;
    typedef chan_list::const_iterator const_chan_iter;
    std::string mServer;
    int         mPort;
    gps_type    mStartGPS;
    gps_type    mEndGPS;
    gps_type    mStride;
    std::string mFrameFile;
    int         mDebug;
    bool        mOnline;
    chan_list   mChanList;
    sends::NDS2Socket mNDS;
    FrWriter    mWriter;
};

using namespace std;
using namespace sends;

//========================================  Main function
int 
main(int argc, const char* argv[]) {
    try {
	NDS2Frame nds2fr(argc, argv);
	nds2fr.main_loop();
    } catch (std::exception& e) {
	cerr << "Caught exception: " << e.what() << endl;
    }
}

//========================================  Print command line syntax
void
NDS2Frame::syntax(void) const {
    cout << "NDS2Frame command line syntax:" << endl;
    cout << "NDS2Frame [--server <server-ip>] [--port <port>] [--online] " 
	 << "[--help] " << endl;
    cout << "          [--start-gps <start-gps>] [--end-gps <end-gps>] "
	 << "[--stride <stride>] " << endl;
    cout << "          [--out-frame <file-name>] [--debug <debug-lvl>] "
	 << "[--compress <c-mode>]" << endl;
    cout << "          <channel-1> [... <channel-n>]" << endl;
    cout << endl;
    cout << "<server-ip>  Server address:port [localhost:31200]" << endl;
    cout << "<port>       Server port [0]" << endl;
    cout << "--online     Get online data" << endl;
    cout << "--help       Print this message." << endl;
    cout << "<start-gps>  Start GPS time (ignored in online requests)" 
	 << endl;
    cout << "<end-gps>    End GPS time (number of data seconds online)"
	 << endl;
    cout << "<stride>     Number of seconds per request (frame length)" 
	 << endl;
    cout << "<file-name>  Output file name template [NDS2Frame-%s-%1n.gwf]" 
	 << endl;
    cout << "<debug-lvl>  Debug print level" << endl;
    cout << "<c-mode>     Compression mode: number, \"gzip\", " 
	 << "\"gzip-differential\", " << endl;
    cout << "             or \"gzip-or-zero-suppress\"" << endl;
    cout << "<channel-1> ... <channel-n>  Channels to be written to frame"
	 << endl;
    cout << endl;
}

//========================================  NDS2Frame constructor
NDS2Frame::NDS2Frame(int argc, const char* argv[])
    : mServer("localhost:31200"), mPort(0), mStartGPS(0), mEndGPS(0), 
      mStride(1), mFrameFile("NDS2Frame-%s-%1n.gwf"), mDebug(0), 
      mOnline(false), mWriter("LIGO_NDS", 0)
{

    //------------------------------------  Parse some arguments
    for (int i=1; i<argc; ++i) {
	string argi = argv[i];
	if (argi[0] != '-') {
	    mChanList.push_back(chan_node(argi, new TSeries));
	} else if (argi == "-c" || argi == "--compress") {
	    FrVectRef::compression_mode c_mode = FrVectRef::kUncompressed;
	    string comp = argv[++i];
	    char* ptr = 0;
	    long lval = strtol(comp.c_str(), &ptr, 0);
	    if (*ptr == 0) {
		c_mode = static_cast<FrVectRef::compression_mode>(lval);
	    } else if (comp == "gzip") {
		c_mode = FrVectRef::kGZip;
	    } else if (comp == "gzip-differential") {
		c_mode = FrVectRef::kGZipDifferential;
	    } else if (comp == "gzip-or-zero-suppress") {
		c_mode = FrVectRef::kZeroSuppOrGZip;
	    } else {
		cerr<< "Unrecognized compression mode: " << comp << endl;
		syntax();
	    }
	    mWriter.setCompress(c_mode);
	} else if (argi == "--debug") {
	    mDebug = strtol(argv[++i], 0, 0);
	} else if (argi == "-e" || argi == "--end-gps") {
	    mEndGPS = strtoul(argv[++i], 0, 0);
	} else if (argi == "--help") {
	    syntax();
	    throw runtime_error("Help request");
	} else if (argi == "--online") {
	    mOnline = true;
	} else if (argi == "-o" || argi == "--out-frame") {
	    mFrameFile = argv[++i];
	} else if (argi == "-p" || argi == "--port") {
	    mPort = strtol(argv[++i], 0, 0);
	} else if (argi == "--server") {
	    mServer = argv[++i];
	} else if (argi == "-s" || argi == "--start-gps") {
	    mStartGPS = strtoul(argv[++i], 0, 0);
	} else if (argi == "--stride") {
	    mStride = strtoul(argv[++i], 0, 0);
	} else {
	    syntax();
	    throw runtime_error(string("Unknown command argument: ")+argi);
	}
    }

    //------------------------------------ Make some trivial argument checks
    if (!mOnline) {
	Time now = Now();
	if (mStartGPS > now.getS()) {
	    cerr << "Warning: Specified start time (" << mStartGPS
		 << ") is in the future" << endl;
	    throw runtime_error("Command error");
	}
	if (mEndGPS && mEndGPS <= mStartGPS) {
	    cerr << "Warning: Specified end time (" << mStartGPS
		 << ") is befor ethe specified start" << endl;
	    throw runtime_error("Command error");	    
	}
    }

    //------------------------------------ Open the server client
    string::size_type inx = mServer.find(":");
    if (inx == string::npos) {
	if (!mPort) mPort = 31200;
    } else if (!mPort) {
	mPort = strtol(mServer.c_str() + inx + 1, 0, 0);
	mServer.erase(inx);
    } else {
	mServer.erase(inx);
    }

    mNDS.setDebug(mDebug);
    if (mDebug) cerr << "Connecting socket to server: " << mServer 
		     << ":" << mPort << endl;
    int rc = mNDS.open(mServer, mPort);
    if (rc) {
	throw runtime_error(string("Unable to connect client to: ")+mServer);
    }

    chantype ctype = cUnknown;
    if (mOnline) {
	ctype = cOnline;
	if (mStartGPS) {
	    cerr << "Start time for online data transfers must be 0" << endl;
	    mStartGPS = 0;
	}
	if (mEndGPS == 0) mEndGPS = 86400;
    } else {
	if (!mStartGPS) {
	    throw invalid_argument("Start time is zero or not specified.");
	}
	if (!mEndGPS) mEndGPS = mStartGPS + mStride;
    }

    int N = mChanList.size();
    for (int i=0; i<N; ++i) {
	mNDS.AddChannel(mChanList[i].first, ctype, 0);
	mWriter.addChannel(mChanList[i].first.c_str(), &mChanList[i].second);
    }
    mWriter.setDebug(mDebug);
    
}

//========================================  NDS2Frame constructor
int
NDS2Frame::main_loop(void) {
    int rc;
    union {
	const char*   c;
	const double* d;
	const float*  f;
	const short*  s;
	const int*    i;
    } p;

    int nChan = mChanList.size();

    //------------------------------------  Request online data
    if (mOnline) {
	if (mDebug) cerr << "Request online data" << endl;
	rc = mNDS.RequestOnlineData(mStride);
	if (rc) throw runtime_error("RequestData failed.");
    }

    //------------------------------------  Loop over strides
    for (gps_type t0=mStartGPS; t0<mEndGPS; t0 += mStride) {
	Time t;

	//--------------------------------  Request a bunch of data.
	int rc = 0;
	if (!mOnline) {
	if (mDebug) cerr << "Request offline data" << endl;
	    rc = mNDS.RequestData(t0, mStride);
	    if (rc) throw runtime_error("RequestData failed.");
	}

	//--------------------------------  Get the data
	Interval delta(mStride);
	for (Interval tOff=0; tOff<Interval(mStride); tOff += delta) {
	    if (mDebug) cerr << "Get data... ";
	    rc = mNDS.GetData();
	    if (mDebug) cerr << "rc = " << rc << endl;
	    if (rc <= 0) throw runtime_error("GetData failed.");

	    //----------------------------  Set frame size parameters
	    Time t(Time(mNDS.mRecvBuf.ref_header().GPS));
	    delta = Interval(mNDS.mRecvBuf.ref_header().Secs);

	    //----------------------------  Loop over channels, unpack TSeries
	    for (int i=0; i<nChan; ++i) {
		DAQC_api::const_channel_iter 
		    cp = mNDS.FindChannel(mChanList[i].first);
		Interval dt(1./cp->mRate);
		unsigned long nw = cp->nwords(delta);
		p.c = mNDS.mRecvBuf.ref_data() + cp->mBOffset;
		if (mDebug) cerr << "Get channel: " << cp->mName 
				 << " data-type: "  << cp->mDatatype << endl;
		switch (cp->mDatatype) {
		case _16bit_integer:
		    if (tOff) mChanList[i].second->Append (t, dt, p.s, nw);
		    else      mChanList[i].second->setData(t, dt, p.s, nw);
		    break;
		case _32bit_integer:
		    if (tOff) mChanList[i].second->Append (t, dt, p.i, nw);
		    else      mChanList[i].second->setData(t, dt, p.i, nw);
		    break;
		case _32bit_float:
		    if (tOff) mChanList[i].second->Append (t, dt, p.f, nw);
		    else      mChanList[i].second->setData(t, dt, p.f, nw);
		    break;
		case _64bit_double:
		    if (tOff) mChanList[i].second->Append (t, dt, p.d, nw);
		    else      mChanList[i].second->setData(t, dt, p.d, nw);
		    break;
		default:
		    throw runtime_error("Unsupported data type");
		}
	    }
	}
	mWriter.buildFrame(Time(t0), mStride);
	mWriter.addWriterHistory();

	//------------------------------  Generate the file name
	char file[1024];
	Time tBogus(t0);
	tBogus.setN(mStride);
	TimeStr(tBogus, file, mFrameFile.c_str());

	//------------------------------  Write the frame
	rc = mWriter.open(file);
	if (rc) {
	    cerr << "NDS2Frame unable to open output file: " << file << endl;
	    return rc;
	}
	mWriter.writeFrame();
	mWriter.close();
	cout << "NDS2Frame: Wrote file: " << file << endl;
    }
    return 0;
}

NDS2Frame::~NDS2Frame(void) {
    mWriter.close();
    int N = mChanList.size();
    for (int i=0; i<N; ++i) {
	delete mChanList[i].second;
	mChanList[i].second = 0;
    }
}
