/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef SEGMENT_HH
#define SEGMENT_HH

#include "Interval.hh"
#include "Time.hh"
#include <string>
#include <iostream>

namespace trig {

/**  The trigger base class contains the information needed from all 
  *  triggers. It may be used as a base for a specific trigger object. 
  *  Data contained in the trigger base provide all the information 
  *  needed to select triggers for cross correlation studies, etc. 
  *  A 1kB space is allocated for additional, trigger specific (result) 
  *  data to be provided by the concrete trigger method.
  *  \brief Trigger result base class.
  *  \author J. Zweizig
  *  \version $Id: Segment.hh 6802 2013-04-22 06:21:02Z john.zweizig@LIGO.ORG $
  */
class Segment {
  public:
    /**  Index data type.
      */
    typedef unsigned int index_t;

  public:

    //----------------------------------  Constructors
    /**  Constructor an empty trigger object.
      *  @memo Default constructor.
      */
    Segment(void) {}

    /**  A %Segment object is created.
      *  \brief %Segment constructor.
      *  \param group  %Segment group name string.
      *  \param version %Segment version number.
      *  \param start   %Segment start time.
      *  \param end     %Segment end time.
      */
    Segment(const std::string& group, int version=1, const Time& start=Time(0), 
	    const Time& end=Time(0));

    /**  Release the data after recording the trigger.
      *  @memo Result destructor.
      */
    virtual ~Segment(void) {}

    //----------------------------------  Accessors
    /**  Get the segment activity code. On=1, Off=0.
      *  @memo Get the segment activity code.
      *  @return Activity code.
      */
    int getActivity(void) const;

    /**  Get the segment comment.
      *  @memo Get the comment.
      *  @return segment name string.
      */
    const char* getComment(void) const;

    /**  Get the segment end time.
      *  @memo Get end time.
      *  @return GPS segment end time.
      */
    Time getEndTime(void) const;

    /**  Get the segment name.
      *  @memo Get the segment name.
      *  @return segment name.
      */
    const char* getGroup(void) const;

    /**  Get the trigger ID.
      *  @memo Get the trigger ID.
      *  @return Trigger ID.
      */
    const char* getIfos(void) const;

    /**  Get the minor version number.
      *  @memo Get the mnor version number.
      *  @return Minor version ID.
      */
    int getMinor(void) const;

    /**  Get the segment plus/minus code. Add=1, Remove=0.
      *  @memo Get the segment plus/minus code.
      *  @return Plus/Minus code.
      */
    int getPlusMinus(void) const;

    /**  Get process table key for the generating process.
      *  @memo Process ID.
      *  @return Process table key.
      */
    const char* getProcess(void) const;

    /**  Get the internal segment id.
      *  @memo Segment ID.
      *  @return Segment table key.
      */
    const char* getSegID(void) const;

    /**  Get the segment start time.
      *  @memo Get start time.
      *  @return GPS segment start time.
      */
    Time getStartTime(void) const;

    /**  Get the segment version number.
      *  @memo Segment version number.
      *  @return Segment version number.
      */
    int getVersion(void) const;

    //-----------------------------------  Overloaded operators
    /**  Compare two segments.
      *  @memo Compare operator.
      *  \param seg Segment to be compared.
      *  \return True if the segments are identical.
      */
    bool operator==(const Segment& seg) const;

    //-----------------------------------  Mutators
    /**  Set the activity flag.
      *  @memo Set the activity flag, on=1, off=0, remove=-1.
      *  @param act Activity value.
      */
    void setActivity(int act);

    /**  Set the ending time of this segment.
      *  @memo Set the segment end time.
      *  @param com Segment end Time.
      */
    void setComment(const std::string& com);

    /**  Set the ending time of this segment.
      *  @memo Set the segment end time.
      *  @param t0 Segment end Time.
      */
    void setEndTime(const Time& t0);

    /**  Set the group ID for this segment.
      *  @memo Set the segment group ID.
      *  @param group Segment group ID.
      */
    void setGroup(const char* group);

    /**  Set the interferometer ID(s) for this segment.
      *  @memo Set the segment IFO list.
      *  @param ifos Segment IFO list.
      */
    void setIfos(const char* ifos);

    /**  Set the process ID for this segment.
      *  @memo Set the trigger process ID.
      *  @param Process Unique process ID.
      */
    void setProcess(const char* Process);

    /**  Set the segment ID.
      *  @memo Set the segment ID.
      *  @param SegID Process Unique segment ID.
      */
    void setSegID(const std::string& SegID);

    /**  Set the start time of this segment.
      *  @memo Set the segment start time.
      *  @param t0 Segment start Time.
      */
    void setStartTime(const Time& t0);

    /**  Set the version for this segment.
      *  @memo Set the version.
      *  @param version Segment version number.
      */
    void setVersion(int version);

  private:
    /**  Creator database.
      */
    int mCreatorDB;

    /**  Identifier of the trigger source process.
      */
    std::string mProcess;

    /**  Segment group identifier.
      */
    std::string mGroup;

    /**  Segment group identifier.
      */
    std::string mIfos;

    /**   Segment version number.
      */
    int mVersion;

    /**  Activity flag 0=off, 1=on, -1=remove.
      */
    int mActivity;

    /**  mStartTime contains the segment start time.
      *  @memo Segment start time.
      */
    Time mStartTime;

    /**  mEndTime contains the segment end time.
      *  @memo Segment end time.
      */
    Time mEndTime;

    /**  Additional segment information.
      */
    std::string mComment;

    /**  Unique segment ID
      */
    std::string mSegID;

};   // Segment

}    //  namespace trig

#ifndef __CINT__
//======================================  Inline accessors
inline int 
trig::Segment::getActivity(void) const {
    return mActivity;
}

inline const char*
trig::Segment::getComment(void) const {
    return mComment.c_str();
}

inline Time
trig::Segment::getEndTime(void) const {
    return mEndTime;
}

inline const char*
trig::Segment::getGroup(void) const {
    return mGroup.c_str();
}

inline const char*
trig::Segment::getSegID(void) const {
    return mSegID.c_str();
}

inline const char*
trig::Segment::getIfos(void) const {
    return mIfos.c_str();
}

inline int 
trig::Segment::getPlusMinus(void) const {
    return (mActivity < 0) ? 0 : 1;
}

inline const char*
trig::Segment::getProcess(void) const {
    return mProcess.c_str();
}

inline Time
trig::Segment::getStartTime(void) const {
    return mStartTime;
}

inline int
trig::Segment::getVersion(void) const {
    return mVersion;
}

#endif

#endif   //  SEGMENT_HH
