/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef SBTRIGGER_HH
#define SBTRIGGER_HH
#include "TrigBase.hh"
#include <cmath>

namespace trig {

    /**  The SBTrigger class defines all the sngl_burst trigger fields over 
      *  the gds trigger base class (TrigBase).
      *  \brief Single Burst Trigger class
      *  \author John Zweizig (john.zweizig@ligo.org)
      *  \version $Id$ 
      */
    class SBTrigger : public TrigBase {
    public:
	/**  Construct a  single burst trigger and initialize all data fields 
	  *  to their empty values. For the time being a few data fields 
	  *  \e e.g. the time-span and frequency band are over-specified.
	  *  See the accessor documentation below for a description of the
	  *  actual parameters that are stored.
	  *  \brief Default constructor
	  */
	SBTrigger(void);

	/**  Construct a  single burst trigger and initialize the required 
	  *  data fields.
	  *  \brief Data constructor.
	  *  \param srch   Search name string.
	  *  \param chan   Channel name string.
	  *  \param start  Trigger start time.
	  *  \param dt     Trigger duration. 
	  *  \param freq   Trigger central frequency
	  *  \param bw     Trigger frequency band-width.
	  *  \param sig    Trigger significance.
	  */
	SBTrigger(const std::string& srch, const std::string& chan,
		  const Time& start, Interval dt, double freq, double bw, 
		  double sig);

	/**  Destroy the trigger.
	  * \brief Destructor.
	  */
	virtual ~SBTrigger(void);

	//==============================  Field accessors.

	/**  Burst signal amplitude.
	  *  \brief Amplitude
	  *  \return Signal amplitude.
	  */
	double amplitude(void) const;

	/**  The frequency bandwidth is used with the central frequency to
	  *  define a symmetric band around the specified frequency point.
	  *  The band ranges from \c f_c-0.5*bw to \c f_c+0.5*bw.
	  *  \brief Frequency band width.
	  *  \return Frequency band width.
	  */
	double bandwidth(void) const;

	/**  The central frequency and bandwidth are used to define a 
	  *  symmetric band around the central frequency point.
	  *  The band ranges from \c f_c-0.5*bw to \c f_c+0.5*bw.
	  *  \brief Centeral trigger frequency.
	  *  \return Center of trigger frequency band.
	  */
	double central_freq(void) const;

	/**  Name of the channel in which the trigger was found.
	  *  \brief Channel name string.
	  *  \return Channel name string.
	  */
	std::string channel(void) const;

	/**  Chi-square value for the trigger.
	  *  \note It isn't clear what this is supposed to mean. At present 
	  *  returns zero.
	  *  \brief Chi-square
	  *  \return Chi-square value
	  */
	double chisq(void) const;

	/**  Chi-square per degree of freedom for the trigger
	  *  \note It isn't clear what this is supposed to mean. At present 
	  *  returns zero.
	  *  \brief Chi-square per degree of freedom
	  *  \return Chi-square per degree of freedom
	  */
	double chisq_dof(void) const;

	/**  Confidence value for the trigger
	  *  \brief confidence
	  *  \return confidence
	  */
	double confidence(void) const;

	/**  Trigger duration defined by the interval between the trigger
	  *  start time and the stop time.
	  *  \brief Trigger duration.
	  *  \return Trigger duration.
	  */
	Interval duration(void) const;

	/**  Upper edge of the trigger frequency band, nominally 
	  *  \c center_freq+0.5*bandwidth.
	  *  \brief Trigger upper frequency.
	  *  \return Frequency upper limit.
	  */
	double fhigh(void) const;

	/**  Lower edge of the trigger frequency band, nominally 
	  *  \c center_freq-0.5*bandwidth.
	  *  \brief Trigger lower frequency.
	  *  \return Frequency lower limit.
	  */
	double flow(void) const;

	/**  Trigger rss in calibrated units. At present this is assumed 
	  *  to be equal to the square-root of the signal power in its native 
	  *  units (\e i.e. no further calibration is performed on the channel).
	  *  \brief Calibrated channel energy..
	  *  \return Trigger signal h_rss.
	  */
	double hrss(void) const;

	/**  Ifo(s) in which the trigger was found.
	  *  \brief Ifo name string.
	  *  \return Ifo name string.
	  */
	std::string ifo(void) const;

	/**  Frequency bandwidth of the most significant pixel that is
	  *  contained in this trigger.
	  *  \note This is not yet implemented.
	  *  \brief Most significant pixel band-width.
	  *  \return Bandwidth of the most significant pixel.
	  */
	double ms_bandwidth(void) const;

	/**  Confidence level of the most significant pixel that is
	  *  contained in this trigger.
	  *  \note This is not yet implemented.
	  *  \brief Most significant pixel confidence.
	  *  \return Confidence level of the most significant pixel.
	  */
	double ms_confidence(void) const;

	/**  Duration of the most significant pixel that is contained
	  *  in this trigger.
	  *  \note This is not yet implemented.
	  *  \brief Most significant pixel duration.
	  *  \return Duration of the most significant pixel.
	  */
	Interval ms_duration(void) const;

	/**  Upper frequency limit of the most significant pixel that is
	  *  contained in this trigger.
	  *  \note This is not yet implemented.
	  *  \brief Most significant pixel upper frequency.
	  *  \return Upper frequency of the most significant pixel.
	  */
	double ms_fhigh(void) const;

	/**  Lower frequency limit of the most significant pixel that is
	  *  contained in this trigger.
	  *  \note This is not yet implemented.
	  *  \brief Most significant pixel lower frequency.
	  *  \return Lower frequency of the most significant pixel.
	  */
	double ms_flow(void) const;

	/**  Start time of the most significant pixel that is contained
	  *  in this trigger.
	  *  \note This is not yet implemented.
	  *  \brief Most significant pixel start time.
	  *  \return Start of the most significant pixel.
	  */
	Time ms_start(void) const;

	/**  Stop time of the most significant pixel that is contained
	  *  in this trigger.
	  *  \note This is not yet implemented.
	  *  \brief Most significant pixel stop time.
	  *  \return Stop of the most significant pixel.
	  */
	Time ms_stop(void) const;

	/**  Get name field of ith parameter. 
	  *  \brief Parameter name
	  *  \param i Parameter number
	  *  \return Parameter name string.
	  */
	std::string param_name(int i) const;

	/**  Get value field of ith parameter. 
	  *  \brief Parameter value
	  *  \param i Parameter number
	  *  \return Parameter value.
	  */
	double param_value(int i) const;

	/**  Signal frequency at peak intensity.
	  *  \brief Signal peak frequency
	  *  \return Signal frequency at peak intensity.
	  */
	double peak_frequency(void) const;

	/**  Signal frequency width at peak intensity.
	  *  \brief Signal peak frequency width
	  *  \return Signal frequency width at peak intensity.
	  */
	double peak_frequency_error(void) const;

	/**  Calibrated signal amplitude at peak intensity.
	  *  \brief Signal peak calibrated amplitude.
	  *  \return Signal calibrated amlitude at peak intensity.
	  */
	double peak_strain(void) const;

	/**  Signal amplitude width at peak intensity.
	  *  \brief Signal peak amplitude width
	  *  \return Signal amplitude width at peak intensity.
	  */
	double peak_strain_error(void) const;

	/**  %Time of maximum signal amplitude.
	  *  \brief Peak time
	  *  \return %Time of signal peak.
	  */
	Time peak_time(void) const;

	/**  Error in maximum signal amplitude time.
	  *  \brief Error on peak time
	  *  \return Error on time of signal peak.
	  */
	Interval peak_time_error(void) const;

	/**  Search name string.
	  *  \brief search name string.
	  *  \return Search name string.
	  */
	std::string search(void) const;

	/** Ratio of signal to noise amplitudes.
	  * \brief signal to noise ratio.
	  * \return signal to noise ratio.
	  */
	double snr(void) const;

	/**  %Time of start of earliest triggered pixel.
	  *  \brief Trigger start time.
	  *  \return Start time.
	  */
	Time start_time(void) const;

	/**  %Time of stop of latest triggered pixel. The stio time is equal
	  *  to the start time plus the duration.
	  *  \brief Trigger stop time.
	  *  \return Stop time.
	  */
	Time stop_time(void) const;

	/**  The sum of the area of all time-frequency pixels in this trigger.
	  *  \brief Time-frequency volume.
	  *  \return Time-frequency volume.
	  */
	double tfvolume(void) const;

	/**  Lag time used in accidental coincidence test.
	  *  \brief Lag time.
	  *  \return Lag interval.
	  */
	Interval time_lag(void) const;

	//==============================  SBTrigger mutators.
	/**  Set the trigger signal amplitude.
	  *  \brief Set the amplitude.
	  *  \param a Amplitude
	  */
	void amplitude(double a);

	/**  Set the trigger signal bandwidth.
	  *  \brief Set the bandwidth.
	  *  \param b bandwidth
	  */
	void bandwidth(double b);

	/**  Set the trigger signal central frequency.
	  *  \brief Set the central frequency.
	  *  \param f Central frequency
	  */
	void central_freq(double f);

	/**  Set the name of the channel for which the trigger was found.
	  *  \brief Set channel name.
	  *  \param c Channel name string
	  */
	void channel(const std::string& c);

	/**  Set the trigger chi-squared.
	  *  \brief Set the chi-squared.
	  *  \param x Chi-squared
	  */
	void chisq(double x);

	/**  Set the trigger chi-square per degree of freedom.
	  *  \brief Set the chi-square per degree of freedom.
	  *  \param x chi-square perdegree of freedom.
	  */
	void chisq_dof(double x);

	/**  Set the trigger confidence elvel.
	  *  \brief Set the confidence level.
	  *  \param a Confidence level.
	  */
	void confidence(double a);

	/**  Set the trigger signal duration. The duration is the interval
	  *  between the trigger start and stop times. 
	  *  \brief Set the duration.
	  *  \param dt Duration.
	  */
	void duration(Interval dt);

	/**  Set the trigger signal hrss.
	  *  \brief Set the hrss
	  *  \param h hrss.
	  */
	void hrss(double h);

	/**  Set the trigger interferometer name string.
	  *  \brief Set the interferometer name.
	  *  \param i Interferometer name string.
	  */
	void ifo(const std::string& i);

	/**  Set the most significant pixel bandwidth.
	  *  \brief Set the most significant pixel bandwidth.
	  *  \param bw Bandwidth (in Hz).
	  */
	void ms_bandwidth(double bw);

	/**  Set the most significant pixel confidence.
	  *  \brief Set m.s. pixel confidence 
	  *  \param c Confidence.
	  */
	void ms_confidence(double c);

	/**  Set the most significant pixel duration.
	  *  \brief Set m.s. pixel duration.
	  *  \param dt Duration.
	  */
	void ms_duration(Interval dt);

	/**  Set the most significant pixel high frequency.
	  *  \brief Set m.s. pixel high frequency.
	  *  \param f High frequency.
	  */
	void ms_fhigh(double f);

	/**  Set the most significant pixel low frequency.
	  *  \brief Set m.s. pixel low frequency.
	  *  \param f Low frequency.
	  */
	void ms_flow(double f);

	/**  Set the most significant pixel start time.
	  *  \brief Set m.s. pixel start time.
	  *  \param t Start time.
	  */
	void ms_start(const Time& t);

	/**  Set the most significant pixel stop time.
	  *  \brief Set m.s. pixel stop time.
	  *  \param t Stop time.
	  */
	void ms_stop(const Time& t);

	/**  Set the trigger peak frequency.
	  *  \brief Set the peak frequency.
	  *  \param f Peak frequency
	  */
	void peak_frequency(double f);

	/**  Set the trigger signal peak frequency error.
	  *  \brief Set the peak frequency error.
	  *  \param df Peakfrequency error.
	  */
	void peak_frequency_error(double df);

	/**  Set the trigger signal peak strain.
	  *  \brief Set the peak strain..
	  *  \param h Peak strain
	  */
	void peak_strain(double h);

	/**  Set the trigger signal peak strain error.
	  *  \brief Set the peak strain error.
	  *  \param dh Peak strain error
	  */
	void peak_strain_error(double dh);

	/**  Set the trigger signal peak time.
	  *  \brief Set the peak time.
	  *  \param t peak time.
	  */
	void peak_time(const Time& t);

	/**  Set the trigger signal peak time error.
	  *  \brief Set the peak time error.
	  *  \param dt Peak time error.
	  */
	void peak_time_error(Interval dt);

	/**  Set the search name strin.
	  *  \brief Set the search.
	  *  \param s Search name string.
	  */
	void search(const std::string& s);

	/**  Set the trigger signal to noise ratio.
	  *  \brief Set the SNR
	  *  \param a Signa to noise ratio.
	  */
	void snr(double a);

	/**  Set the trigger start time.
	  *  \brief Set the start time.
	  *  \param t Start time.
	  */
	void start_time(const Time& t);

	/**  Set the trigger time-frequency volume.
	  *  \brief Set the time-frequency volume.
	  *  \param a %Time-frequency volume.
	  */
	void tfvolume(double a);

	/**  Set the time lag use in accidental oicidence searchs.
	  *  \brief Set the time lag.
	  *  \param tl Time lag.
	  */
	void time_lag(Interval tl);

	/**  Set the frequency band limits.
	  *  \brief Set frequency band.
	  *  \param fhi Low frequency limit.
	  *  \param flo High frequency limit.
	  */
	void set_frequency_band(double flo, double fhi);

	/**  Set the numbered parameter name and value.
	  *  \brief Record a parameter value.
	  *  \param     i Parameter number.
	  *  \param  name Parameter name.
	  *  \param value Parameter value.
	  */
	void set_param(int i, const std::string& name, double value);

	/**  Set the burst time and duration from the start and end times.
	  *  \brief Set start/end times.
	  *  \param start Start time.
	  *  \param stop  End time.
	  */
	void set_time(const Time& start, const Time& stop);

	/**  Set the most significant pixel time and duration from the start
	  *  and end times.
	  *  \brief Set most significant pixel start/end times.
	  *  \param start Start time.
	  *  \param stop  End time.
	  */
	void set_ms_time(const Time& start, const Time& stop);

    private:
	struct sb_param {
	    double _value;
	    char   _name[64];
	};
    };

    //==================================  Inline methods
    inline double
    SBTrigger::amplitude(void) const {
	return getIntensity();
    }

    inline double
    SBTrigger::bandwidth(void) const {
	return getBandwidth();
    }

    inline double
    SBTrigger::central_freq(void) const {
	return getFrequency();
    }

    inline std::string
    SBTrigger::channel(void) const {
	return getSubID();
    }

    inline double
    SBTrigger::confidence(void) const {
	return getConfidence();
    }

    inline Interval
    SBTrigger::duration(void) const {
	return getDt();
    }

    inline double
    SBTrigger::fhigh(void) const {
	return getHighF();
    }

    inline double
    SBTrigger::flow(void) const {
	return getLowF();
    }

    inline double
    SBTrigger::hrss(void) const {
	return sqrt(getSignalPower());
    }

    inline std::string
    SBTrigger::ifo(void) const {
	return std::string(getIFOs());
    }

    inline double 
    SBTrigger::peak_frequency(void) const {
	return getFreqPeak();
    }

    inline Time 
    SBTrigger::peak_time(void) const {
	return getPeakTime();
    }

    inline std::string
    SBTrigger::search(void) const {
	return getID();
    }

    inline Time
    SBTrigger::ms_stop(void) const {
	return ms_start() + ms_duration();
    }

    inline Time
    SBTrigger::start_time(void) const {
	return getTime();
    }

    inline Time
    SBTrigger::stop_time(void) const {
	return start_time() + duration();
    }

}

#endif //  !defined(SBTRIGGER_HH)
