/* -*-  mode: c++; c-basic-offset: 3; -*- */
#ifndef LOCKSEGMENT_HH
#define LOCKSEGMENT_HH

#include "Time.hh"
#include "Interval.hh"
#include <iosfwd>
#include <string>

#define SEGMENT_TINF (0x7fffffff)

/**  LockSegment represents a single segment for list manipulation. The 
  *  segment contains a start time, duration, ID and type flag-word.
  *  @memo Segment manipulation class.
  *  @author J.Zweizig
  *  @version 1.2; last modified 10/05/2006 
  */
class LockSegment {
public:
   /** Define type of data to represent flag.
     */
   typedef unsigned long flag_type;

   /** Define type of data to represent ID.
     */
   typedef long          id_type;
public:
   /**  LockSegment data constructor.
     *  @memo Segment constructor.
     *  @param   id Segment identifier.
     *  @param   t0 Segment start time.
     *  @param   dT Segment duaration.
     *  @param  flg Segment flags.
     */
   LockSegment(id_type id, const Time& t0, const Interval dT, flag_type flg=0);

   /**  LockSegment data constructor.
     *  @memo Segment constructor.
     *  @param id   Segment identifier.
     *  @param t0   Segment start time.
     *  @param tEnd Segment end time.
     *  @param flg  Segment flags.
     */
   LockSegment(id_type id, const Time& t0, const Time& tEnd, flag_type flg=0);

   /**  Get the segment duration.
     *  \return Segment duration.
     */
   Interval getDuration(void) const;

   /**  Get the segment end time.
     *  \return End time.
     */
   Time getEndTime(void) const;

   /**  Get the segment flag field.
     *  \return Flag field contents.
     */
   flag_type getFlags(void) const;

   /**  Get the segment ID.
     *  \return segment ID
     */
   id_type getID(void) const;

   /**  Get (a reference to)the start time.
     *  \return Reference to start time
    */
   const Time& getStartTime(void) const;

   /**  Test whether the specified time is in this segment.A time is
     *  outside the segment if it is equal to the end time.
     *  @memo Test if a time is inside the segment.
     *  @param t Test time.
     *  @return true if time is in segment.
     */
   bool inSegment(const Time& t) const;

  /**  Test whether a specified time interval is entirely inside the 
    *  segment. The segment [t1:t2] is inside the segment only if t1 < t2,
    *  and the start time is inside the segment and the end time is not
    *  not after the end of this segment.
    *  @memo Test whether a time interval is inside the segment.
    *  @param t1 Start of test segment.
    *  @param t2 End of test segment.
    *  @return True if test segment is valid and is entirely inside segment.
    */
   bool inSegment(const Time& t1, const Time& t2) const;

  /**  Test whether start of segment is earlier than start time of the 
    *  argument segment.
    *  @memo Test if segment is earlier than argument.
    *  \param s Argument segment
    *  \return True If segment starts earlier than argument.
    */
   bool operator< (const LockSegment& s) const;

   /**  Test whether the specified time is before this segment. The argument 
     *  time is considered to be before the segment if it is earlier than the
     *  start time of this segment. 
     *  \param t Argument time
     *  \return True If segment starts earlier than argument.
     */
   bool operator< (const Time& t) const;

   /**  And the current segment with the argument. This segment is replaced 
     *  by a segment representing its overlap with the argument. The ID is 
     *  left unchanged and the flags are 'and'ed.
     *  \param s Argument segment
     *  \return Reference to this instance
     */
   LockSegment& operator&= (const LockSegment& s);

   /**  Or the current segment with the argument. This segment is replaced 
     *  by a segment representing its union with the argument. The ID ls 
     *  left unchanged and the flags are 'or'ed.
     *  \param s Argument segment
     *  \return Reference to this instance
     */
   LockSegment& operator|= (const LockSegment& s);

   /**  Test whether the argument time span overlaps this segment at any 
     *  time.
     *  \param t  Argument segment start time
     *  \param t2 Argument segment end time
     *  \return True if the argument segment overlaps this segment.
     */
   bool overlap(const Time& t, const Time& t2) const;

   /**  Test whether the argument segment overlaps this segment at any 
     *  time.
     *  \param s argument segment
     *  \return True if the argument segment overlaps this segment.
     */
   bool overlap(const LockSegment& s) const;

   /**  Pad the front and back of the segment by pFront and pBack. The 
     *  pFront and pBack intervals are defined such that positive intervals 
     *  increase the segment duration and negative intervals decrease the
     *  resulting segment duration. \e i.e. the resulting segment is 
     *  \verbatim [start - pStart, end + pBack] \endverbatim . The result of 
     *  pad is limited so that the segment can't extend outside of the 
     *  interval \verbatim [0, SEGMENT_TINF] \endverbatim .
     *  @memo Extend the segment start and/or end times.
     *  @param pFront Length of padding to add to the start of the segment.
     *  @param pBack  Length of padding to add to the back of the segment.
     */
   void pad(Interval pFront=0, Interval pBack=0);

   /** Modify the segment time range from a start time and duration.
    *  \param t0   Start time
    *  \param dT   Segment duration.
     */
   void setRange(const Time& t0, Interval dT);

   /** Set the segment time range from a start time and end time.
    *  \param t0   Start time
    *  \param tEnd End time
    */
   void setRange(const Time& t0, const Time& tEnd);

   /**  Test whether the argument segment overlaps or is adjacent to 
     *  this segment.
     *  \param s Argument segment
     *  \return True if segments are adjacent or overlap.
     */
   bool touch(const LockSegment& s) const;

   /**  Write this segment with the appropriate list.
     *  \brief Write segment.
     *  \param s Output stream.
     *  \param f Format string.
     *  \return Reference to output stream.
     */
   std::ostream& write(std::ostream& s, const std::string& f) const;

private:
   id_type   mID;
   Time      mStart;
   Interval  mDuration;
   flag_type mFlags;
};

//======================================  non-member functions
std::ostream&
operator<<(std::ostream& o, const LockSegment& s);

//======================================  inline methods
inline Interval
LockSegment::getDuration(void) const {
   return mDuration;
}

inline Time 
LockSegment::getEndTime(void) const {
   return mStart+mDuration;
}

inline LockSegment::flag_type
LockSegment::getFlags(void) const {
   return mFlags;
}

inline LockSegment::id_type
LockSegment::getID(void) const {
   return mID;
}

inline const Time& 
LockSegment::getStartTime(void) const {
   return mStart;
}

inline bool 
LockSegment::inSegment(const Time& t) const {
   return ((t >= mStart) && (t < getEndTime()));
}

inline bool 
LockSegment::inSegment(const Time& t1, const Time& t2) const {
   return ((t1 < t2) && (t1 >= mStart) && (t2 <= getEndTime()));
}

inline bool 
LockSegment::operator< (const LockSegment& s) const {
   return mStart < s.mStart;
}

inline bool 
LockSegment::overlap(const Time& t1, const Time& t2) const {
   return ((t1 < t2) && (t2 > mStart) && (t1 < getEndTime()));
}

inline bool 
LockSegment::overlap(const LockSegment& s) const {
   return overlap(s.mStart, s.getEndTime());
}

inline void
LockSegment::setRange(const Time& t0, const Time& tEnd) {
   setRange(t0, tEnd-t0);
}

inline bool 
LockSegment::touch(const LockSegment& s) const {
   return mStart <= s.getEndTime() && s.mStart <= getEndTime();
}

#endif  // !defined(LOCKSEGMENT_HH)
