#ifndef _LIGO_EVENTLIST_H
#define _LIGO_EVENTLIST_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: List							*/
/*                                                         		*/
/* Module Description: Defines a list of events				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: List.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <string>
#include <vector>
#include "events/Event.hh"
#include "events/Iterator.hh"
#include "events/Function.hh"
#include "events/FunctionPtr.hh"


namespace events {


/** An event list contains a series of events. An event list is
    sorted in time. An event list has the following four states:
    (1) Empty: An empty list contains nothing and has not been 
    loaded from file; (2) Loaded: A loaded event list has been
    read from file and is still in its original state (meaning
    it can be unloaded again); (3) Locked: A locked event list
    is loaded but can not be discared; and (4) Modified: The event 
    list was modified by the user (meaning it can not be unloaded).

    An event list contains events ordered by time with the oldest 
    event being stored first. For efficiency reasons List provides
    algorithms which let you add events without checking that 
    they are in correct orderer. In this case the user has to 
    guarantee that the sort order is maintained, or call the Sort
    method afterwards.
   
    @memo List of events
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class List {
      friend class Chain;
   
   public:
      /// State of the event list
      enum state {
      /// Empty
      kEmpty = 0,
      /// Loaded into memory
      kLoaded = 1,
      /// Loaded into memory and locked
      kLocked = 1,
      /// In memory and modified
      kModified = 2
      };
      /// Event list basic type
      typedef std::vector<Event> eventlist;
      /// Value type
      typedef eventlist::value_type value_type;
      /// Size type
      typedef eventlist::size_type size_type;
      /// Difference type 
      typedef eventlist::difference_type difference_type;
      /// Reference type
      typedef eventlist::reference reference;
      /// Const reference type
      typedef eventlist::const_reference const_reference;
      /// Pointer type
      typedef eventlist::pointer pointer;
      /// Const pointer type
      typedef eventlist::const_pointer const_pointer;      
      /// Iterator
      typedef Iterator iterator;
      /// Const iterator
      typedef ConstIterator const_iterator;
      /// Iterator
      typedef ReverseIterator reverse_iterator;
      /// Const iterator
      typedef ConstReverseIterator const_reverse_iterator;
   
      /** Creates an empty eventlist.
          @memo Default constructor
       ******************************************************************/
      List() : mState (kEmpty) {
      }
      /** Reads an event list from file.
          @memo Constructor
          @param filename Name of file
          @param keepforever Load and lock in memory if true
       ******************************************************************/
      explicit List (const char* filename, bool keepforever = true)
      : mState (kEmpty) {
         Load (filename, keepforever); }
      /** Returns a copy of the event list. This method must be overriden 
          by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      List* Copy() const {
         return new List (*this); }
   
      /** Size of list.
          @memo Size
       ******************************************************************/
      int Size() const {
         return mList.size(); }
      /** Empty list?
          @memo Empty
       ******************************************************************/
      bool Empty() const {
         return mList.empty(); }
      /** Maximum size of list.
          @memo Maximum Size
       ******************************************************************/
      int MaxSize() const {
         return mList.max_size(); }
      /** Capacity of list.
          @memo Capacity
       ******************************************************************/
      int Capacity() const {
         return mList.capacity(); }
      /** Capacity of list.
          @memo Capacity
       ******************************************************************/
      void Reserve (size_type num) {
         mList.reserve (num); }
      /** Equality operator.
          @memo Equality
       ******************************************************************/
      bool operator== (const List& l) const {
         return mList == l.mList; }
      /** Inequality operator.
          @memo Equality
       ******************************************************************/
      bool operator!= (const List& l) const {
         return mList != l.mList; }
      /** Check if events are stored in proper time order.
          @memo Check order
       ******************************************************************/
      bool CheckOrder() const;
      /** Swap the list.
          @memo Swap
       ******************************************************************/
      void Swap (List& l);
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      iterator LowerBound (const Time& t) {
         return LowerBound (Event (t)); }
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      const_iterator LowerBound (const Time& t) const {
         return LowerBound (Event (t)); }
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      iterator UpperBound (const Time& t) {
         return UpperBound (Event (t)); }
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      const_iterator UpperBound (const Time& t) const {
         return UpperBound (Event (t)); }
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      iterator LowerBound (const Event& t);
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      const_iterator LowerBound (const Event& t) const;
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      iterator UpperBound (const Event& t);
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      const_iterator UpperBound (const Event& t) const;
      /** Sort the list by event time.
          @memo Sort
       ******************************************************************/
      void Sort ();
      /** Sort the list by the specified function in ascending or 
          descending order.
          @memo Sort
       ******************************************************************/
      void Sort (const Function& func, bool ascending = true);
      void Sort (const FunctionPtr& func, bool ascending = true) {
         Sort (*func, ascending); }
   
      /** At.
          @memo At
       ******************************************************************/
      reference At (size_type idx) {
         return mList[idx]; }
      /** At.
          @memo At
       ******************************************************************/
      const_reference At (size_type idx) const {
         return mList[idx]; }
      /** Operator [].
          @memo Operator[]
       ******************************************************************/
      reference operator[] (size_type idx) {
         return mList[idx]; }
      /** Operator [].
          @memo Operator[]
       ******************************************************************/
      const_reference operator[] (size_type idx) const {
         return mList[idx]; }
      /** Front.
          @memo Front
       ******************************************************************/
      reference Front() {
         return mList.front(); }
      /** Front.
          @memo Front
       ******************************************************************/
      const_reference Front() const {
         return mList.front(); }
      /** Back.
          @memo Back
       ******************************************************************/
      reference Back() {
         return mList.back(); }
      /** Back.
          @memo Back
       ******************************************************************/
      const_reference Back() const {
         return mList.back(); }
      /** Begin.
          @memo Begin
       ******************************************************************/
      iterator Begin();
      /** Begin.
          @memo Begin
       ******************************************************************/
      const_iterator Begin() const;
      /** End.
          @memo End
       ******************************************************************/
      iterator End();
      /** End.
          @memo End
       ******************************************************************/
      const_iterator End() const;
      /** RBegin.
          @memo RBegin
       ******************************************************************/
      reverse_iterator RBegin () {
         return reverse_iterator (End()); }
      /** RBegin.
          @memo RBegin
       ******************************************************************/
      const_reverse_iterator RBegin () const {
         return const_reverse_iterator (End()); }
      /** REnd.
          @memo REnd
       ******************************************************************/
      reverse_iterator REnd () {
         return reverse_iterator (Begin()); }
      /** REnd.
          @memo REnd
       ******************************************************************/
      const_reverse_iterator REnd () const {
         return const_reverse_iterator (Begin()); }
   
      /** Insert an event to the list at the proper location.
          @memo Add
       ******************************************************************/
      iterator Insert (const Event& event);
      /** Insert an event to the list at the specified location.
          @memo Add
       ******************************************************************/
      iterator Insert (const iterator& pos, const Event& event);
      /** Inserts a range of events.
          @memo Add
       ******************************************************************/
      void Insert (const iterator& beg, const iterator& end);
      /** Inserts an event at the back.
          @memo PushBack
       ******************************************************************/
      void PushBack (const Event& event) {
         mList.push_back (event); }
      /** Erase an event.
          @memo Erase
       ******************************************************************/
      iterator Erase (const iterator& pos);
      /** Erase a rane of events.
          @memo Erase
       ******************************************************************/
      iterator Erase (const iterator& beg, const iterator& end);
      /** Removes an event from the back.
          @memo PopBack
       ******************************************************************/
      void PopBack () {
         mList.pop_back(); }
      /** Clear the list.
          @memo Clear
       ******************************************************************/
      void Clear() {
         mList.clear(); }
   
      /** Get the state.
          @memo Get the state
       ******************************************************************/
      state GetState() const;
      /** Load the list from file.
          @memo Load
          @param keepforever Load and lock in memory if true
       ******************************************************************/
      bool Load (bool keepforever = true);
      /** Load the list from specified file.
          @memo Load
          @param filename Name of file
          @param keepforever Load and lock in memory if true
       ******************************************************************/
      bool Load (const char* filename, bool keepforever = true) {
         SetFilename (filename); 
         return Load (keepforever); }
      /** Unload the event list if possible.
          @memo Unload
          @return true if able to unload
       ******************************************************************/
      bool Unload();
      /** Save the list to the specified file.
          @memo Save
          @param filename Name of file
       ******************************************************************/
      bool Save (const char* filename) const;
   
      /** Set the filename.
          @memo Set the filename
       ******************************************************************/
      void SetFilename (const char* filename) {
         if (!Unload()) Clear();
         mFilename = filename ? filename : ""; }
      /** Get the filename.
          @memo Get the filename
       ******************************************************************/
      const char* GetFilename() const {
         return mFilename.c_str(); }
   
   private:
      /// Event list
      eventlist		mList;
      /// Event list state
      state		mState;
      /// Filename
      std::string 	mFilename;
   };


}

#endif // _LIGO_EVENTLIST_H
