//////////////////////////////////////////////////////////////////////////
//  									//
//  LayoutInfo								//
//  									//
//////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include "events/LayoutInfo.hh"
#include "events/FixedColumns.hh"
#include "events/Factory.hh"
#include "events/Algorithm.hh"
#include "events/Event.hh"

   using namespace std;

namespace events {

//______________________________________________________________________________
   LayoutInfo::LayoutInfo()
   : mRegistered (false), mDataSize (0), mRefCount (0)
   {
      for (ColumnList::const_iterator i = FixedColumns::List().begin();
          i != FixedColumns::List().end(); ++i) {
         mColumns.Add (*i);
      }
      Recalculate();
   }

//______________________________________________________________________________
   LayoutInfo::LayoutInfo (const Type& type)
   : mRegistered (false),  mType (type), mDataSize (0), mRefCount (0)
   {
      for (ColumnList::const_iterator i = FixedColumns::List().begin();
          i != FixedColumns::List().end(); ++i) {
         mColumns.Add (*i);
      }
      Recalculate();
   }

//______________________________________________________________________________
   bool LayoutInfo::operator== (const LayoutInfo& info) const
   {
      if (mColumns.Size() != info.mColumns.Size()) {
         return false;
      }
      ColumnList::const_iterator i = mColumns.begin();
      ColumnList::const_iterator j = info.mColumns.begin();
      for (; i !=  mColumns.end(); ++i, ++j) {
         if (*i != *j) {
            return false;
         }
      }
      return true;
   }

//______________________________________________________________________________
   bool LayoutInfo::GetValue (const char* name, const_data_ptr data, 
                     Value& val) const
   {
      std::string nameFirst, nameRem;
      if ( !Parse (name, nameFirst, nameRem) ) {
         val = Value();
         return false;
      }
      const ColumnInfo* col = GetColumn (nameFirst.c_str());
      if (!col) {
         val = Value();
         return false;
      }
      if (col->GetColumn() < 0) {
         val = Value();
         return false;
      }
   
      int colnum = ((Common_t*)data)->mColumns;
      if (col->GetColumn() < colnum) {
         if ( !nameRem.empty() && col->GetType() == ColumnType::kEvent ) {
            ((Event*)Address (data, col->GetOffset()))->GetValue(nameRem.c_str(),val);
         }
         else if (nameRem.empty()) {
            val.Read (col->GetType(), Address (data, col->GetOffset()));
         }
         else {
            val = Value();
            return false;
         }
      }
      else {
         val.Read (col->GetType(), 0);
      }
   
      return true;
   }

//______________________________________________________________________________
   bool LayoutInfo::SetValue (const char* name, data_ptr& data, 
                     const Value& val)
   {
      std::string nameFirst, nameRem;
      if ( !Parse (name, nameFirst, nameRem) ) {
         return false;
      }
   
      const ColumnInfo* col = GetColumn (nameFirst.c_str());
      if (!col) {
         return false;
      }
      if (col->GetColumn() < 0) {
         return false;
      }
      // if not enough columns in event; make room first
      int colnum = ((Common_t*)data)->mColumns;
   
      if (col->GetColumn() < colnum) {
         if ( !nameRem.empty() && col->GetType() == ColumnType::kEvent ) {
            ((Event*)Address (data, col->GetOffset()))->SetValue(nameRem.c_str(),val);
         }
         else if (nameRem.empty()) {
            val.Write (col->GetType(), Address (data, col->GetOffset()));
         }
         else {
            return false;
         }
      }
      else {
         // val.Write (col->GetType(), 0);
         if (!Update (data)) 
            return false;
         val.Write (col->GetType(), Address (data, col->GetOffset()));
      }
   
      // if ((col->GetColumn() >= colnum) && !Update (data)) {
         // return false;
      // }
      // return val.Write (col->GetType(), Address (data, col->GetOffset()));
      return true;
   }

//______________________________________________________________________________
   const ColumnInfoList& LayoutInfo::GetColumnList() const
   {
      return mColumns.List();
   }

//______________________________________________________________________________
   bool LayoutInfo::AddColumn (const ColumnInfo& col)
   {
      // Cannot add a fixed or invalid column
      if (col.IsFixed() || (col.GetType() == kInvalid)) {
         return false;
      }
      // If registerd, add at the end
      if (IsRegistered()) {
         // Cannot add a column twice
         ColumnList::iterator i = mColumns.GetPos (col.GetName());
         if (i != mColumns.end()) {
            return false;
         }
         mColumns.Add (col);
         Factory::Get().IncreaseLayoutAddColVers();
      }
      // else maintain the ColumnInfo order
      else {
         mColumns.Insert (col);
      }
      Recalculate();
      return true;
   }

//______________________________________________________________________________
   bool LayoutInfo::RemoveColumn (const char* name)
   {
      // If registerd, do not remove
      if (!name || IsRegistered()) {
         return false;
      }
      // eliminate space
      std::string n = name;
      std::string::size_type pos;
      while ((pos = n.find_first_of (" \t\f\n\r\v")) != std::string::npos) {
         n.erase (pos, 1);
      }
      // Cannot remove a fixed column
      ColumnList::iterator i = mColumns.GetPos (n.c_str());
      if ((i == mColumns.end()) || (i->IsFixed())) {
         return false;
      }
      // Remove it
      mColumns.Remove (i);
      Recalculate();
      return true;
   }

//______________________________________________________________________________
   const ColumnInfo* LayoutInfo::GetColumn (const char* name) const
   {
      // eliminate space
      std::string n = name;
      std::string::size_type pos;
      while ((pos = n.find_first_of (" \t\f\n\r\v")) != std::string::npos) {
         n.erase (pos, 1);
      }
      return mColumns.Get (n.c_str());
   }

//______________________________________________________________________________
   void LayoutInfo::Recalculate()
   {
      // Recalculate offset, column number and total size
      int offset = FixedColumns::VaryingOffset();
      int column = FixedColumns::VaryingNumber();
      for (ColumnList::iterator i = mColumns.begin(); 
          i != mColumns.end(); ++i) {
         if (!i->IsFixed()) {
            // make sure alignment is correct
            offset = ((offset + i->GetTypeAlignment()) - 1) / 
               i->GetTypeAlignment() * i->GetTypeAlignment();
            i->SetOffset (offset);
            offset += i->GetTypeSize();
            i->SetColumn (column++);
         }
      }
      mDataSize = offset;
   }


//______________________________________________________________________________
   bool LayoutInfo::Construct (data_ptr data, const_data_ptr init)
   {
      // Only if registerd
      if (!IsRegistered()) {
         return false;
      }
      int colnum = init ? ((Common_t*)init)->mColumns : 0;
      for (ColumnInfoList::iterator i = mColumns.begin();
          i != mColumns.end(); ++i) {
         if (colnum) {
            i->Construct (data, init);
            --colnum;
         }
         else {
            i->Construct (data);
         }
      }
      ((Common_t*)data)->mColumns = mColumns.Size();
      return true;
   }

//______________________________________________________________________________
   bool LayoutInfo::Destruct (data_ptr data)
   {
      // No data?
      if (!data) {
         return false;
      }
      // Call column destructors
      int colnum = ((Common_t*)data)->mColumns;   
      for (ColumnInfoList::iterator i = mColumns.begin();
          i != mColumns.end(); ++i) {
         if (colnum) {
            i->Destruct (data);
            --colnum;
         }
         else {
            break;
         }
      }
      return true;
   }

//______________________________________________________________________________
   bool LayoutInfo::Update (data_ptr& data)
   {
      // for a registered layout and an event which has fewer columns than
      // indicated in the column list of the layout, resize the data block
      if (IsRegistered() && (((Common_t*)data)->mColumns < mColumns.Size())) {
         data_ptr newdata = new (std::nothrow) char [DataSize()];
         if (!Construct (newdata, data) || !Destruct (data)) {
            delete [] (char*) newdata;
            return false;
         }
         delete [] (char*) data;
         data = newdata;
      }
      return true;
   }

//______________________________________________________________________________
   bool LayoutInfo::Compare (const_data_ptr d1, const_data_ptr d2) const
   {
      // Only if registerd
      if (!IsRegistered()) {
         return false;
      }
      int coln1 = ((Common_t*)d1)->mColumns;
      int coln2 = ((Common_t*)d2)->mColumns;
      int colmin = (coln1 < coln2) ? coln1 : coln2;
      int colmax = (coln1 > coln2) ? coln1 : coln2;
      if (colmax <= 1) {
         return true;
      }
      // Check common length of data array
      for (int i = 0; i < colmin; ++i) {
         if (!mColumns[i].Compare (d1, d2)) {
            return false;
         }
      }
      // if one array is larger than the other, 
      // all surplus elements have to be zero
      if (colmax > colmin) {
         const_data_ptr d = (coln1 > coln2) ? d1 : d2;
         for (int i = colmin; i < colmax; ++i) {
            if (!mColumns[i].IsZero (d)) {
               return false;
            }
         }
      }
      return true;
   }

//______________________________________________________________________________
   const LayoutInfo* LayoutInfo::Register()
   {
      // Can not register twice
      if (IsRegistered()) {
         return 0;
      }
      // Must have a valid type
      if (mType.GetId() == 0) {
         return 0;
      }
      return Factory::Get().RegisterLayout (*this);
   }

//______________________________________________________________________________
   void LayoutInfo::Dump (std::ostream& os) const
   {
      os << "Type = " << mType.GetName() << endl;
   
      for (ColumnInfoList::const_iterator i = mColumns.begin();
          i != mColumns.end(); ++i) {
      
         std::string colname = i->GetName();
         int offset = i->GetOffset();
         std::string vtype = i->GetTypeName();
      
         std::string ctype;
         if (i->IsFixed()) {
            ctype = "Fixed";
         }
         else {
            ctype = "Variable";
         }
      
         os << "\t"
            << colname << ": [" 
            << ctype << "," 
            << vtype << "," 
            << offset << "]" 
            << endl;
      }
   
   }

//______________________________________________________________________________
   bool LayoutInfo::Parse (const char* nameOrg,
                     std::string& nameFirst, std::string& nameRem) {
      std::string name = nameOrg;
      std::string rem = "";
      std::string::size_type pos;
      while ((pos = name.find_first_of (" \t\f\n\r\v")) != 
            std::string::npos) {
         name.erase (pos, 1);
      }
   
      // find first dot
      if ((pos = name.find ('.')) != std::string::npos) {
         rem = name.substr (pos + 1);
         name.erase (pos);
         // check if predot string contains an array index
         if ((pos = name.find ('(')) != std::string::npos) {
            // parse index
            int first;
            std::string left;
            if (!ParseArrayIndex (name.c_str() + pos, first, left)) {
               return false;
            }
            // Is there more than one index?
            if (!left.empty()) {
               // remove the first index and 
               // prepend the rest of them to the remainder
               std::string n = name.substr (0, pos);
               if (n.empty()) { // column name can not be empty
                  return false;
               }
               char buf[64];
               sprintf (buf, "%s(%i)", n.c_str(), first);
               name = buf;
               n += left;
               rem = n + "." + rem;
            }
         }
      }
      // if no dot, find first array index
      else if ((pos = name.find ('(')) != std::string::npos) {
         // parse index
         int first;
         std::string left;
         if (!ParseArrayIndex (name.c_str() + pos, first, left)) {
            return false;
         }
         // remove array index
         rem = name.substr (0, pos);
         if (rem.empty()) { // column name can not be empty
            return false;
         }
         // add the rest of them back
         if (!left.empty()) {
            rem += left;
         }
         // resolve index
         char buf[64];
         sprintf (buf, "Event(%i)", first);
         name = buf;
         if (strcasecmp (rem.c_str(), "Event") == 0) {
            rem = "";
         }
      }
      // if no dot and no array index, it must be a column
      else {
         rem = "";
         if (name.empty()) { // column name can not be empty
            return false;
         }
      }
   
      nameFirst = name;
      nameRem = rem;
   
      return true;
   }

//______________________________________________________________________________
   bool LayoutInfo::ParseArrayIndex (const char* str, 
                     int& first, std::string& left)
   {
      // check if non null
      const char* p = str;
      if (!p) {
         return false;
      }
      // open parenthesis?
      if (*p != '(') {
         return false;
      }
      ++p;
      // digit?
      if (!isdigit (*p)) {
         return false;
      }
      // get firs index
      first = atoi (p);
      while (isdigit (*p)) ++p;
      // check if finished
      left = "";
      if (*p == ')') {
         return true;
      }
      // get remainder
      left = "(";
      while (isdigit (*p) || (*p == ',')) {
         left += *p;
         ++p;
      }
      left += ")";
      // check for closing parenthesis
      return (*p == ')');
   }

//______________________________________________________________________________
   const LayoutInfo* LayoutInfo::Lookup (const Type& type)
   {
      return Factory::Get().LookupLayout (type);
   }

//______________________________________________________________________________
   const LayoutInfo* LayoutInfo::GetSimple()
   {
      return Lookup (Type (Simple()));
   }

//______________________________________________________________________________
   const LayoutInfo* LayoutInfo::GetStandard()
   {
      return Lookup (Type (Standard()));
   }

//______________________________________________________________________________
   std::string LayoutInfo::Coincidence (int order) 
   {
      char buf[64];
      sprintf (buf, "Coincidence%i", order);
      return buf; 
   }

//______________________________________________________________________________
   std::string LayoutInfo::Cluster (int order) 
   {
      char buf[64];
      sprintf (buf, "Cluster%i", order);
      return buf; 
   }

}
