#ifndef _LIGO_EVENTLAYOUT_H
#define _LIGO_EVENTLAYOUT_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Layout							*/
/*                                                         		*/
/* Module Description: Defines an event					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Layout.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <iosfwd>
#include "Time.hh"
#include "events/Name.hh"
#include "events/Type.hh"
#include "events/ColumnType.hh"
#include "events/ColumnInfo.hh"


namespace events {


   class LayoutInfo;
   class Value;


/** An event layout is used to describe an event. The layout basically
    contains a pointer to a event layout information record. A layout
    can be registered which means it points to a global event layout
    registered by the event factory. A registerd layout is read-only.
    A layout owns the layout information record if it isn't registered.
   
    @memo Defines an event layout
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Layout : public ColumnType {   
   public:      
      /** Creates an event layout from a layout information record.
          @memo Default constructor
       ******************************************************************/
      Layout (const LayoutInfo* info = 0);
      /** Creates an event layout from a type.
          @memo Constructor
       ******************************************************************/
      explicit Layout (const Type& type) : mInfo (0) {
         SetType (type); }
      /** Creates an event layout from type name and subtype name.
          @memo Constructor
       ******************************************************************/
      explicit Layout (const char* name) 
      : mInfo (0) {
         SetType (Type (name)); }
      /** Creates an event layout from type name and subtype name.
          @memo Constructor
       ******************************************************************/
      explicit Layout (const std::string& name) 
      : mInfo (0) {
         SetType (Type (name)); }
      /** Copy constructor.
          @memo Copy constructor
       ******************************************************************/
      Layout (const Layout& layout) : mInfo (0) {
         *this = layout; }
      /** Destroys an event layout.
          @memo Destructor
       ******************************************************************/
      ~Layout();
      /** Assignment operator.
          @memo Assignment operator
       ******************************************************************/
      Layout& operator= (const Layout& layout);
   
      /** Equality operator.
          @memo Equality operator
       ******************************************************************/
      bool operator== (const Layout& layout) const;
      /** Inequality operator.
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const Layout& layout) const {
         return !(*this == layout); }
   
      /** Is this a valid registered layout?
          @memo Registered layout?
       ******************************************************************/
      bool IsRegistered() const;
      /**  Get reference count
           @memo Get reference count
       ******************************************************************/
      int GetRefCount () const;
   
      /**  Returns the number of columns of the event.
           @memo Gets the column number of an event
       ******************************************************************/
      int GetColumNum (const_data_ptr data) const;
      /**  Returns the event name
           @memo Gets the event name
       ******************************************************************/
      Name GetName (const_data_ptr data) const;
      /**  Sets the event name
           @memo Sets the event name
       ******************************************************************/
      void SetName (data_ptr data, const Name& name);
      /**  Returns the event time
           @memo Gets the event time
       ******************************************************************/
      Time GetTime (const_data_ptr data) const;
      /**  Sets the event time
           @memo Sets the event time
       ******************************************************************/
      void SetTime (data_ptr data, const Time& time);
      /**  Returns the event ifo set
           @memo Gets the event ifo set
       ******************************************************************/
      ColumnType::ColumnIfo_t GetIfo (const_data_ptr data) const;
      /**  Sets the event ifo set
           @memo Sets the event ifo set
       ******************************************************************/
      void SetIfo (data_ptr data, const ColumnType::ColumnIfo_t& ifo);
      /**  Returns the event ifo set string
           @memo Gets the event ifo string
       ******************************************************************/
      std::string GetIfoStr (const_data_ptr data) const;
      /**  Sets the event ifo set
           @memo Sets the event ifo set
       ******************************************************************/
      void SetIfoStr (data_ptr data, const char* ifo);
      /**  Returns the data value of the specified column. 
           If the specified column was added to the layout after the 
           event was created, the default value for this column type 
           will be returned. The event data block will not be changed.
           @memo Gets the column value of an event
           @param name Name of column
           @param data Pointer to event data block
           @param val Return value (return)
           @return true if column exists and can be read
       ******************************************************************/
      bool GetValue (const char* name, const_data_ptr data, 
                    Value& val) const;
      /**  Sets the data value of the specified column. If the specified
           column was added to the layout after the event was created,
           the event data block will be automatically extended.
           @memo Sets the column value of an event
           @param name Name of column
           @param data Pointer to event data block
           @param val Set value
           @return true if column exists and can be set
       ******************************************************************/
      bool SetValue (const char* name, data_ptr& data, 
                    const Value& val);
   
      /**  Sets the event type of the layout. The type has to be 
           corerspond to a registered layout.
           @memo Sets the event type
       ******************************************************************/
      bool SetType (const Type& type);
      /**  Returns the event type of the layout
           @memo Gets the event type
       ******************************************************************/
      bool GetType (Type& type) const;
      /**  Checks if the layout corresponds to the specified type.
           @memo Compatible type?
       ******************************************************************/
      bool IsCompatible (const Type& type) const;
      /** Get the column information
          @memo Column information
       ******************************************************************/
      const ColumnInfoList& GetColumnList() const;
      /** Add a column to the layout.
          @memo Add a column
       ******************************************************************/
      bool AddColumn (const ColumnInfo& col);
      /** Add a column to the layout.
          @memo Add a column
       ******************************************************************/
      bool AddColumn (const char* name, Enum type) {
         return AddColumn (ColumnInfo (name, type)); }
      /** Remove a column from the layout.
          @memo Remove a column
       ******************************************************************/
      bool RemoveColumn (const char* name);
      /** Get a column from the layout.
          @memo Get a column
       ******************************************************************/
      const ColumnInfo* GetColumn (const char* name) const;
   
      /**  Size of data block described by layout.
           @memo Data block size
       ******************************************************************/
      int DataSize () const;
      /**  Construct the event data (optional copy construct)
           @memo Construct event data
       ******************************************************************/
      bool Construct (data_ptr data, const_data_ptr init = 0);
      /**  Destruct the event data
           @memo Destruct event data
       ******************************************************************/
      bool Destruct (data_ptr data);
      /** Updates the event data block, if columns have been added. 
          Fills added columns with default values.
          @memo Update the event
       ******************************************************************/
      bool Update (data_ptr& data);
      /**  Compares event data
           @memo Compares event data
           @return true if equal
       ******************************************************************/
      bool Compare (const_data_ptr d1, const_data_ptr d2) const;
      /**  Swap a layout with the current
           @memo Swap a layout
       ******************************************************************/
      void Swap (Layout& l);
   
      /**  Register the current layout with the event factory.
           @memo Register the layout globaly
           @param usesubtype True if layout should be subtype specific
           @return True if successful
       ******************************************************************/
      bool Register();
   
      /**  Dump column names to specified output stream.
           @memo Dump column names to specified output stream.
           @param os output stream
       ******************************************************************/
      void Dump (std::ostream& os) const;
      /**  Dump column names to standard output stream.
           @memo Dump column names to specified output stream.
           @param os output stream
       ******************************************************************/
      void Dump() const;
   
      /**  Dump all registered layouts to specified output stream.
           @memo Dump all registered layouts to specified output stream.
           @param os output stream
       ******************************************************************/
      static void DumpAll (std::ostream& os);
      /**  Dump all registered layouts to standard output stream.
           @memo Dump all registered layouts to specified output stream.
           @param os output stream
       ******************************************************************/
      static void DumpAll();
   
      /**  Get the minimal layout.
           @memo Get "simple" layout
           @return Simple layout
       ******************************************************************/
      static const Layout& GetSimple();
      /**  Get the standard layout.
           @memo Get "standard" layout
           @return Standard layout
       ******************************************************************/
      static const Layout& GetStandard();
      /**  Get the coincidence layout. The coincidence order determines 
           the subtype.
           @memo Get "coincidence" layout
           @param Coincidence order
           @return Coincidence layout
       ******************************************************************/
      static Layout GetCoincidence (int order = 2);
      /**  Get the cluster layout. The number of events in the cluster
           determines the subtype.
           @memo Get "cluster" layout
           @param eventnum Number of events in cluster
           @return Cluster layout
       ******************************************************************/
      static Layout GetCluster (int eventnum);
   
   private:
      /// Pointer to layout information
      LayoutInfo*	mInfo;
   
      /// Simple Layout
      static Layout gSimple;
      /// Standard Layout
      static Layout gStandard;
   };


}

#endif // _LIGO_EVENTLAYOUT_H
