/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef READTREND_HH
#define READTREND_HH

#include "TSeries.hh"
#include "framedir.hh"
#include "Dacc.hh"
#include <string>
#include <vector>

class Time;

/**  %ReadTrend class reads in one or more trends and returns requested
  *  statistical fields as TSeries. The trend files to be read in are
  *  found from a FrameDir directory built from all files matching the
  *  format \c \<directory\>/\<prefix\>*\<postfix\>. The directory, 
  *  prefix and postfix may be specified explicitly, or a monitor name may 
  *  be specified, in which case, the file name fields are derived from the 
  *  monitor name and local defaults.
  * 
  *  The following statistical values can be extracted from the trend frames:
  *  <table>
  *  <tr><td> Name  </td><td> Contents </td></tr>
  *  <tr><td> count </td><td> Number of points in average </td></tr>
  *  <tr><td> delta </td><td> %Difference between min and max </td></tr>
  *  <tr><td> error </td><td> Error on the mean </td></tr>
  *  <tr><td> max   </td><td> Maximum value </td></tr>
  *  <tr><td> mean  </td><td> Mean value. </td></tr>
  *  <tr><td> min   </td><td> Minimum value </td></tr>
  *  <tr><td> rms   </td><td> RMS value </td></tr>
  *  <tr><td> sigma </td><td> Standard deviation </td></tr>
  *  </table>
  *  @memo Read trend data
  *  @author J. Zweizig
  *  @version 1.3; Last modified: July 13, 2010
  */
class ReadTrend {
public:
    /**  Enumerate trend types
      */
    enum TrendType {
	/// Second trend
	kSecond,
	/// Minute trend
	kMinute,
	/// Trend with a non-standard time increment
	kNonStandard
    };

    /**  Enumerated data fields
      */
    enum typeID {
	///  Mean value
	kMean,
	///  Standard deviation
	kSigma,
	///  Minimum value
	kMin,
	///  Maximum value
	kMax,
	///  Error on the mean (sigma / sqrt(N))
	kError,
	///  Number of averaged pointe
	kCount,
	///  Root mean square
	kRMS,
	///  Peak range (Max - Min)
	kDelta
    };

    /**  Vector of channel names or trend type names.
      */
    typedef std::vector<std::string> string_vect;

    /**  Vector of time series.
      */
    typedef std::vector<TSeries>     tseries_vect;

    /**  Vector of data field IDs
      */
    typedef std::vector<typeID>      type_vect;

public:
    /**  Construct a null trend reader.
      *  @memo Default constructor.
      */
    ReadTrend(void);

    /**  Construct a trend reader object, specifying the file prefix and 
      *  postfix.
      *  @memo Data constructor.
      *  @param dir     A name or RE specifying the directory(ies) containing 
      *                 the minute trends.
      *  @param prefix  %Trend file name prefix.
      *  @param postfix %Trend file postfix (extension).
      *  @param Type    %Trend type
      *  @return Error code or 0.
      */
    ReadTrend(const char* dir, const char* prefix=0, 
	      const char* postfix=0, TrendType Type=kMinute);

    /**  Get a time series containing the maximum values for each trend 
      *  increment for the specified channel.
      *  @memo Get the maximum values from the specified trend.
      *  @param chan Channel name
      *  @param t0   Start time.
      *  @param dT   %Time interval.
      *  @param Max  Maximum value time series.
      *  @return Error code or 0.
      */
    int getMaxSeries(const char* chan, const Time& t0, Interval dT, 
		     TSeries* Max);

    /**  Get the mean value and optionally the error on the mean for a 
      *  specified channel and time interval.
      *  @memo Get mean and error time series.
      *  @param chan Channel name
      *  @param t0   Start time.
      *  @param dT   Time interval.
      *  @param Avg  Average value time series.
      *  @param Err  Error time series.
      *  @return Error code or 0.
      */
    int getSeries(const char* chan, const Time& t0, Interval dT, 
		  TSeries* Avg, TSeries* Err=0);

    /**  Get a time series of one or more statistical quantities.
      *  @memo get Statistics time series.
      *  @param chan  Vector of channel names.
      *  @param field Vector of field names.
      *  @param t0    Start time.
      *  @param dT    Time interval.
      *  @param Data  Vector of time series to receive data.
      *  @return Error code or 0.
      */
    int getSeries(const string_vect& chan, const string_vect& field, 
		  const Time& t0, Interval dT, tseries_vect& Data);


    /**  Get time series of one or more statistical quantities extracted from 
      *  channels in a trend frame. The number of %channel/quantity pairs
      *  is arbitrary. The error codes are as reported by Dacc::fillData().
      *  @memo get Statistics time series.
      *  @param chan  Vector of channel names.
      *  @param field Vector of field IDs.
      *  @param t0    Start time.
      *  @param dT    %Time interval.
      *  @param Data  Vector of time series to receive data.
      *  @return Error code
      */
    int getSeries(const string_vect& chan, const type_vect& field, 
		  const Time& t0, Interval dT, tseries_vect& Data);

    /**  Specify the frame directory.
      *  @memo Set file name search pattern
      *  @param fd Existing frame directory (FrameDir) to be copied.
      */
    void setDirectory(const FrameDir& fd);

    /**  Specify file name search pattern. The file search look for all
      *  files of the form &lt;dir&gt;/&lt;prefix&gt;*&lt;postfix&gt;
      *  @memo Set file name search pattern
      *  @param dir     Directory name pattern.
      *  @param prefix  prefix string.
      *  @param postfix postfix string.
      */
    void setDirectory(const std::string& dir, const std::string& prefix, 
		      const std::string& postfix);

    /**  Specify file name search pattern for appropriate data. %setMonitor
      *  sets the file names as follows:
      *  <table>
      *  <tr><td>&lt;directory&gt;</td><td>$DMTRENDOUT </td></tr>
      *  <tr><td>&lt;prefix&gt;   </td>
      *      <td>&lt;site&gt;-&lt;monitor&gt;_[TM]- </td></tr>
      *  <tr><td>&lt;postfix&gt;  </td><td>-&lt;duration&gt;.gwf </td></tr>
      *  </table>
      *  @memo Set file name search pattern
      *  @param mon Monitor name.
      */
    void setMonitor(const char* mon);

    /**  Set the trend type.
      *  @memo Set the trend type.
      *  @param Typ %Trend type code.
      */
    void setType(TrendType Typ);

    /**  Set the debug flag to print a lot of debugging information.
      *  @memo set the debug flag.
      */
    void setDebug(void);

    /**  Remove all file name entries from the frame directory.
      *  @memo expunge frame directory.
      */
    void expunge(void);

    //==================================  Static entries
    /**  Concatenate a trend sub-channel extension to the end of a channel 
      *  name.
      *  @memo Get channel name plus extension string.
      *  @param chan %Channel name (no extension).
      *  @param ext  %Trend sub-channel type code.
      *  @returns String containing the sub-channel name.
      */
    static std::string namex(const std::string& chan, typeID ext);

    /**  Convert an extension name to an trend sub-channel type code.
      *  @memo Get sub-channel type code.
      *  @param ext Extension name.
      *  @return %Trend sub-channel type code.
      */
    static typeID extcode(const std::string& ext);

    //==================================  Private methods
private:
    void buildDirectory(const char* name);
    void open(const Time& t0, Interval dT);
    void close(void);

    //==================================  Local data
private:
    std::string mDirec;
    std::string mPrefix;
    std::string mPostfix;
    TrendType   mType;

    enum {
	kEmpty,
	kPattern,
	kAssign
    } mFrameStat;
    FrameDir    mFrameDir;
    Dacc        mIn;
    bool        mDebug;
};

#endif // !def(READTREND_HH)
