/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "PConfig.h"
#include <sstream>
#include <cstring>
#include <cstdio>
#include <iostream>
#include "monapi/monitorapi.hh"
#include "monapi/monaccess.hh"
#include "Time.hh"
#include "Interval.hh"
#include "TSeries.hh"
#include "FSeries.hh"
#include "FSpectrum.hh"
#include "Histogram1.hh"
#include "PlotSet.hh"
#include "xml/Xsil.hh"


namespace monapi {
   using namespace std;
   using namespace xml;

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// TLGMonitorDatum                                                      //
//                                                                      //
// A data object                                                        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   void TLGMonitorDatum::Init()
   {
      fUpdateTime = Time(0);
      fUpdateInterval = 60; // 60 sec
      fGraph = "";
      fAChn = "";
      fBChn = "";
      fParam = new ParameterDescriptor ();
      fCal = new calibration::Descriptor ();
      fMC = 0;
   }

//______________________________________________________________________________
   void TLGMonitorDatum::SetOpt (const char* opt)
   {
      if (fCal) {
         fCal->Init();
         fCal->SetTime (fUpdateTime);
      }
      if (true || (opt == 0) || (strlen (opt) == 0)) {
         switch (fType) {
            case TimeSeries:
               {
                  fGraph = kPTTimeSeries;
                  if (fCal) {
                     fCal->SetDomain (kCalUnitX, kCalDomainTime);
                  }
                  break;
               }
            case FrequencySeries:
               {
                  fGraph = kPTFrequencySeries;
                  if (fCal) {
                     fCal->SetDomain (kCalUnitX, kCalDomainFrequency);
                     fCal->SetDensity (kCalUnitX, kCalDensityAmpPerRtHz);
                  }
                  break;
               }
            case Spectrum:
               {
                  fGraph = kPTPowerSpectrum;
                  if (fCal) {
                     fCal->SetDomain (kCalUnitX, kCalDomainFrequency);
                     fCal->SetDensity (kCalUnitX, kCalDensityAmpPerRtHz);
                  }
                  break;
               }
            case Histogram1D:  // hist (mito)
               {
                  fGraph = kPTHistogram1;
                  break;
               }
         }
         fAChn = second;
         fBChn = "";
      }
      else {
         // option parsing
      }
      if (fCal) {
         fCal->SetChannel (0, fAChn.c_str());
         fCal->SetChannel (1, fBChn.c_str());
         fCal->SetValid();
      }
   }

//______________________________________________________________________________
   TLGMonitorDatum::~TLGMonitorDatum() 
   {
      if (fMC) delete fMC;
      if (fCal) delete fCal;
      if (fParam) delete fParam;
   }

//______________________________________________________________________________
   TLGMonitorDatum& TLGMonitorDatum::operator= 
   (const TLGMonitorDatum& mon)
   {
      if (this != &mon) {
         first = mon.first;
         second = mon.second;
         fType = mon.fType;
         fUpdate = mon.fUpdate;
         fUpdateInterval = mon.fUpdateInterval;
         fUpdateTime = mon.fUpdateTime;
         fGraph = mon.fGraph;
         fAChn = mon.fAChn;
         fBChn = mon.fBChn;
         if (fParam) delete fParam;
         fParam = new ParameterDescriptor (mon.fParam);
         if (fCal) delete fCal;
         fCal = new calibration::Descriptor (mon.fCal);
         fMC = 0;
         if ((fUpdate != kMonUpdateNever) && (mon.fMC != 0)) {
            Connect(); 
         }
      }
      return *this;
   }

//______________________________________________________________________________
   bool TLGMonitorDatum::Connect() 
   {
      if (fUpdate == kMonUpdateNever) {
         return false;
      }
      if (fMC == 0) {
         fMC = monaccess::install();
      }
      if (fMC == 0) {
         cerr << "Unable to install monitor access library" << endl;
         return false;
      }
      if (first.empty()) {
         cerr << "No server defined" << endl;
         return false; 
      }
      return fMC->setServer (first.c_str());
   }

//______________________________________________________________________________
   bool TLGMonitorDatum::Ready() const
   {
      switch (fUpdate) {
         case kMonUpdateNever:
         case kMonUpdateUser:
         default:
            return false;
         case kMonUpdateRepeat:
            return (Now() > fUpdateTime + Interval (fUpdateInterval));
         case kMonUpdateWhenChanged:
            return false;
      }
   }

//______________________________________________________________________________
   void TLGMonitorDatum::Touch() 
   {
      fUpdateTime = Now();
   }

//______________________________________________________________________________
   PlotDescriptor* TLGMonitorDatum::GetData ()
   {
      int avrg;
   
      // check if valid
      if ((fUpdate == kMonUpdateNever) || 
         (fParam == 0) || (fCal == 0)) {
         return 0;
      }
      if ((fMC == 0) && !Connect()) {
         return 0;
      }
      // get data
      cerr << "Get " << second << " from " << first << endl;
      PlotDescriptor* pd = 0;
      bool succ = false;
      switch (fType) {
      // Time series
      case TimeSeries:
	 {
	    TSeries* series = new TSeries();
	    DataCopy* dc = new DataCopy();
	    if (series) {
	       // set data
	       succ = fMC->getData (second.c_str(), series);
	       if (!succ) cerr << "Failed: " << second << endl;
	       // set parameters
	       if (succ && fParam) {
		  fParam->SetThird (0);
		  ostringstream os;
		  Time t0 = series->getStartTime();
		  if (fCal) fCal->SetTime (t0);
		  if (!t0) {
		     t0 += Interval (series->getTStep());
		  }
		  os << xsilTime ("t0", t0.getS(), t0.getN()) << endl;
		  double dt = (double)series->getTStep(); //TSeries
		  os << xsilParameter<double> ("dt", dt) << endl;
		  double f0 = (double)series->getF0();
		  if (f0 != 0) 
		     os << xsilParameter<double> ("f0", f0) << endl;
		  avrg = 0; // TSeries
		  if (!avrg) avrg =1;
		  os << xsilParameter<int> ("Averages", avrg) << endl;
		  if (GetAChannel()) 
		     os << xsilParameter<const char*> 
			("Channel", GetAChannel()) << endl;
		  int n = series->getNSample();
		  os << xsilParameter<int> ("N", n);
		  fParam->SetUser (os.str().c_str());

		  // set temporaly DataCopy
		  float* data = new float[n];
		  series->getData(n,data);
		  dc->SetData(0.0,(float)dt,data,n);
		  if (data) delete[] data;
	       }
	       // set calibration
	       if (fCal) {
		  fCal->SetBW (kCalUnitY, 0);
	       }
	    }

	    // make plot descriptor
	    if (series && succ) {
	       if (fParam) {
		  fParam->SetStartTime (series->getStartTime().getS(),
					series->getStartTime().getN());
		  if (avrg != 0) {
		     fParam->SetAverages (avrg);
		  }
		  else {
		     fParam->ResetAverages ();
		  }
	       }

	       pd = new PlotDescriptor (dc, GetGraphType(), GetAChannel(), 
					GetBChannel(), fParam, fCal);
	       dc = 0;
	    }
	    if (series) delete series;
	    if (dc)     delete dc;
	    Touch();
	    break;
	 }
      // frequency series
      case FrequencySeries:
	 {
	    FSeries* series = new FSeries();
	    DataCopy* dc = new DataCopy();
	    if (series) {
	       succ = fMC->getData (second.c_str(), series);
	       if (!succ) cerr << "Failed: " << second << endl;
	       // double bw = ((FSeries*)series)->getBW();
	       double bw = 0; // FSeries
	       if (succ && fParam) {
		  if (bw > 0) {
		     char buf[128];
		     sprintf (buf, "BW=%g", bw);
		     fParam->SetThird (buf);
		  }
		  else {
		     fParam->SetThird (0);
		  }
		  ostringstream os;
		  Time t0 = series->getStartTime();
		  if (fCal) fCal->SetTime (t0);
		  os << xsilTime ("t0", t0.getS(), t0.getN()) << endl;
		  double f0 = series->getCenterFreq(); // FSeries
		  os << xsilParameter<double> ("f0", f0) << endl;
		  double df = series->getFStep(); // FSeries
		  os << xsilParameter<double> ("df", df) << endl;
		  os << xsilParameter<double> ("BW", bw) << endl;
		  int win = 1;
		  os << xsilParameter<int> ("Window", win) << endl;
		  // int avrg = series->getAvrg();
		  avrg = 0; // FSeries
		  if (!avrg) avrg = 1;
		  os << xsilParameter<int> ("Averages", avrg) << endl;
		  if (GetAChannel()) 
		     os << xsilParameter<const char*>("ChannelA", GetAChannel())
			<< endl;
		  if (GetBChannel()) 
		     os << xsilParameter<const char*>
			   ("ChannelB[0]", GetBChannel()) << endl;
		  int n = series->getNStep()+1; // FSeries
		  os << xsilParameter<int> ("N", n) << endl;
		  os << xsilParameter<int> ("M", 1);
		  fParam->SetUser (os.str().c_str());
                  
                  // set temporaly DataCopy	
		  fComplex* cmplx = new fComplex[n];
		  series->getData(n,cmplx);
		  float* data = new float[2*n];
		  memcpy(data,cmplx,2*n*sizeof(float));
		  dc->SetData((float)f0,(float)df,data,n,true);
		  if (cmplx) delete cmplx;
		  if (data) delete data;
	       }
	       if (succ && fCal) {
		  fCal->SetBW (kCalUnitY, bw > 0 ? bw : 0);
	       }
	    }

            // make plot descriptor
	    if (series && succ) {
	       if (fParam) {
		  fParam->SetStartTime (series->getStartTime().getS(),
					series->getStartTime().getN());
		  if (avrg != 0) {
		     fParam->SetAverages (avrg);
		  }
		  else {
		     fParam->ResetAverages ();
		  }
	       }

	       pd = new PlotDescriptor (dc, GetGraphType(), GetAChannel(), 
					GetBChannel(), fParam, fCal);
	       dc = 0;
	    }
	    if (series) delete series;
	    if (dc)     delete dc;
	    Touch();
	    break;
	 }
         // spectrum
      case Spectrum:
	 {
	    FSpectrum* series = new FSpectrum();
	    DataCopy* dc = new DataCopy();
	    if (series) {
	       succ = fMC->getData (second.c_str(), series);
	       if (!succ) cerr << "Failed: " << second << endl;
	       // double bw = ((FSpectrum*)series)->getBW();
	       double bw = 0; // FSpectrum
	       if (succ && fParam) {
		  if (bw > 0) {
		     char buf[128];
		     sprintf (buf, "BW=%g", bw);
		     fParam->SetThird (buf);
		  }
		  else {
		     fParam->SetThird (0);
		  }
		  ostringstream os;
		  Time t0 = series->getStartTime();
		  if (fCal) fCal->SetTime (t0);
		  os << xsilTime ("t0", t0.getS(), t0.getN()) << endl;
		  double f0 = series->getLowFreq(); //FSpectrum
		  os << xsilParameter<double> ("f0", f0) << endl;
		  double df = series->getFStep(); //FSpectrum
		  os << xsilParameter<double> ("df", df) << endl;
		  os << xsilParameter<double> ("BW", bw) << endl;
		  int win = 1;
		  os << xsilParameter<int> ("Window", win) << endl;
		  avrg = series->getCount(); //FSpectrum   
		  if (!avrg) avrg = 1;
		  os << xsilParameter<int> ("Averages", avrg) << endl;
		  if (GetAChannel()) 
		     os << xsilParameter<const char*> 
			("ChannelA", GetAChannel()) << endl;
		  if (GetBChannel()) 
		     os << xsilParameter<const char*> 
			("ChannelB[0]", GetBChannel()) << endl;
		  int n = series->getNStep()+1; //FSpectrum
		  os << xsilParameter<int> ("N", n) << endl;
		  os << xsilParameter<int> ("M", 1);
		  fParam->SetUser (os.str().c_str());

                  // set temporaly DataCopy
		  float* data = new float[n];
		  series->getData(n,data);
		  dc->SetData((float)f0,(float)df,data,n);
		  if (data) delete data;
	       }
	       if (succ && fCal) {
		  fCal->SetBW (kCalUnitY, bw > 0 ? bw : 0);
	       }
	    }

	    // make plot descriptor
	    if (series && succ) {
	       if (fParam) {
		  fParam->SetStartTime (series->getStartTime().getS(),
					series->getStartTime().getN());
		  if (avrg != 0) {
		     fParam->SetAverages (avrg);
		  }
		  else {
		     fParam->ResetAverages ();
		  }
	       }

	       pd = new PlotDescriptor (dc, GetGraphType(), GetAChannel(), 
					GetBChannel(), fParam, fCal);
	       dc = 0;
	    }
	    if (series) delete series;
	    if (dc)     delete dc;
	    Touch();
	    break;
	 }
         // histogram (mito)
      case Histogram1D:
	 {
	    Histogram1* series = new Histogram1();
	    HistDataCopy* dc = new HistDataCopy();
	    if (series) {
	       succ = fMC->getData(second.c_str(),series);
	       if (!succ) cerr << "Failed: " << second << endl;
	       int nent = series->GetNEntries();
	       int nbinx = series->GetNBins();
	       if (nbinx>0){
		  double*  edge  = new double[nbinx+1];
		  double*  content = new double[nbinx+2];
		  double*  error = new double[nbinx+2];
		  double*  stat = new double[4];
		  series->GetBinLowEdges(edge);
		  series->GetBinContents(content);
		  series->GetStats(stat);

		  bool xydata=(series->GetBinType()==Histogram1::kVariableBin);

		  if (series->IsErrorFlagON()) {
		     series->GetBinErrors(error);
		     dc->SetData(edge,content,error,nbinx,
				 series->GetXLabel(),
				 series->GetNLabel(),nent,stat,xydata);
		  }
		  else {
		     dc->SetData(edge,content,nbinx,
				 series->GetXLabel(),
				 series->GetNLabel(),nent,stat,xydata);
		  }

		  if (fCal) fCal->SetTime (series->GetTime());
		  ostringstream os;
		  os << xsilTime("t0",
				 series->GetTime().getS(),
				 series->GetTime().getN())
		     << endl;
		  os << xsilParameter<int> ("NBinx",nbinx) << endl;
		  os << xsilParameter<int> ("NData",nent) << endl;
		  os << xsilParameter<double> ("SumWeight",stat[0]) << endl;
		  os << xsilParameter<double> ("SumWeightSqr",stat[1]) << endl;
		  os << xsilParameter<double> ("SumWeightX",stat[2]) << endl;
		  os << xsilParameter<double> ("SumWeightXSqr",stat[3]) << endl;

		  if (strcasecmp(series->GetTitle(),GetAChannel()) == 0 ) {
		     os << xsilParameter<string> ("Title",series->GetTitle()) 
			<< endl;
		  }
		  else {
		     os << xsilParameter<string> ("Title",GetAChannel()) 
			<< endl;
		  }
		  if ( series->GetXLabel() )
		     os << xsilParameter<string> ("XLabel",series->GetXLabel())
			<< endl;
		  if ( series->GetNLabel() )
		     os << xsilParameter<string> ("NLabel",series->GetNLabel())
			<< endl;
		  if (series->GetBinType() == Histogram1::kFixedBin) {
		     os << xsilParameter<double> ("XLowEdge",
						  series->GetBinLowEdge(1))
			<< endl;
		     os << xsilParameter<double> ("XSpacing", 
						  series->GetBinSpacing())
			<< endl;
		  }
		  fParam->SetUser (os.str().c_str());

		  if (edge)  delete[] edge;
		  if (content) delete[] content;
		  if (error) delete[] error;
		  if (stat) delete[] stat;
	       }
	    }
	    // make plot descriptor
	    if (series && succ) {
	       if (fParam) {
		  fParam->SetStartTime (series->GetTime().getS(),
					series->GetTime().getN());
	       }
	       pd = new PlotDescriptor (dc, GetGraphType(), GetAChannel(), 
					GetBChannel(), fParam, fCal);
	       dc = 0;
	    }
	    if (series) delete series;
	    if (dc)     delete dc;
	    Touch();
	    break;
	 }
      default:
	 pd = 0;
      }
      return pd;
   }

//______________________________________________________________________________
   bool TLGMonitorDatum::write (ostream& os, int index) const
   {
      char pname[256];
      if (index >= 0) {
         sprintf (pname, "Monitor[%i]", index);
      }
      else {
         sprintf (pname, "Monitor");
      }
      os << xsilDataBegin (pname, GetTypename().c_str(), 
                          "Settings") << endl;
      os << xsilParameter<const char*> ("Server", 
                           GetMonitorName()) << endl;
      os << xsilParameter<const char*> ("DataObject",
                           GetDataObjectName()) << endl;
      os << xsilParameter<int> ("UpdateType", 
                           (int) GetUpdateOpt()) << endl;
      os << xsilParameter<double> ("UpdateInterval", 
                           GetUpdateInterval()) << endl;
      os << xsilParameter<const char*> ("PlotType",
                           GetGraphType()) << endl;
      os << xsilParameter<const char*> ("AChannel",
                           GetAChannel()) << endl;
      if (GetBChannel() != 0) {
         os << xsilParameter<const char*> ("BChannel",
                              GetBChannel()) << endl;
      }
      os << xsilDataEnd<float> ();
      return !!os;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// TLGMonitorDatumList                                                  //
//                                                                      //
// A data object list                                                   //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   void TLGMonitorDatumList::add (TLGMonitorDatum* mon)
   {
      if (mon == 0) {
         return;
      }
      TLGMonitorDatum::name_t key (mon->GetMonitorName(), 
                           mon->GetDataObjectName());
      if (fList[key] != 0) {
         delete fList[key];
      }
      fList[key] = mon;
   }

//______________________________________________________________________________
   void TLGMonitorDatumList::remove (const TLGMonitorDatum::name_t& mon)
   {
      iterator iter = find (mon);
      if (iter != end()) {
         delete iter->second;
         iter->second = 0;
         fList.erase (iter);
      }
   }

//______________________________________________________________________________
   void TLGMonitorDatumList::clear ()
   {
      for (iterator iter = begin(); iter != end(); iter++) {
         delete iter->second;
         iter->second = 0;
      }
      fList.clear();
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// Stream functions (XML)                                               //
//                                                                      //
// Write and read monitors to and from a stream in XML format           //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   ostream& operator<< (ostream& os, const TLGMonitorDatum& mon)
   {
      mon.write (os);
      return os;
   }

//______________________________________________________________________________
   ostream& operator<< (ostream& os, const TLGMonitorDatumList& mlist)
   {
      int count = 0;
      for (TLGMonitorDatumList::const_iterator 
          i = mlist.begin(); i != mlist.end(); i++) {
         if (count > 0) os << endl;
         i->second->write (os, count);
         count++;
      }
      return os;
   }

//______________________________________________________________________________
   xsilHandlerMonitor::xsilHandlerMonitor (TLGMonitorDatumList& mon,
                     TLGMonitorDatum::SeriesType mtype)
   : fMList (&mon)
   {
      fMon = new (nothrow) TLGMonitorDatum ();
      if (fMon) fMon->SetType (mtype);
   }

//______________________________________________________________________________
   xsilHandlerMonitor::~xsilHandlerMonitor ()
   {
      if (fMList && fMon &&
         (strlen (fMon->GetMonitorName()) != 0) && 
         (strlen (fMon->GetDataObjectName()) != 0)) {
         fMon->Set (fMon->GetMonitorName(), fMon->GetDataObjectName(),
                   fMon->GetType());
         fMList->add (fMon);
         fMon = 0;
      }
      if (fMon) delete fMon;
   }

//______________________________________________________________________________
   bool xsilHandlerMonitor::HandleParameter (const string& name,
                     const attrlist& attr, const int& p, int N)
   {
      if (fMon == 0) {
         return false;
      }
      if (strcasecmp (name.c_str(), "UpdateType") == 0) {
         fMon->SetUpdateOpt ((TLGMonitorDatum::MonitorUpdate_t)p);
         return true;
      }
      else {
         return false;
      }
   }

//______________________________________________________________________________
   bool xsilHandlerMonitor::HandleParameter (const string& name,
                     const attrlist& attr, const double& p, int N)
   {
      if (fMon == 0) {
         return false;
      }
      if (strcasecmp (name.c_str(), "UpdateInterval") == 0) {
         fMon->SetUpdateInterval (p);
         return true;
      }
      else {
         return false;
      }
   }

//______________________________________________________________________________
   bool xsilHandlerMonitor::HandleParameter (const string& name,
                     const attrlist& attr, const string& p)
   {
      if (fMon == 0) {
         return false;
      }
      if (strcasecmp (name.c_str(), "Server") == 0) {
         fMon->SetMonitorName (p.c_str());
         return true;
      }
      else if (strcasecmp (name.c_str(), "DataObject") == 0) {
         fMon->SetDataObjectName (p.c_str());
         return true;
      }
      else if (strcasecmp (name.c_str(), "PlotType") == 0) {
         fMon->SetGraphType (p.c_str());
         return true;
      }
      else if (strcasecmp (name.c_str(), "AChannel") == 0) {
         fMon->SetAChannel (p.c_str());
         return true;
      }
      else if (strcasecmp (name.c_str(), "BChannel") == 0) {
         fMon->SetBChannel (p.c_str());
         return true;
      }
      else {
         return false;
      }
   }

//______________________________________________________________________________
   xsilHandler* xsilHandlerQueryMonitor::GetHandler (const attrlist& attr)
   {
      attrlist::const_iterator ni = attr.find (xmlName);
      attrlist::const_iterator ti = attr.find (xmlType);
      if ((ni == attr.end()) || (ti == attr.end())) {
         return 0;
      }
      TLGMonitorDatum::SeriesType mtype = (TLGMonitorDatum::SeriesType) -1;
      if (strcasecmp (ti->second.c_str(), kSeriesTypenames[0]) == 0) {
         mtype = TLGMonitorDatum::TimeSeries;
      }
      else if (strcasecmp (ti->second.c_str(), kSeriesTypenames[1]) == 0) {
         mtype = TLGMonitorDatum::FrequencySeries;
      }
      else if (strcasecmp (ti->second.c_str(), kSeriesTypenames[2]) == 0) {
         mtype = TLGMonitorDatum::Spectrum;
      }
      else if (strcasecmp (ti->second.c_str(), kSeriesTypenames[3]) == 0) {
         mtype = TLGMonitorDatum::Histogram1D;
      }
      if ((strncasecmp (ni->second.c_str(), "Monitor", 7) == 0) &&
         ((int) mtype != -1)) {
         return new xsilHandlerMonitor (*fMon, mtype);
      }
      else {
         return 0;
      }
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// TLGMonitorMgr                                                        //
//                                                                      //
// List of monitor servers and data object names                        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool TLGMonitorMgr::UpdateServices ()
   {
      fServices.clear();
      if (fMC == 0) {
         fMC = monaccess::install ();
      }
      if (fMC == 0) {
         return false;
      }
      std::string index;
      bool rc = fMC->getIndex (index);
      if (!rc) {
         cout << "error in nameserver lookup " << endl;
         return false;
      }
      // go through index
      while (!index.empty()) {
         string line;
         string::size_type pos = index.find_first_of (';');
         if (pos == string::npos) {
            line = index;
            index = "";
         }
         else {
            line = index.substr (0, pos);
            index.erase (0, pos + 1);
         }
         if (!line.empty()) {
            fServices.insert (servicelist::value_type (line, dobjects()));
         }
      }
      return true;
   }

//______________________________________________________________________________
   bool TLGMonitorMgr::Initialized (const char* service)
   {
      if (fMC == 0) {
         return false;
      }
      // find service and clear data object list
      servicelist::iterator i = fServices.find (service);
      if (i == fServices.end()) {
         return false;
      }
      return i->second.Initialized();
   }

//______________________________________________________________________________
   bool TLGMonitorMgr::UpdateDObjects (const char* service)
   {
      if (fMC == 0) {
         return false;
      }
      // find service and clear data object list
      servicelist::iterator i = fServices.find (service);
      if (i == fServices.end()) {
         return false;
      }
      i->second.clear();
   
      // get data object index
      fMC->setServer (service);
   
      string name;
      string typen;
      string opt;
      for (int n = 0; fMC->getServerEntry (n,name,typen,opt); ++n) {
         // cout << "object name: " << name << endl;
         // cout << "object type: " << typen << endl;
         // cout << "object comm: " << opt << endl;
         TLGMonitorDatum::SeriesType type;
         if ((typen == kSeriesTypenames[0]) ||
            (typen == kSeriesTypenamesAlt[0])) {
            type = TLGMonitorDatum::TimeSeries;
         }
         else if ((typen == kSeriesTypenames[1]) ||
                 (typen == kSeriesTypenamesAlt[1])) {
            type = TLGMonitorDatum::FrequencySeries;
         }
         else if ((typen == kSeriesTypenames[2]) ||
                 (typen == kSeriesTypenamesAlt[2]))  {
            type = TLGMonitorDatum::Spectrum;
         }
         else if ((typen == kSeriesTypenames[3]) ||
                 (typen == kSeriesTypenamesAlt[3]))  {
            type = TLGMonitorDatum::Histogram1D;
         }
         else {
            continue;
         }
         dobject obj;
         obj.fType = type;
         obj.fOption = opt;
         i->second.insert (dobjlist::value_type (name, obj));
      }
      i->second.SetInit();
      return true;
   }

//______________________________________________________________________________
   bool TLGMonitorMgr::UpdateAll ()
   {
      if (!UpdateServices()) {
         return false;
      }
      for (servicelist::iterator i = fServices.begin(); 
          i != fServices.end(); i++) {
         UpdateDObjects (i->first.c_str());
      }
      return true;
   }


}
