/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIG_DQSEGTABLE_HH
#define TRIG_DQSEGTABLE_HH

#include "ProcTable.hh"

namespace trig {

    class Segment;
    class TrigBase;

    /**  S5 Data quality database segment definer descriptor class.
      */
    class DQSegDef {
    public:
	DQSegDef(void);
	DQSegDef(const Segment& seg);
	bool operator==(const DQSegDef& x) const;
	int getVersion(void) const;
	const std::string& refProcess(void) const;
	const std::string& refDefinerID(void) const;
	const std::string& refIfos(void) const;
	const std::string& refName(void) const;
	const std::string& refComment(void) const;
	void setDefinerID(const std::string& defid);
    private:
	std::string mProcID;
	std::string mDefID;
	std::string mIfos;
	std::string mName;
	int         mVersion;
	std::string mComment;
    };

    /**  The SegTable class is a meta-database interface producing a segment
      *  database table.
      */
    class DQSegTable : public xsil::MetaTable {
    public:
	DQSegTable(void);
	~DQSegTable(void);
	void addRow(const Segment& s);
	void check(const char* title) const;
    private:
	DBTypes::integer_4 start_time;
	DBTypes::integer_4 end_time;
	DBTypes::integer_4 active;
	DBTypes::varchar   segid;
	DBTypes::varchar   procid;
    };

    /**  The DQSegDefTable class is a meta-database interface producing a 
      *  segment definition database table.
      */
    class DQSegDefTable : public xsil::MetaTable {
    public:
	DQSegDefTable(void);
	~DQSegDefTable(void);
	void addRow(const DQSegDef& df);
	void check(const char* title) const;
    private:
	DBTypes::varchar   defid;
	DBTypes::varchar   procid;
	DBTypes::varchar   run;
	DBTypes::varchar   ifos;
	DBTypes::varchar   name;
	DBTypes::integer_4 version;
	DBTypes::varchar   comment;
    };

    /**  The SegMapTable class is a meta-database interface producing a
      *  segment map table.
      */
    class DQSegMapTable : public xsil::MetaTable {
    public:
	DQSegMapTable(void);
	~DQSegMapTable(void);
	void addRow(const std::string& segid, const std::string& defid, 
		    const std::string& procid);
	void addRow(const Segment& s, const DQSegDef& d);
	void check(const char* title) const;
    private:
	DBTypes::varchar segid;
	DBTypes::varchar defid;
	DBTypes::varchar procid;
    };

    //======================================  S5 Segment definer inline methods
    inline bool 
    DQSegDef::operator==(const DQSegDef& x) const {
	return mIfos == x.mIfos && mName == x.mName && mVersion == x.mVersion;
    }

    inline int 
    DQSegDef::getVersion(void) const {
	return mVersion;
    }

    inline const std::string& 
    DQSegDef::refProcess(void) const {
	return mProcID;
    }

    inline const std::string& 
    DQSegDef::refDefinerID(void) const {
	return mDefID;
    }

    inline const std::string& 
    DQSegDef::refIfos(void) const {
	return mIfos;
    }

    inline const std::string& 
    DQSegDef::refName(void) const {
	return mName;
    }

    inline const std::string& 
    DQSegDef::refComment(void) const {
	return mComment;
    }

}  // namespace trig

#endif // TRIG_DQSEGTABLE_HH

