/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef _GDS_FRAMEXMITTYPE_H
#define _GDS_FRAMEXMITTYPE_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: framexmit						*/
/*                                                         		*/
/* Module Description: API for broadcasting frames			*/
/*		       implements a reliable UDP/IP broadcast for	*/
/*                     large data sets over high speed links		*/
/*                                                         		*/
/* Module Arguments: none				   		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 10Aug99  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: doc/index.html (use doc++)				*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/* Code Compilation and Runtime Specifications:				*/
/*	Code Compiled on: Ultra-Enterprise, Solaris 5.7			*/
/*	Compiler Used: egcs-1.1.2					*/
/*	Runtime environment: sparc/solaris				*/
/*                                                         		*/
/* Code Standards Conformance:						*/
/*	Code Conforms to: LIGO standards.	OK			*/
/*			  Lint.			TBD			*/
/*			  ANSI			OK			*/
/*			  POSIX			OK			*/
/*									*/
/* Known Bugs, Limitations, Caveats:					*/
/*								 	*/
/*									*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1996.			*/
/*                                                         		*/
/*                                                         		*/
/* California Institute of Technology			   		*/
/* LIGO Project MS 51-33				   		*/
/* Pasadena CA 91125					   		*/
/*                                                         		*/
/* Massachusetts Institute of Technology		   		*/
/* LIGO Project MS NW17-161				   		*/
/* Cambridge MA 01239					   		*/
/*                                                         		*/
/* LIGO Hanford Observatory				   		*/
/* P.O. Box 1970 S9-02					   		*/
/* Richland WA 99352					   		*/
/*                                                         		*/
/* LIGO Livingston Observatory		   				*/
/* 19100 LIGO Lane Rd.					   		*/
/* Livingston, LA 70754					   		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

// include files
#include <sys/types.h>
#include <netinet/in.h>
#include <inttypes.h>
#include <iosfwd>

#define PBTEASEPACKET0

namespace framexmit {


/** @name Flow control
    Constants and classes used for controling the data flow.
    @memo Flow control classes and constants
************************************************************************/

/*@{*/

   /// default IP port [7096]
   const int 		frameXmitPort = 7096;
   /// multicast time-to-live (i.e., number of hops)
   const unsigned char 	mcast_TTL = 1;
   /// priority of daemon (lower values = higher priority) [1]
   const int 		daemonPriority = 1;
   /// default number of buffers for the sender [7]
   const int		sndDefaultBuffers = 7;
   /// packet size (in bytes) [64000]
   const int		packetSize = 64000;
   /// continues packet burst (in # of packets) [4]
   const int		packetBurst = 4;
   /// minimum time interval between bursts (in usec) [1000]
   const int		packetBurstInterval = 1000;
   /// sender delay tick if there are no packets to send (in usec) [1000]
   const int		sndDelayTick = 1000;
   /// receiver delay tick if there were no packets received (in usec) [1000]
   const int		rcvDelayTick = 1000;
   /// Maximum wait time for a lost packet in original stream (in usec) [10000]
   const int            lostPacketWait = 15000; // used to be 10000
   /// maximum number of packets allowed for retransmit [200]
   const int		maximumRetransmit = 200;
   /// sender socket input buffer length (in bytes) [65536]
   const int 		sndInBuffersize = 65536;
   /// sender socket output buffer length (in bytes) [262144]
   const int 		sndOutBuffersize = 4 * 65536;
   /// receiver socket input buffer length (in bytes) [262144]
   const int 		rcvInBuffersize = 4 * 65536;
   /// receiver socket input buffer maximum length (in bytes) [1048576]
   const int 		rcvInBufferMax = 16 * 65536;
   /// receiver socket output buffer length (in bytes) [65536]
   const int 		rcvOutBuffersize = 65536;
   /// receiver packet buffer length (in number of packets) [1024]
   const int 		rcvpacketbuffersize = 1024;
   /// maximum number of retries to rebroadcast [5]
   const int 		maxRetry = 5;
   /// timeout for each retry (in usec) [250000]
   const int 		retryTimeout = 250000;
   /// maximum value a sequnce can be out of sync [5]
   const int		maxSequenceOutOfSync = 5;
   /// number of data segments used for determining the average 
   /// retransmission rate [10.1]
   const double		retransmissionAverage = 10.1;
   /// artifically introduced receiver error rate [0], only valid if 
   /// compiled with DEBUG
   const double		rcvErrorRate = 0;
#if defined(PBTEASEPACKET0)
   const unsigned int lastFrameDurationDefault = 4; // The duration of the last frame (in seconds)
   const int frameDrawRebroadcastInterval = 450; // Issue rebroadcasts to draw out an unseen frame in 125 ms intervals
   const int maxRetryDrawOut = 5;      // The maximum number of times we try to draw out a frame
   const int drawOutFrameCountMax = 5; // The maximum number of attempts to try to draw out an unseen frame
   const unsigned int maxFrameDuration = 10000; // The maximum allowable length of a frame
#endif // PBTEASEPACKET0

   /**  Parameter structure
     *  This structure echos the parameters in the framexmittypes header.
     *  To start, the parameters will be left unused, but as parameters a
     *  needed to be varied, or new parameters are added, they will be
     *  fetched from the global structure.
     */
   struct fxparameters {
      /// initialiaize to the standard values.
      fxparameters(void);
      void display(std::ostream& out) const;
      /**  Modify the contents of the parameter structure as specified by a resource file in 
        *  \c \$HOME/.framexmitrc. The file format is as indicated in read_parameters().
	*/
      void init(void);

      /**  Read parameters from the specified file. Each line of the parameter file must contain
        *  a valid parameter name followed by one or more spaces and the parameter value.
	*/
      void read_parameters(const std::string& filename);
      
      bool set_parameter(const std::string& name, double value);

      /// default IP port [7096]
      int      	frameXmitPort;
      /// multicast time-to-live (i.e., number of hops)
      unsigned char 	mcast_TTL;
      /// priority of daemon (lower values = higher priority) [1]
      int      	daemonPriority;
      /// default number of buffers for the sender [7]
      int      	sndDefaultBuffers;
      /// packet size (in bytes) [64000]
      int      	packetSize;
      /// continues packet burst (in # of packets) [4]
      int      	packetBurst;
      /// minimum time interval between bursts (in usec) [1000]
      int      	packetBurstInterval;
      /// sender delay tick if there are no packets to send (in usec) [1000]
      int      	sndDelayTick;
      /// receiver delay tick if there were no packets received (in usec) [1000]
      int      	rcvDelayTick;
      /// Maximum wait time for a lost packet in original stream (in usec) [10000]
      int       lostPacketWait; // used to be 10000
      /// maximum number of packets allowed for retransmit [200]
      int      	maximumRetransmit;
      /// sender socket input buffer length (in bytes) [65536]
      int      	sndInBuffersize;
      /// sender socket output buffer length (in bytes) [262144]
      int      	sndOutBuffersize;
      /// receiver socket input buffer length (in bytes) [262144]
      int      	rcvInBuffersize;
      /// receiver socket input buffer maximum length (in bytes) [1048576]
      int      	rcvInBufferMax;
      /// receiver socket output buffer length (in bytes) [65536]
      int      	rcvOutBuffersize;
      /// receiver packet buffer length (in number of packets) [1024]
      int      	rcvpacketbuffersize;
      /// maximum number of retries to rebroadcast [5]
      int      	maxRetry;
      /// timeout for each retry (in usec) [250000]
      int      	retryTimeout;
      /// maximum value a sequnce can be out of sync [5]
      int      	maxSequenceOutOfSync;
      /// number of data segments used for determining the average 
      /// retransmission rate [10.1]
      double   	retransmissionAverage;
      /// artifically introduced receiver error rate [0], only valid if 
      /// compiled with DEBUG
      double    rcvErrorRate;
      /// Maximum rebroadcasts per packet per epoch
      int       sndMaxPacketRebroadcast;
      ///  Length of rebroadcast epoch (usec)
      int       sndRebroadcastEpoch;
#if defined(PBTEASEPACKET0)
      /// Set the following to a non-zero number to tease out packet 0 of an unseen frame
      int teasePacket0;
      /// The duration of the last frame (in seconds)
      unsigned int lastFrameDurationDefault;
      /// Issue rebroadcasts to draw out an unseen frame in 
      int frameDrawRebroadcastInterval;
      /// The maximum number of attempts to try to draw out an unseen frame
      int drawOutFrameCountMax;
      /// The maximum allowable length of a frame (10000 s)
      unsigned int maxFrameDuration;
      /// The maximum number of times we try to draw out a frame
      int maxRetryDrawOut;
#endif // PBTEASEPACKET0
      ///  Minimum time between buffer broadcasts.
      int       minInterBufferTime;
      ///  Maximum buffer size (in packets) for automatic rebroadcast.
      int       nRepeatPktMax;
      ///  Maximum number of packets to be automatically rebroadcast.
      int       nRepeatPktNum;
   };

   extern fxparameters par;

   ///  define timestamp type in microseconds.
   typedef int64_t timestamp_type;

   /**  Get the current time in microseconds.
     *  \brief Get current time.
     *  \return Time stamp in microseconds.
     */
   timestamp_type get_timestamp(void);

   /**  Sleep for the specified number of microseconds. Note that the maximum
     *  delay assuming 32-bit integers is ~2000s.
     *  \brief Wait the specified number of microseconds.
     *  \param delay Number of microseconds to wait
     *  \return Zero on success, <0 indicates error as specified by errno.
     */
   int micro_delay(int delay);

   /// broadcast packet type [123]
   const int		PKT_BROADCAST = 123;
   /// rebroadcast packet type [124]
   const int		PKT_REBROADCAST = 124;
   /// retransmit request packet [125]
   const int		PKT_REQUEST_RETRANSMIT = 125;

   /// Packet header (information is stored in network byteorder)
   struct packetHeader {
      /// packet type
      int32_t		pktType;
      /// length of payload in bytes
      int32_t		pktLen;
      /// sequence number
      uint32_t		seq;
      /// packet number
      int32_t		pktNum;
      /// total number of packets in sequence
      int32_t		pktTotal;
      /// checksum of packet (currently not used)
      uint32_t		checksum;
      /// GPS time stamp of data
      uint32_t		timestamp;
      /// time length of data
      uint32_t		duration;
      /// swap byte-order in header from host to network
      inline void hton();
      /// swap byte-order in header from network to host
      inline void ntoh();
   };

   /// Standard data packet
   struct packet {
      /// packet header
      packetHeader	header;
      /// packet payload
      char		payload[packetSize];
      /// swap byte-order in header from host to network
      inline void hton();
      /// swap byte-order in header from network to host
      inline void ntoh();
   };

   /// Retransmit request packet
   struct retransmitpacket {
      /// packet header
      packetHeader	header;
      /// packet payload
      int32_t		pktResend[packetSize / sizeof (int32_t)];
      /// swap byte-order in header from host to network
      inline void hton();
      /// swap byte-order in header from network to host
      inline void ntoh();
   };


/** Auto pointer for a packet.
    Packets are ordered by sequence number and packet number.
    @ingroup IO_frxmit
 ************************************************************************/
   class auto_pkt_ptr {
   private:
      packet* ptr;
      mutable bool owns;
   public:
   
      /**  Construct a new autopointer and assign it to the specified 
        *  packet address. The auto-pointer takes ownership if the pointer
	*  is valid.
	*  @brief Constructor.
	*/
      explicit auto_pkt_ptr (packet* p = 0) : ptr(p), owns(p) {
      }

      /**  Construct a new autopointer and assign the argument pointer to
        *  this. The ownership of the packet is transfered from the argument 
	*  auto-pointer to this auto-pointer if it was owned by the argument.
        *  @brief Assignment operator.
        */
      auto_pkt_ptr (const auto_pkt_ptr& a) : ptr(a.ptr), owns(a.owns) {
         a.owns = 0;
      }

      /**  Assign the packet to this auto-pointer. The ownership of the 
        *  packet is transfered from the argument auto-pointer to this
        *  auto-pointer if it was owned by the argument.
        *  @brief Assignment operator.
        */
      auto_pkt_ptr& operator= (const auto_pkt_ptr& a) {
         if (&a != this) {
            if (owns)
               delete ptr;
            owns = a.owns;
            ptr = a.ptr;
            a.owns = 0;
         }
         return *this;
      }

      /** Destroy packet auto-pointer. The packet is deleted if it is owned.
        */
      ~auto_pkt_ptr() {
         if (owns)
            delete ptr;
      }

      /**  Dereference pointer
       */
      packet& operator*() const { 
         return *ptr; }

      /**  Dereference pointer.
        */
      packet* operator->() const { 
         return ptr; }

      /**  Get the pointer value.
        */
      packet* get() const  { 
         return ptr; }

      /** Release packet without deleting.
        */
      packet* release() const { 
         owns = false; 
         return ptr; }
   
      /**  Equal comparison for auto_pkt_ptr.
        */
      bool operator== (const auto_pkt_ptr& a) const {
         if (!owns || !a.owns) {
            return false;
         }
         return ((ptr->header.seq == a.ptr->header.seq) && 
                (ptr->header.pktNum == a.ptr->header.pktNum));
      }

      /**  Less-than comparison for auto_pkt_ptr.
        */
      bool operator< (const auto_pkt_ptr& a) const {
         if (!owns || !a.owns) {
            return owns;
         }
         return ((ptr->header.seq < a.ptr->header.seq) || 
                ((ptr->header.seq == a.ptr->header.seq) &&
                (ptr->header.pktNum < a.ptr->header.pktNum)));
      }
   };

   inline void packetHeader::hton()
   {
      pktType = htonl (pktType);
      pktLen = htonl (pktLen);
      seq = htonl (seq);
      pktNum = htonl (pktNum);
      pktTotal = htonl (pktTotal);
      checksum = htonl (checksum);
      timestamp = htonl (timestamp);
      duration = htonl (duration);
   }

   inline void packetHeader::ntoh()
   {
      pktType = ntohl (pktType);
      pktLen = ntohl (pktLen);
      seq = ntohl (seq);
      pktNum = ntohl (pktNum);
      pktTotal = ntohl (pktTotal);
      checksum = ntohl (checksum);
      timestamp = ntohl (timestamp);
      duration = ntohl (duration);
   }

   inline void packet::hton()
   {
      // payload not swapped
      header.hton();
   }

   inline void packet::ntoh()
   {
      header.ntoh();
      // payload not swapped
   }

   inline void retransmitpacket::hton()
   {
      for (int i = 0; i < header.pktLen/(int)sizeof(int32_t); ++i) 
         pktResend[i] = htonl (pktResend[i]);
      header.hton();
   }

   inline void retransmitpacket::ntoh()
   {
      header.ntoh();
      for (int i = 0; i < header.pktLen/(int)sizeof(int32_t); ++i) 
         pktResend[i] = ntohl (pktResend[i]);
   }

/*@}*/


}

#endif // _GDS_FRAMEXMITTYPE_H
