#ifndef SUSPENSIONMON_HH
#define SUSPENSIONMON_HH

#include <fstream>
#include <map>

#include "DatEnv.hh"
#include "TSeries.hh"
#include "Histogram1.hh"
#include "MonServer.hh"
#include "TrigPar.hh"
#include <deque>

#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/suspensionMon/suspensionMon.hh 6385 2010-12-10 23:00:42Z john.zweizig@LIGO.ORG $"
#define PIDTITLE "Glitch Monitor"
#include "ProcIdent.hh"
#include "TrigClient.hh"
#include "TrigRslt.hh"
#include "Time.hh"
#include "OperStateCondList.hh"
#include "Trend.hh"

#include "chInterface.hh"
#include "chConstructs.hh"
#include "susConstructs.hh"

/** @name suspensionMon Monitor
  * The suspensionMon monitor is the first in a series of monitors to use the
  * \Ref{chInterface} class for a standard channel interface.  Included in
  * this documentation are a few channel interfacing templates.  It is the
  * hope of the author that a collection of templates can be gathered and
  * used among a variety of monitors.
  * @memo a monitor which uses a standardized channel interface
  * @authors Rauha Rahkola and Brian Stubbs
  * @version 0.1 October 15, 2003
  ***************************************************************************/
//@{

// iterator for chInterface objects
typedef std::vector<channel::chInterface>::iterator CHiter;
// typedef std::vector<channel::chInterface>::const_iterator cCHiter;

/** This monitor triggers on glitches in filtered data based on absolute
  * thresholds.  The any data values (thresholds, RMS) are given in counts, and
  * times are recorded in GPS and UTC for each event.  A log file is also
  * available which gives a regular update on the frequency of the glitches in
  * each channel.
  *
  * {\large {\bf Using the suspensionMon monitor}}
  *
  * According to the help info:\\
  * {\bf Usage}: #suspensionMon [-conf <file>] [-osc <osc_file>] #
  *              #[-infile <frame_file>] [+toc | -toc] [-h | --help] #\\
  *
  * suspensionMon looks for glitches with absolute thresholds (in counts).
  * Options include:
  * \begin{tabular}{ll}
  *   #-conf   <file>#        & -- The configuration file to use as input\\
  *   #-osc    <osc_file>#    & -- Use an Operating State Condition
  *                                configuration file \\
  *   #-infile <frame_file>#  & -- Used when running offline. Frame files
  *        can include a wildcard character, which must be escaped, as in:
  *        #-infile <frame_path>/H-R-\*.gwf# \\
  *   #+toc# or #-toc#        & -- access frames using TOC (faster) or not \\
  *   #-h# or #--help#        & -- Generates this help text and exits \\
  *   #-debug [<debug_level>]#& -- To be used only for debugging purposes\\
  * \end{tabular}
  * 
  * {\bf Operating State Conditions- }
  * Currently only one Operating State Condition (OSC) can be used each time
  * the monitor is run.  It is not necessary to set or use an OSC to run the
  * monitor, but if one is desired, both the above command-line parameter and a
  * parameter in the Configuration file (discussed below) must be set, as well
  * as identifying the channels which will be using the OSC.
  *
  * {\bf Applications of suspensionMon- }
  * The goal of suspensionMon is to be a monitor which provides timestamps for
  * which the gravity-wave data cannot be trusted.  These times are singled out
  * as possible correlations with glitches in environmental or other diagnostic
  * channels.  
  *
  *
  * {\large {\bf The Configuration File}}
  * 
  * A configuration file consists of two sections: run-time parameters and
  * channel-specific parameters.  The `\#' character signifies the start of a
  * comment, and is not used as part of the configuration file.  One
  * self-documented example might look like the following:
  * \begin{verbatim}
 #====================================
 # file path to the text output data file
 Data_File_Prefix : /home/bstubbs/suspensionMon/test_data
 # file path to the log file
 Log_File_Prefix : /home/bstubbs/suspensionMon/test_log
 # length of data to process in seconds (0 for infinity)
 Total_Runtime    : 0
 # of bins which composes each histogram
 Number_Bins      : 20
 # Overall allows/disallows triggers to be sent
 Trigger_ON/OFF   : 1
 ##### This first one-liner corresponds to a 100Hz HighPass Filter
 #Channel H2:LSC-MICH_CTRL H2:Both_arms_locked "-7.9577 13.7832 -7.9577 -13.7832 -15.9155 0" "0 0 0 0 0 0"   H2:LSC-MICH_CTRL_HP100Hz
 Channel H2:LSC-MICH_CTRL 0
 ### filter:     Name                            Poles                                   Zeroes          Gain
 ### --------------------------------------------------------------------------------------------------------
   Filter    H2:LSC-MICH_CTRL_HP100Hz "-7.9577 13.7832 -7.9577 -13.7832 -15.9155 0" "0 0 0 0 0 0"
 ### glitch:     Name        Threshold(counts) Time_constant
 ### -------------------------------------------------------
   Glitch    suspensionMon           1.0             2.0
 ### stats:      Name            Type  Cumulative?    Refresh Initial_Val
 ### --------------------------------------------------------------------
   Statistic MICH_CTRL-Mean      Mean    0
 ### results:    Name            Type  Cumulative?    Refresh  Output_mask
 ### ---------------------------------------------------------------------
   Result    MICH_CTRL-updates   Log     0               20      4
   Result    MICH_CTRL-events    Events  1                       2
 \end{verbatim}
  *
  * Some notes:\begin{itemize}
  *  \item Using a Wait Time of `0' will cause suspensionMon to make a histogram
  *    of the filtered data.  This is useful when determining how many datapts
  *    actually fall within the defined thresholds.  No data file will be
  *    produced when Wait Time is set to `0'.
  *  \item The log file from a previous run of the monitor will create a valid
  *    configuration file for easy reference and re-running over off-line data.
  *  \item {\bf Please note that the filtering scheme has changed since 
  *    5/5/2002.  The simple `LowPass', `HighPass', `BandPass' scheme has been
  *    replaced by a poles, zeroes, gain scheme.}
  *  \item A channel may receive no filtering by replacing the poles and zeroes
  *    fields with `0'.
  *  \item The Trigger Comment parameter will show up in either/both the data
  *    file or the trigger as it is sent out to the `GDS Triggers' table in the
  *    database.  In the latter case, it appears in the `data' field.
  * \end{itemize}
  *
  *
  * {\large {\bf Output from suspensionMon}}
  *
  * {\bf Log File Output- }
  * suspensionMon will always write out a log file (which can then be re-used as a
  * configuration file, see above).  The log file currently contains the
  * configuration parameters which were used to produce the output, followd by
  * a series of snapshots of the monitor's progress.  These snapshots contain
  * multiple lines per channel, as seen in the following example:
  * \begin{verbatim}
 ###====================================
 ### Monitor Start = 693644209
 ###------------------------------------
 ### Log_Interval: 0     GPS_start: 693644209    Interval_Duration: 1000
 ### Channel Statistics
 ### Channel | Filtered | Threshold | # Triggers  | # Triggers | Current  |  Dead Time  | Dead Time | Filtered | # Single-
 ###  Name   |   RMS    |           | in Interval |    Total   | Run Time | in Interval |   Total   | Kurtosis | Bin Trigs
 #0 H2:IOO-MC_F  1435.06 Totals  3163    3163    1000    140.333 140.333 0.0220399       136     
 #0 H2:IOO-MC_F  1435.06 18529.7 0       0       1000    0       0       0.0220399       0
 #0 H2:IOO-MC_F  1435.06 17724.1 0       0       1000    0       0       0.0220399       0
 #0 H2:IOO-MC_F  1435.06 16918.4 0       0       1000    0       0       0.0220399       0
 ...
 \end{verbatim}
  * 
  * The section of lines for each channel describe a histogram of trigger
  * threholds and the number of triggers fitting between that threshold and
  * the next one.  The first line for each channel gives the totals for each
  * histogram.  The columns are described below.
  *
  * Each line is "commented out" so that the log file can be re-used as a
  * configuration file.  The lines are preceded by the current log interval to
  * be distinguishable, say, by an `awk' or `grep' script.
  *
  * Description of columns:
  * \begin{tabular}{ll}
  *   #Filtered RMS#       & RMS of filtered data for the interval\\
  *   #Filtered Kurtosis#  & Kurtosis of filtered data for the interval\\
  *   #Threshold#          & Threshold level (in filtered ADC counts)\\
  *   \# #Triggers ...#    & Number of triggers with max amplitudes falling
  *          above the threshold (but not above the next highest threshold) \\
  *   #Dead Time ...#      & Accumulated time of glitch durations (in seconds)\\
  *   \# #Single-Bin Trigs#& Number of triggers with durations less than or
  *          equal to (sample rate * order of filter) for the interval \\
  *   #Current Run Time#   & Number of seconds which the monitor has processed\\
  * \end{tabular}
  *
  * {\bf Data File Output/Trigger Generation- }
  * Depending on the #Trigger_ON/OFF# parameter in the configuration file,
  * suspensionMon may produce individual triggers.  These triggers may be sent to
  * a datafile or to the meta-database, but in either case will include the same
  * information.  Below is a snippet from a sample data file, followed by a
  * description of the columns.
  * \begin{verbatim}
 ### Trigger Statistics:
 ### Event Start | Channel | Event Start | Duration  | Max Glitch Size | RMS during | Trigger
 ###    (UTC)    |   Name  |    (GPS)    | (seconds) |   (in counts)   |   Glitch   | Comment
 01.12.29 6:56:36 H0:PEM-EX_TILTT 693644209.1054688 0.00390625 10.7039 10.7039 H0:PEM-EX_TILTT_HP1Hz
 01.12.29 7:08:30 H0:PEM-HAM7_ACCY 693644923.8007812 0.000488281 774.843 774.843 H0:PEM-HAM7_ACCY_HP1Hz
 01.12.29 7:08:32 H0:PEM-BSC8_ACCX 693644925.0883789 0.000488281 788.026 788.026 H0:PEM-BSC8_ACCX_HP1Hz
 \end{verbatim}
  * 
  * \begin{tabular}{ll}
  *   #Event Start ...#    & Start of trigger (Either in UTC or GPS times) \\
  *   #Channel Name   #    & Name of the channel triggered \\
  *   #Duration (seconds)# & Duration for which filtered data was in a
  *    "triggered state", as defined by the #Wait Time# parameter -- not
  *    necessarily all filtered datapts were outside the thresholds \\
  *   #Max Glitch Size#    & Size of the largest deviation from the mean (in
  *                          filtered ADC counts) \\
  *   #RMS during Glitch#  & RMS deviation of all datapts within the trigger;
  *    also used as `significance' for meta-database trigger \\
  *   #Trigger Comment#    & Also used as `subtype' for meta-database trigger\\
  * \end{tabular}
  *
  * @memo A glitch monitor using absolute thresholds
  * @author Rauha Rahkola, Robert Schofield
  * @version 0.2 (Last modified Tue Apr 2, 2002)
  */
class suspensionMon : public DatEnv, MonServer
{
private:
  /// Provides conditions which must be satisfied to perform analysis
  OperStateCondList      mOSC;
  /// Communicates with Trigger Manager (reference gets passed to each susEvent)
  TrigClient    mTrig;
  /// Returns the initial string common to both OSC parameter strings
  std::string getCommonString(const std::string& s1, const std::string& s2);

  /// monitor duration
  int fEndRuntime;

  /// processData loop iterator
  int dataLoopNum;

  /// default time interval for updating log file
  Interval fLogInterval;
  /// mapping of result names to output masks
  std::map<std::string,unsigned int> mOutMap;
  /// a list of statistics names to use
  std::list<std::string> statList;
  /// a list of glitch names to use
  std::list<std::string> glitchList;
  /// a list of result names to use
  std::list<std::string> resultList;

  /// number of bins in each histogram
  int fBinNumber;

  /// total number of loops executed since monitor started
  long fCount,fLogCount;

  /// initialization time, current time, time of last log entry
  Time fInitTime, fCurrentTime, fLogStartTime;

  /// Configuration file parameters dictionary
  TrigPar mDict;

  /// log and data file name
  std::string fLogPath,fDataPath;

  /// file stream for log and data files
  std::ofstream fLogFile,fDataFile;

  /// for trending
  Trend trendOne, trendTwo;

  /// flags if Log or Data files have been initialized
  bool fLogInit, fDataInit;

  /// 0:off, 1:on
  bool fTriggerON;

  /// read config file and initialize channel members
  void ReadConfig(const std::string& config);

  /// check channel for conspicuous datapts
  //  void CheckEvent( channel::chInterface *ch );

  /// check an individual datapt for being above threshold
  //  bool CheckEvent( const double& datapt, const Time& bintime,
  //		       channel::chInterface *ch );

  /// output event info to data file and/or send trigger
  //  void OutputEvent(channel::chInterface* ch);

  /// output log file information
  void OutputLog(bool check);

  /// reset variables
  void Reset(void);

  /// return mean value of filtered time series
  const float GetMean(const channel::chInterface* ch) const;

  /// return RMS of filtered time series
  const float GetRMS(const int sfreq) const;

  /// Check that an argument follows an option
  bool chkNextArg(int i, const char *argv[], int argc);

  /// Print Usage info
  int  Usage(const char *progName);

  /// Print Help info
  int  Help();

public:
  /// Information for individual channels
  std::vector<channel::chInterface> fChannels;
  
  /** Parses command-line arguments, initializes and allocates configuration
    * parameters, reads the configuration file, and starts a log file.\\
    */
  suspensionMon(int,const char**);
  
  /** Destroys the suspensionMon monitor: watch out! :)
    */
  ~suspensionMon(void);
  
  /** Processes a second of data according to whether or not the OSC for that
    * channel is satisfied.  If so, it checks for events within each channel's
    * data.  It also updates the log file, if necessary.
    */
  void ProcessData(void);
  
  /** Attention is used with the DMT Viewer tool and is possibly not functional
    * yet.
    */
  void Attention(void);
   
};

//@}
#endif
