#ifndef CHCONSTRUCTS_HH
#define CHCONSTRUCTS_HH
#include <vector>
#include <map>
#include <string>
#include <list>
#include <iostream>
#include "Time.hh"
#include "TSeries.hh"
#include "FSpectrum.hh"
#include "IIRFilter.hh"
#include "Histogram1.hh"

namespace channel {
/** The \Ref{dataType} enumeration and \Ref{dataPtr} union are closely related.
  * Both types can be found in the \Ref{glitchResult} and \Ref{statResult}
  * structures as a means of pointing results to some particular data, be it
  * \Ref{TSeries}, \Ref{FSeries}, \Ref{FSpectrum}, or \Ref{Histogram1}.
  * The \Ref{dataType} specifies which element of the \Ref{dataPtr} union
  * should be referenced.

  * The results, in turn, can choose to serve the data to a \Ref{MonClient}, or
  * output the data to file in LIGO_LW format. An example of each can be found
  * in the \Ref{ResultConstruct} documentation.  If no data is present, the
  * \Ref{dataType} element should be #dUnknown#.
  * @memo datatype codes for \Ref{dataPtr} union
  */
enum dataType {
  /** there is no data in \Ref{dataPtr}, or the \Ref{datPtr} data is of unknown
    * type
    */
  dUnknown,
  /// the \Ref{dataPtr} data is of type TSeries
  dTSeries,
  /// the \Ref{dataPtr} data is of type FSeries
  dFSeries,
  /// the \Ref{dataPtr} data is of type FSpectrum
  dFSpectrum,
  /// the \Ref{dataPtr} data is of type Histogram1
  dHistogram1
};

/** The \Ref{dataType} enumeration and \Ref{dataPtr} union are closely related.
  * Both types can be found in the \Ref{glitchResult} and \Ref{statResult}
  * structures as a means of pointing results to some particular data, be it
  * \Ref{TSeries}, \Ref{FSeries}, \Ref{FSpectrum}, or \Ref{Histogram1}.
  * The \Ref{dataType} specifies which element of the \Ref{dataPtr} union
  * should be referenced.

  * The results, in turn, can choose to serve the data to a \Ref{MonClient}, or
  * output the data to file in LIGO_LW format. An example of each can be found
  * in the \Ref{ResultConstruct} documentation.  If no data is present, the
  * \Ref{dataType} element should be #dUnknown#.
  * @memo pointer for data which can be referenced in a \Ref{ResultConstruct}
  */
union dataPtr {
  /// a pointer to unknown datatype
  const void*       dV;
  /// a pointer to data of type TSeries
  const TSeries*    dTS;
  /// a pointer to data of type FSeries
  const FSeries*    dFS;
  /// a pointer to data of type FSpectrum
  const FSpectrum*  dSP;
  /// a pointer to data of type Histogram1
  const Histogram1* dH1;
};

/** update type -- identifies whether construct is meant to be refreshed
  * on occasion.  Most functions can override this flag if desired, thus
  * updating continuous, or even unknown, constructs.
  */
enum updateType {
  /// construct is of unknown update type
  uUnknown,
  /// construct should not normally be updated
  uContinuous,
  /// construct should normally be updated
  uRefreshed
};

/** glitch state -- used to pick-up the state of the glitch where it's
  * been left off
  */
enum glitchState {
  /// no event has been detected yet, or glitch has been recently initialized
  gNoEvent,
  /// currently in an active event
  gInEvent,
  /// event has only recently finished, and may become active again soon
  gWaiting,
  /// event has been finished for some time and is ready to be published
  gDone
};

/// glitch result structure -- basically duplicates the gds_trigger structure
struct glitchResult {
  /// suggested to be the name of the GlitchConstruct
  std::string         name;
  /// suggested to reflect the (filtered?) TSeries used to produce the glitch
  std::string         comment;
  /// suggested to be the actual channel name of the original data
  std::string         channel;
  /// suggested to be the units of the glitch amplitude and/or significance
  std::string         units;
  /// suggested to be the amplitude of the resulting glitch
  double              amplitude;
  /// suggested to be the significance of the resulting glitch
  double              significance;
  /// suggested to be the start-time of the resulting glitch
  Time                startTime;
  /// suggested to be the duration of the resulting glitch
  Interval            duration;
  /// suggested to be the (central?) frequency of the resulting glitch
  double              frequency;
  /** suggested to be the bandwidth of the resulting glitch -- possibly could
   * be the cut-off frequency of the filter
   */
  double              bandwidth;
  /** suggested to reflect the priority the glitch should take: 'info (0)',
   * 'warning (1)', 'emergency/error (2)', 'critical/severe (3)'
   */
  unsigned int        priority;
  /** suggested to reflect the intended disposition of the glitch:
   * 'none (0)', 'metadatabase/metaDB (1)', 'control room/alarm (2)'
   */
  unsigned int        disposition;
  /// required to be the type of data contained in #data#
  dataType            type;
  /// a pointer to data which can be served to a MonClient
  dataPtr             data;
  /// can be changed, e.g. by a ResultsConstruct, from 'kDone' to 'kNoEvent'
  mutable glitchState state;
};

/// state of a statistics calculation
enum statState {
  /** no calculations have been made yet, or statistics has been recently
    * initialized
    */
  sNewStat,
  /// calculations are underway, but have not been finished
  sInStat,
  /// calculations have been finished and are ready to be published
  sDoneStat
};

/** statistics result structure -- almost duplicates \Ref{glitchResult}
  * structure
  */
struct statResult {
  /// usually a generic name, like "RMS"
  std::string      name;
  /// usually the filterID
  std::string      comment;
  /// usually the channel name
  std::string      channel;
  /// usually the units for the final and/or preliminary values
  std::string      units;
  /// for preliminary values
  double           prelim;
  /// for final values
  double           actual;
  /// big N, used in many statistics
  long             statcount;
  /// sometimes used in statistics
  Time             startTime;
  /// sometimes used in statistics
  Interval         duration;
  /// identifies the data in #data#
  dataType         type;
  /// pointer to data which can be served to
  dataPtr          data;
  /// what state the calculation is in
  mutable statState state;
};


/** This class, unlike the *Construct classes, is currently not intended to
  * have any derived classes.  If derived classes are wished for in the future,
  * it will develop some derivable properties.
  * @memo Filter information class for implementing IIRFilters.
  * @author Rauha Rahkola
  * @version 0.1 May 29th, 2002
  ***************************************************************************/
class FilterStatus {
private:
  /// filter ID
  std::string mName;

  unsigned int mSampleRate;
  std::vector<dComplex> mPoles;
  std::vector<dComplex> mZeroes;
  double mGain;

  /// Pointer to the IIR Filter class used by FilterStatus
  IIRFilter* mFilter;
  /// Pointer to the TSeriese data resulting from applying the filter
  TSeries* mTS;
  /// a time indicating the most recent application of filter
  Time mUpdate;

  /// a flag whether the filter has been initialized
  bool mInit;
  /// debug flag
  int mDebug;

  /** FilterBank purposes only: if > 0, this refers to the FilterStatus which
    * should be used as the input TSeries
    */
  unsigned int mInFiltID;

public:
  /*
   * Clients/Derived filters need to be able to use these functions
   */
  /** FilterStatus constructor-- users should initialize FilterStatus with
    * setupFilter(), once the sample rate for the channel is known.
    * @param name   the name of the filter
    * @param debug  (debugging) the verbosity level of debug output
    */
  explicit FilterStatus(const std::string& name, int debug=0): 
    mName(name), mSampleRate(0), mGain(1.0), mFilter(0), mTS(0), mUpdate(0),
    mInit(0), mDebug(debug), mInFiltID(0) {
    if (mDebug > 2) std::cout <<"Creating FilterStatus -- "<<mName<<std::endl;
    if (mDebug > 3) std::cout <<"   (FilterStatus "<<mName<<") "<<this<<std::endl;
  }

  /// FilterStatus copy constructor
  FilterStatus( const FilterStatus& filter );

  /// FilterStatus destructor
  ~FilterStatus();

  /// FilterStatus assignment operator
  FilterStatus& operator =( const FilterStatus& rhs );

  /* FilterStatus modifiers
   */
  /// sets which index of a FilterBank to use as input
  void setUseFilterID(unsigned int filtID) { mInFiltID = filtID; }

  /// sets the sample rate of the data coming in
  void setSampleRate(unsigned int rate) { mSampleRate = rate; }

  /* FilterStatus accessors
   */
  /// returns the name of the filter
  std::string getName(void) const { return mName; }

  /// returns the filtered data
  const TSeries *getData(void) const { return mTS; }

  /** returns the index of the FilterBank to be used as input.  '0' refers to
    * the raw timeseries data.  This allows for multiple filters to be used
    * in succession on a particular channel.
    * @return the FilterBank item to be used for the input TSeries
    */
  int getUseFilterID(void) const { return mInFiltID; }

  /* miscellaneous FilterStatus functions
   */
  /** a function to error-check whether a filter's already been applied or not
    * @param last supposedly the end time of the TSeries to be used
    * @return true if the filter has been applied since the given time
    */
  bool isUpdated(const Time& last) const {
    if (mDebug > 1)
      std::cout <<"Comparing times: "<< last.getS()<<" vs "<< mUpdate.getS()<<std::endl;
    if (mUpdate >= last) return true; else return false;
  }

  /// applies an IIR Filter to the given TSeries.
  const TSeries* apply( const TSeries* ts );

  /** converts filter order & frequency to butterworth poles/zeroes in order to
    * be backwards compatible with glitchMon-type filters
    * -- THIS FUNCTION IS NON-FUNCTIONAL YET (6/4/2002)
    * @param type     indicates the type of filter: 0=none, 1=lowpass,
    *                 2=highpass, 3=bandpass (needs two cut-off frequencies)
    * @param order    how many poles/zeroes should be used
    * @param cutoff_1 (used by lowpass, highpass, and bandpass filters) the
    *                 first cut-off frequency 
    * @param cutoff_2 (only used by bandpass filters) the second cut-off
    *                 frequency
    */
  void convertButter( int type, int order, double cutoff_1, 
		      double cutoff_2=0.0 );

  /** the function which actually creates the IIRFilter class.  This is
    * necessary because usually the client will want to make sure the
    * sampleRates are correct before creating the IIRFilter.  Note that
    * #setupFilter()# must be called even if the FilterStatus is a "raw copy"
    * of the original data. Poles and zeroes are given in complex rad/sec. It
    * is safe to use this more than once to check if the filter has been setup
    * correctly (or use isInitialized() ).
    * @param filtID the filter ID (the index in a FilterBank vector) of the 
    *               FilterStatus to be used as input
    * @param poles  a vector of complex poles
    * @param zeroes a vector of complex zeroes
    * @param gain   the gain setting in the filter.
    * @return whether the IIRFilter has been created successfully or not
    */
  bool setupFilter(unsigned int filtID,
		   std::vector<dComplex> *poles=NULL,
		   std::vector<dComplex> *zeroes=NULL, double gain=0.0);

  /// checks whether the FilterStatus object has been setup correctly.
  bool isInitialized() { return mInit; }
}; // end of FilterStatus class declaration


/** The chConstruct class is an abstract base class from which GlitchConstruct,
  * StatisticsConstruct, and ResultConstruct are derived.  Includes basic
  * member elements & functions, including clone() and mName.
  * @memo A base class for channel constructs
  * @author Rauha Rahkola
  * @version 0.1 May 29th, 2002
  ***************************************************************************/
class chConstruct {
private:
  /// name of the construct
  std::string mName;

protected:
  /// the debug level
  unsigned int mDebug;

  /// the initialization status
  mutable bool mInit;

public:
  /* *********************
   * * No constructors needed since this is an abstract class
   * ********************/
  virtual chConstruct *clone(void) const = 0;

  virtual ~chConstruct(void) {
    if (mDebug > 3) std::cout <<"Destroying chConstruct "<<mName<<std::endl;
  }

  /* *********************
   * * chConstruct modifiers
   * ********************/
  /// sets the name of the chConstruct
  void setName( std::string constID ) { mName = constID; }

  /* ********************
   * * chConstruct accessors
   * ********************/
  /// gets the name of the chConstruct
  std::string getName( void ) const { return mName; }

  /* ********************
   * * common chConstruct functions
   * ********************/
  /// checks whether the chConstruct has been setup correctly
  virtual bool isInitialized( void ) const {
    if (mName.empty()) return false;
    else return true;
  }
};


/** To make use of the chInterface, you can add GlitchConstruct instances
  * to a channel.  This is meant to separate the glitch-finding scheme from
  * the overhead of applying it to each channel.
  * @memo A base class for constructing glitch tests.
  * @author Rauha Rahkola
  * @version 0.1 May 29th, 2002
  ***************************************************************************/
class GlitchConstruct: public chConstruct {
private:
  /*
   * These members are implicit; derived glitches don't need access to these
   */
  /// threshold (event above this level always triggered)
  float mThreshold;

  /// dead time to wait for another glitch
  double mDeadTime;

protected:
  /*
   * All derived glitches need these members, but clients shouldn't see them
   */
  /// working amplitude of the glitch
  double tmpAmp;
  /// working significance of the glitch
  double tmpSig;

  /*
   * Not sure about these next members
   */
  /// working start time of a glitch
  Time tmpStart;
  /// working interval of a glitch (in ADC counts)
  long long tmpDur;

  //  /// number of event statistics calculated;
  //  int fEventCount, fWaitCount;

  /// a linked list of glitch results, if more than one
  std::list<glitchResult> results;

public:
  /*
   * Clients/Derived glitches need to be able to use these functions
   */
  /** GlitchConstruct default constructor. Uses the default algorithm of
    * detecting glitches-- measuring rms deviation from the mean and
    * comparing vs. absolute thresholds
    */
  explicit GlitchConstruct( const std::string& glitchID="", double thresh=0.0,
			    double wait=0.0, unsigned int debug=0 );

  /// GlitchConstruct copy constructor
  GlitchConstruct( const GlitchConstruct& glitch, unsigned int debug=0 );

  /// GlitchConstruct destructor
  virtual ~GlitchConstruct();

  /// GlitchConstruct assignment operator
  virtual GlitchConstruct& operator =( const GlitchConstruct& rhs );

  /// GlitchConstruct "clone" method, to allocate new GlitchConstructs
  GlitchConstruct *clone( void ) const;

  /* GlitchConstruct modifiers
   */
  /// sets the threshold level
  void setThreshold( double threshold ) { mThreshold = threshold; }

  /// sets the wait time
  void setWaitTime( double wait ) { mDeadTime = wait; }

  /* GlitchConstruct accessors
   */
  /// returns the threshold value
  double getThreshold( void ) const { return mThreshold; }

  /// returns the wait time
  double getWaitTime( void ) const { return mDeadTime; }

  /// returns the list of constructed results
   std::list<channel::glitchResult> const* getResults( void ) const { return  &results; }


  /* GlitchConstruct virtual functions
   */
  /** dumps formatted output of (one or more) glitchResult structures
    * @param out   the ostream to direct output to
    * @param total the number of glitchResults to dump, or '0' for all of them
    */
  virtual void dumpResults( std::ostream& out, unsigned int total=0 );

  /** checks for glitches
    * @param  ts       time series to check glitch in
    * @param  glitchID try matching an existing glitchResult
    * @return doubly-linked list of glitchResults (may be empty if no glitch is
    *  found)
    */
  virtual std::list<glitchResult>* checkGlitch( const TSeries& ts, 
						 const std::string glitchID );

  // /// if there exists a glitch which has not yet ended, this returns false
  // virtual bool isDone( void );

  /// checks whether the GlitchConstruct object has been setup correctly
  bool isInitialized() const;
}; // end of GlitchConstruct class declaration


/** To make use of the chInterface, you can add StatisticsConstruct instances
  * to a channel.  This is meant to separate the statistics-calculating scheme
  * from the overhead of applying it to each channel.
  * @memo A base class for constructing statistics.
  * @author Rauha Rahkola
  * @version 0.1 May 29th, 2002
  ***************************************************************************/
class StatisticsConstruct: public chConstruct {
private:
  /*
   * These members can be hidden from derived stats-- they can be accessed
   * through the accessor functions
   */
  /// the type of statistic-- continuous or refreshed?
  updateType mType;

  /// Refresh rate for the statistic (in seconds)
  Interval mRefresh;

  /// Last refresh time
  Time mRefreshTime;

protected:
  /* 
   * All derived stats need to manage these members, but clients shouldn't see
   * them
   */
  /// possible initial value for statistic
  double mInitVal;

  /*
   * Not sure about these next members
   */
  /// temporary value for performing statistics among many filters
  double tmp1;
  /// temporary value for performing statistics among many filters
  double tmp2;

  /// a doubly-linked list of statistical results
  std::list<statResult> results;

public:
  /*
   * Clients/Derived stats need to be able to use these functions
   */
  /** StatisticsConstruct default constructor.  It's possible to start this
    * with an initial value, but you should remember that this value gets
    * applied to all (possibly filtered) TSeries for that channel.
    * @param statID name of the statistic construct
    * @param type   type of the statistic -- 0=continuous, 1=refreshable
    * @param rate   rate at which statistics are refreshed
    * @param init   initial value of statistic
    */
  explicit StatisticsConstruct( const std::string& statID="",
				const updateType type=uUnknown,
				const Interval& rate=0.0,
				const double init=0.0,
				const unsigned int debug=0 );

  /// StatisticsConstruct copy constructor
  StatisticsConstruct( const StatisticsConstruct& stat,
		       const unsigned int debug=0 );

  /// StatisticsConstruct destructor
  virtual ~StatisticsConstruct();

  /// StatisticsConstruct "clone" method, to allocate new StatisticsConstructs
  StatisticsConstruct *clone( void ) const;

  /* StatisticsConstruct modifiers
   */
  /// set the type of the construct -- 0=continuous, 1=refreshable
  void setType(const updateType type) { mType = type; }

  /// set the refresh rate for the construct (in seconds)
  void setRefresh(const Interval& rate ) { mRefresh = rate; }

  /// set the initial value for the construct
  void setInitialValue(const double val) {mInitVal = val; }

  /* StatisticsConstruct accessors
   */
  /// returns the updateType of the construct
  updateType getType( void ) const { return mType; }

  /// returns the type name of the construct -- either "Continuous" or "Refreshable"
  std::string getTypeName( void ) const {
    switch (mType) {
    case uContinuous:
      return std::string("Continuous");
    case uRefreshed:
      return std::string("Refreshed");
    default:
      return std::string("Unknown");
    }
  }

  /// returns the interval necessary to refresh the statistic
  Interval getRefresh( void ) const { return mRefresh; }

  /// returns the list of constructed statistics results
  std::list<statResult> const* getResults( void ) const { return &results; }

  /* StatisticsConstruct virtual functions
   */
  /** dumps formatted output of (one or more) statResult structures
    * @param out   the ostream to direct output to
    * @param total the number of results to dump, or '0' for all of them
    */
  virtual void dumpResults( std::ostream& out, unsigned int total=0 );

  /** resets the statistics -- continuous stats must be overridden to be reset
    * overridden.
    * @param override if true, resets continuous StatisticsConstructs as well
    */
  virtual void Reset( const bool override=false );

  /** updates the statistics for (possibly filtered) data-- a unique statID
    * must be given for the result to be distinguished among other data for
    * that channel
    * @param  ts     the TSeries to update statistics for
    * @param  statID try matching an existing statResult
    * @return list of statistics results obtained so far
    */
  virtual std::list<statResult>* updateStats( const TSeries& ts,
					       const std::string statID );

  /// checks whether the StatisticsConstruct object has been setup correctly
  bool isInitialized() const;

  /* StatisticsConstruct misc functions
   */
  /** will check a refreshable statistic and reset it if the refresh interval
    * time has been exceeded since the last isDue() call. If no time is given, 
    * current computer time is used (Warning: may be inexact!)
    * @param compare the time used for checking against the last isDue() call
    * @return true if result has been reset
    */
  bool isDue( const Time& compare=Time(0) );

  // /// returns false if any statistics still need more data to update
  // bool isDone();
}; // end of StatisticsConstruct class declaration;


/** To make use of the chInterface, you can add ResultConstruct instances
  * to a channel.  This is meant to separate the presentation of results
  * from the overhead of obtaining info from glitch-tests or statistics.  Also,
  * results can be sorted by channel.
  * @memo A base class for constructing results.
  * @author Rauha Rahkola
  * @version 0.1 May 29th, 2002
  ***************************************************************************/
class ResultConstruct: public chConstruct {
private:
  /*
   * These members can be hidden from derived results; use accessors instead
   */
  /// Type of the result-- continuous or refreshed?
  updateType mType;

  /// Update interval (in seconds) -- mainly used in refreshed results
  Interval mUpdate;

  /// Entry time of the previous result -- mainly used in refreshed results
  Time mEntryTime;

  /*
   * These functions are implicit; derived results don't need access to these
   */

protected:
  /*
   * All derived results need these members, but clients shouldn't see them
   */
  /// default I/O stream to use
  std::ostream* mOutput;

  /// default output mask to use
  unsigned int mOutmask;

  /*
   * Not sure whether all derived results need these
   */
  /// Histogram data -- depending on the type of result, this could be useful
  std::map<std::string,Histogram1> mHistMap;

public:
  /*
   * All derived results need these members, and clients need access to them
   */

  /*
   * Clients/Derived results need to be able to use these functions
   */
  /** ResultConstruct default constructor
    * @param resultID name of the result
    * @param type     type of the result -- 0=continuous, 1=refreshable
    * @param io       pointer to the iostream to use
    * @param update   update interval (in seconds)
    */
  explicit ResultConstruct( const std::string& resultID="",
			    const updateType type=uUnknown,
			    const unsigned int outmask=0,
			    std::ostream& out=std::cout,
			    const Interval& update=0.0,
			    const unsigned int debug=0);

  /// ResultConstruct copy constructor
  ResultConstruct( const ResultConstruct& result, const unsigned int debug=0 );

  /// ResultConstruct destructor
  virtual ~ResultConstruct();

  /// ResultConstruct assignment operator
  virtual ResultConstruct& operator =( const ResultConstruct& rhs );

  /// ResultConstruct "clone" method, to allocate new ResultConstructs
  ResultConstruct *clone( void ) const;

  /* ResultConstruct modifiers
   */
  /// sets the type of the result -- 0=continuous, 1=refreshable
  void setType(const updateType type) { mType = type; }

  /// sets the default I/O stream to use
  void setOutput( std::ostream *out ) { mOutput = out; }

  /// sets the default output mask to use
  void setOutmask( const unsigned int outmask ) { mOutmask = outmask; }

  /// sets the update interval
  void setUpdate( const Interval& update ) { mUpdate = update; }

  /* ResultConstruct accessors
   */
  /// returns the type of the result construct
  updateType getType( void ) { return mType; }

  /** returns the type name of the result construct -- "Continuous" or
    * "Refreshed" if known, otherwise "Unknown"
    */
  std::string getTypeName( void ) const {
    switch (mType) {
    case uContinuous:
      return std::string("Continuous");
    case uRefreshed:
      return std::string("Refreshed");
    default:
      return std::string("Unknown");
    }
  }



  /// returns the default I/O stream used by the result
  std::ostream *getOutput( void ) const { return mOutput; }

  /// returns the default output mask used by the result
  unsigned int getOutmask( void ) const { return mOutmask; }

  /// returns the interval which is necessary to update the result
  Interval getUpdate( void ) const { return mUpdate; }

  /// returns the latest refresh time
  Time getEntryTime( void ) const { return mEntryTime; }
     
  /* ResultConstruct virtual functions
   */    
  /** updates a result entry -- will not output result if override is not set
    * and glitch has not finished
    * @param result   a glitchResult entry
    * @param outmask  a bitset mask to direct the output.  Although the
    *   bitmask may be used arbitrarily by derived classes, the suggested
    *   bitmapping is: \begin{tabular}{cl}
    *   #2^0# & standard output\\
    *   #2^1# & output to an internally-defined ostream
    *           (derived classes only!)\\
    *   #2^2# & output to ostream 'out' in function call\\
    *   #2^3# & output to Trigger Manager (derived classes only!)\\
    *   #2^4# & output to DMTViewer (derived classes only!)\\
    *   #2^5# & output to trend file (derived classes only!)\\
    *   #2^6# & output all stored histograms of glitches since last refreshed
    *   \end{tabular}
    *   When no bits are set, results are tabulated internally.
    * @param override if true, forces output of result
    */
  virtual bool output( const glitchResult& result, unsigned int outmask=0,
		       const bool override=false, std::ostream& out=std::cout );

  /** updates a result entry -- will not output result if override is not set
    * and the statistics calculation has not finished
    * @param result   a statResult entry
    * @param outmask  a bitset mask to direct the output.  Although the
    *   bitmask may be used arbitrarily by derived classes, the suggested
    *   bitmapping is: \begin{tabular}{cl}
    *   #2^0# & standard output\\
    *   #2^1# & output to an internally-defined ostream
    *           (derived classes only!)\\
    *   #2^2# & output to ostream 'out' in function call\\
    *   #2^3# & output to Trigger Manager (derived classes only!)\\
    *   #2^4# & output to DMTViewer (derived classes only!)
    *   #2^6# & output all stored histograms of statistics since last refreshed
    *   \end{tabular}
    *   When no bits are set, results are tabulated internally.
    * @param override if true, forces output of result
    */
  virtual bool output( const statResult& result, unsigned int outmask=0,
		       const bool override=false, std::ostream& out=std::cout );

  /** resets the result -- continuous results must be overridden to be reset
    * @param override if true, will even reset a continuous result
    */
  virtual void Reset( const bool override=false );

  /// checks whether the ResultConstruct object has been setup correctly
  virtual bool isInitialized() const;

  /* ResultConstruct misc functions
   */
  /** will check a refreshable result and reset it if the refresh interval
    * time has been exceeded since the last isDue() call. If no time is given, 
    * current computer time is used (Warning: may be inexact!)
    * @param compare  the time used for checking against the last isDue() call
    * @param override if true, will reset the ResultConstruct and set the
    *   refresh time to the current computer time
    * @return true if result has been reset
    */
  bool isDue( const Time& compare=Time(0), const bool override=false );
}; // end of ResultConstruct class declaration

} // end of namespace channel

#endif // #ifndef CHCONSTRUCTS_HH
