/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef DQ_BIT_HH
#define DQ_BIT_HH 1

#include "TSeries.hh"
#include <iosfwd>
#include <map>
#include <vector>
#include <string>
#include <cmath>


/**  DQ_bit is a base class for classes that return a boolean state that 
  *  will be recorded as a bit in a specified channel name and bit number.
  */
class DQ_bit {
public:
   typedef std::vector<std::string> chanlist_type; ///< Channel list data type
   /**  %Time series vector used to pass a set of channel data to the 
     *  bit evaluation method.
     */	
   typedef std::vector<TSeries>     tser_vect;
public:
   /** Default constructor. Automatic data constructors only.
     */  
   DQ_bit(void);

   /** Destructor.
     */
   virtual ~DQ_bit(void);

   /**  Get the channel name in which the data quality bit is to be set.
     *  This name is defined by the configuration file.
     *  \brief Get the data quality channel name.
     * 	\return Channel name string.
     */ 
   const std::string& bit_channel(void) const {return _flag_channel;}

   /**  Set the name of the channel in which the data quality flag bit 
     *  is to be set.
     *  \brief Set the data quality channel name.
     *  \param chan Channel name string.
     */
   void bit_channel(const std::string& chan);

   /**  Get the bit number to be set in the specified data quality channel
     *  if this flag is true. THe bit number is an integer from 0-31 where 
     *  0 is the least significant bit.,
     *  \brief Get the flag bit number.
     *  \returns Bit number for this flag.
     */
   int bit_number(void) const {return _flag_bit;}

   /**  Set the bit number in the specified data quality channel that is to
     *  indicate the state of this flag. The bit number is an integer from 
     *  0-31 where 0 is the least significant bit.
     *  \brief Set the flag bit number.
     *  \param bit bit number.
     */
   void bit_number(int bit);

   /**  This function is called to evaluate the current state of the data 
     *  quality flag. The input channels listed in the configuration file
     *  are passed as a vector of TSeries in the order of appearance
     *  in the input channel list.
     *  \brief Flag state evaluation method. 
     *  \param data Vector of time series.
     *  \return true if the data quality flag is to be set.
     */
   virtual bool bit_value(const tser_vect& data)=0;

   /**  Get a constant reference to the list of channel names that are to 
     *  passed to the evaluation function.
     *  \brief Get the list of input channel names.
     *  \returns Constant reference to the list of input channel names.
     */
   const chanlist_type& channel_list(void) const {return _chan_list;}

   /**  Set the list of names of channels to be passed to the data quality 
     *  evaluation method.
     *  \brief Set the input channel name list.
     *  \param chans A list of channel names from which the list for this
     *               flag is to be set.
     */ 
   void channel_list(const chanlist_type& chans);

   /**  Get the data quality tool (class) name.
     *  \brief Data Quality class name.
     *  \returns Reference to the data quality class name string.
     */
   const std::string& dq_class(void) const {return _dq_class;}

   /**  Set the data quality tool (class) name.
     *  \brief Set the DQ class name.
     *  \param dqclass Constant reference to the data quality class name string.
     */
   void dq_class(const std::string& dqclass);

   /**  Get the data quality flag name. If this is used to generate a segment,
     *  this would be used as the segment name.
     *  \brief Get the DQ flag name.
     *  \returns Constant reference to the data quality flag name.
     */
   const std::string& dq_name(void) const {return _name;}

   /**  Set the data quality flag name. If this is used to generate a segment,
     *  this would be used as the segment name.
     *  \brief Set the DQ flag name.
     *  \param dqname Constant reference to the data quality flag name.
     */
   void dq_name(const std::string& dqname);

   /**  Initializ the flag evaluation tool. Not that this is an optional 
     *  virtual method that may be used to perform initialization of the
     *  tool instance after it has been constructed.
     *  \brief Tool initialization function.
     */
   virtual void init(void);

   /**  Get a numeric parameter value that was defined in the configuration 
     *  file. Numeric paramters are converted to double precision floating
     *  point quantities. Boolean parameters are converted to 1.0 (true) or
     *  0.0 (false).
     *  \brief Get a numeric parameter value.
     *  \param name Name of the numeric or boolean parameter as specified in 
     *              the configuration file.
     *  \returns Value of the numeric parameter.
     */
   double numeric_param(const std::string& name) const;

   /**  Define a numeric parameter from its name and double precision floating
     *  point value.
     *  \brief Define a numeric parameter value.
     *  \param name Name of the numeric parameter.
     *  \param val  Numeric parameter value.
     */
   void numeric_param(const std::string& name, double val);

   /**  Print the configuration reconstructed from the contents of the %DQ_bit 
     *  instance to the specified output stream.
     *  \brief Print the configuration.
     *  \param out Output strem reference.
     */
   void put_config(std::ostream& out) const;

   /**  Get a string parameter value that was defined in the configuration 
     *  file.
     *  \brief Get a string parameter value.
     *  \param name Name of the string parameter as specified in the 
     *              configuration file.
     *  \returns Constant reference to the string value.
     */
   const std::string& string_param(const std::string& name) const;

   /**  Define a string parameter and set it to its nominal value.
     *  \brief Define a string parameter.
     *  \param name Name of the string parameter.
     *  \param val  Constant reference to the string value.
     */
   void string_param(const std::string& name, const std::string& val);

private:
   std::string   _name;
   std::string   _dq_class;
   chanlist_type _chan_list;
   int           _flag_bit;
   std::string   _flag_channel;
protected:
   typedef std::map<std::string, double>      num_par_type;
   typedef num_par_type::iterator num_par_iter;
   typedef num_par_type::const_iterator const_num_par_iter;
   num_par_type _num_par;
   typedef std::map<std::string, std::string> str_par_type;
   typedef str_par_type::iterator str_par_iter;
   typedef str_par_type::const_iterator const_str_par_iter;
   str_par_type _str_par;
};

/** Define a plugin definition macro.
 */
#define DQ_PLUGIN(name) extern "C" DQ_bit* name##_init(void) {return new name;}

#endif
