// this class manipulates data for output into a text file
//
// Edward Daw, 8th September 2000

#include "Rms_output.hh"
#include <cstdio>
#include "logicals.h"
#include "Time.hh"
#include <iostream>

using namespace std;

Rms_output::
Rms_output(const std::string& file_name, int no_of_columns )
  : output_file_name(file_name), samp_rate_array(no_of_columns, 0), 
    header_line_1_set(no_of_columns), header_line_2_set(no_of_columns),
    data_array(no_of_columns, 0)
{
  // set the number of data columns
  total_no_of_columns = no_of_columns;

  // enable writing of text to the file
  _make_text_file = true;

  header_finished = NO;
  current_column_number = 0;
  lines_of_data_written = 0;
}

void
Rms_output::disableTextOutput(void) {
  _make_text_file = false;
  return;
}

void
Rms_output::enableTextOutput(void) {
  _make_text_file = true;
  return;
}

int 
Rms_output::update_samp_rate( int column, int value) {
  if(samp_rate_array.empty()) {
    return -1;
  } else if ( (column < 0) || (column > total_no_of_columns) ) {
    return -2;
  }
  else {
    samp_rate_array[ column ] = value;
    return 0;
  }
}

int
Rms_output::add_header(const std::string& first_header_line_entry,
		       const std::string& second_header_line_entry,
		       int column_number ) {
  // check that the header isn't already written
  if(header_finished == YES) {
    printf("Error: header already finished.\n");
    return ERROR;
  }
  
  // add column data, as long column_number matches the internal
  // counter, current_column_number
  if( column_number == current_column_number ) {
    header_line_1_set[column_number] = first_header_line_entry;
    header_line_2_set[column_number] = second_header_line_entry;
    ++current_column_number;
  }
  else {
    cerr << "Error: column number mismach." << endl;
    return ERROR;
  }

  // if we have reached the last column, append the file with the
  // header.
  if( current_column_number == total_no_of_columns ) {
 
    if( _make_text_file ) {
      
      FILE* out = fopen(output_file_name.c_str(), "a");
      if(out == NULL) {
	cerr << "Error opening " << output_file_name << ". Exiting." << endl;
	return ERROR;
      }
      
      int header_count;
      //    fprintf(out,"#GPS\t");
      //The following was added by KSG on 9/29/05
      for( header_count = 0;
	   header_count < (total_no_of_columns);
	   header_count++ ) {
	fprintf(out,"#H0:DMT-BRMS_%s:%s\n",
		(header_line_1_set[header_count].c_str()),
		(header_line_2_set[header_count].c_str()) );
      }
      fclose(out);

    } // if _make_text_file

    header_finished = YES;
    current_column_number = 0;
  }

  return OK;
}

int
Rms_output::
add_data( float value, int column_number )
{
  // check that I'm not still writing the header.
  if(header_finished == NO) {
    printf("Error: still writing header during data write attempt.\n");
    return ERROR;
  }

  // if the data column on which a write was attempted is the same
  // as the internal data column count, then load the data into the
  // data array.
  if( column_number == current_column_number ) {
    data_array[ column_number ] = value;
    ++ current_column_number;
  }
  // if the row of data is complete and the _make_text_file bit is
  // set to true, append the data file. Otherwise, reset the values of
  // the data columns to zero but don't write them to a text file.

  if( current_column_number == total_no_of_columns ) {

    if ( _make_text_file ) {
      FILE* out = fopen(output_file_name.c_str(), "a");
      if (out == NULL) {
	cerr << "Error opening " << output_file_name << ". Exiting." << endl;
	return ERROR;
      }
      fprintf(out,"%ld\t",gps_time_for_entry);
      for( int data_count = 0;
	   data_count < (total_no_of_columns-1);
	   data_count++ ) {
	fprintf(out,"%.4e\t",data_array[ data_count ] );
	data_array[ data_count ] = 0;
      }
      fprintf(out,"%.4e\n",data_array[ total_no_of_columns - 1 ] );
      data_array[ total_no_of_columns - 1 ] = 0;
      fclose(out);

    } else {

      for( int data_count = 0;
	   data_count < total_no_of_columns ;
	   data_count++ ) {
	data_array[ data_count ] = 0;
      }

    }

    current_column_number = 0;
    ++lines_of_data_written;
  }

  return OK;
}

int
Rms_output::log_error(const char* error_string, const Time& gpstime) const {
  if( _make_text_file ) {
    
    FILE* out = fopen(output_file_name.c_str(),"a");
    if (out == NULL) {
      return -1;
    }
    fprintf(out,"# %ld ERROR: %s\n", gpstime.getS(), error_string);
    fclose(out);

  } // if _make_text_file

  cerr << "Sent error to file " << output_file_name << "." << endl;
  return 0;
}

Rms_output::~Rms_output(void) {
}



