///////////////////////////////////////////////////////////////////////////////
//
//	Class StrainbandsMon: implementation of DMT to monitor strain
//	version 1.2 (2006.05.26)
//	author: Ramon S. Armen (rarmen@umich.edu)
//
///////////////////////////////////////////////////////////////////////////////

//Class definition
#include "StrainbandsMon.hh"

//Standard library includes

//DMT library classes used
#include "Hamming.hh"
#include "Hanning.hh"
#include "Blackman.hh"
#include "FlatTop.hh"
#include <sstream>

//call the macro which creates the main function, and begins
//execution of the StrainbandsMon DMT, as well as handling the
//main event processing loop.
EXECDAT( StrainbandsMon );


//Creation of a new monitor
StrainbandsMon::StrainbandsMon( int argc, const char *argv[] )
	: DatEnv( argc, argv ), mOSC( getDacc() )
{
	setDefaultParams(); //set parameters to default values
	parseCommandLine(argc, argv); //change parameters from command line args

	//read the default config file if none was specified
	if( configFile == "" ) {
		configFile = "StrainbandsMon.cfg";
		readConfigFile();
	}

	//If the user did not enter an IFO or a channel, then terminate.
	//If the user only entered one of the above, then determine the other
	//from the one that they entered.
	if( ifo == "" && channel == "" ) {
		cerr << "StrainbandsMon usage error: you must specify either "
			 << "a channel or an IFO to monitor." << endl;
		dumpUsage();
		exit( 1 );
	}
	else if( ifo == "" ) ifo = channel.substr( 0, 2 );
	else if( channel == "" ) channel = ifo + ":LSC-AS_Q";
	else if( channel.at(0) == ':' ) channel = ifo + channel;
	
	//if the user didnt enter lock conditions or a reference data file,
	//then set them to their default values for the given IFO.
	if( lockConditions == "" ) 
		lockConditions = ifo + ":Both_arms_locked_strict_cm";
	if( refDataFile == "" ) 
		refDataFile = "ReferenceCalibration_" + ifo + ".xml";

	//set up some values that we use
	tserieslength = historyLength / timeStride;
	isFirstRun = true;
	lastDataTime = Time(0);
	//IFO arm lengths, specified in nm (which is what easycalibrate uses)
	if( ifo == "H1" || ifo == "L1" ) arm_length = 4.0e12;
	else if( ifo == "H2" ) arm_length = 2.0e12;
	else {
		cerr << "StrainbandsMon error: unrecognized IFO " << ifo << endl;
		dumpUsage();
	}
			
	//Print out parameter information
	if( verbosity > 1 ) cout << "StrainbandsMon starting." << endl;
	if( verbosity > 3 ) {
		cout << "Parameters: " << endl 
			 << "    Verbosity: " << verbosity << endl
			 << "    OSC file: " << oscFile << endl
			 << "    Config File: " << configFile << endl
			 << "    Calibration Reference File: " << refDataFile << endl
			 << "    Lock Conditions: " << lockConditions << endl
			 << "    Window type: " << windowType << endl
			 << "    IFO: " << ifo << endl
			 << "    Channel: " << channel << endl
			 << "    Timestride: " << timeStride << endl
			 << "    Number of Intervals: " << numIntervals << endl;
		cout << "Monitoring bands: " << endl;
		for( unsigned int i = 0; i < bands.size(); i++ ) {
			cout << "    " << bands[i]->lowerFreq << " - "
				 << bands[i]->upperFreq << " Hz" << endl;
		}
		cout << endl << endl;
	}

	//Initialize the bands' values which werent set in their construction.
	if( verbosity > 4 ) cout << "Creating channels:" << endl;
	for( unsigned int i = 0; i < bands.size(); i++ ) {
		if( bands[i]->channelName == "" ) {
			//give it a default name
			stringstream str;
			str << ifo << ":DMT-SBND_AS_Q_" << ((int)bands[i]->lowerFreq)
				<< "_" << ((int)bands[i]->upperFreq)
				<< ((bands[i]->notches.size() > 0) ? "_notched" : "");
			str >> bands[i]->channelName;
		} else {
			bands[i]->channelName = ifo + bands[i]->channelName;
		}
		if( verbosity > 4 ) 
			cout << "    Channel created: " << bands[i]->channelName << endl;
		bands[i]->historyVals = new float[ tserieslength ];
		for( int h = 0; h < tserieslength; h++ ) {
			bands[i]->historyVals[h] = -1;
		}
	}
	if( verbosity > 4 ) cout << endl << endl;
	
	if( verbosity > 1 )
		cout << "Calibrating (this may take a few moments)..." << endl;
			
	//Initialize variables used to serve data to us, validate that the data
	//is valid, and calculate the strain.
	mOSC.setStride( 1 );
	mOSC.readConfig( oscFile );
	psd = new PSD( window, numIntervals );
	getDacc().setStride( Interval(timeStride) );
	setStrideAlignment( timeStride, 0.0 );
	
	//add our channel to retrieve data from
	getDacc().addChannel( channel.c_str(), 2 );
	getDacc().setIgnoreMissingChannel(true);
	
	//initialize the calibrator to calibrate between the frequency ranges
	//for which we will be computing strain.
	mCalibrate = new FDEasyCalibrate( &getDacc(), refDataFile.c_str(),
		true, 0.0, numIntervals*1.0/timeStride, (int)(7000.*timeStride/numIntervals+1) );
	
	if( verbosity > 1 ) 
		cout << "    Calibration done. Setting up server and trends." << endl;
	
	//Set the DMT name
	string serverName = "StrainbandsMon_" + ifo;
	if( debugMode ) serverName += "_debug";
	setServerName( serverName.c_str() );
	
	//Set up the trend
	trend.setName( serverName.c_str() );
	trend.setType(Trend::kMinute);
	trend.setIFO( ifo.c_str() );
	trend.setAutoUpdate( false );
	trend.setFrameCount( 1 );
	for( unsigned int i = 0; i < bands.size(); i++ ) {
		trend.addChannel(bands[i]->channelName.c_str());
	}
	trend.writeIndex();
	
	//Copy the config file to the HTML directory so that it can be viewed
	if( verbosity > 1 ) cout << "    Copying config file." << endl;

	string outFileName = getenv( "DMTHTMLOUT" );
	if( outFileName.substr( outFileName.length()-1 ) != "/" )
		outFileName += "/";
	outFileName = outFileName + configFile + ".txt";

	ifstream configIn(configFile.c_str(), ios::in|ios::binary);
	ofstream configOut(outFileName.c_str(), ios_base::out|ios_base::trunc|ios::binary );
	int transferBytes = 1;
	char buffer[255];
	
	while( transferBytes != 0 ) {
		configIn.read( (char*)buffer, 255 );
		transferBytes = configIn.gcount();
		configOut.write((char*)buffer, transferBytes);
	}
	
	startTime = Now();
	
	if( verbosity > 1 ) cout << "    Startup done. Waiting for data."
		<< endl << endl << endl;
}


//Set up default parameters
void StrainbandsMon::setDefaultParams() {
	verbosity = 1;
	oscFile = "LockLoss.conf";
	configFile = "";
	refDataFile = ""; //By default sets to ReferenceCalibration_XX.xml,
	                  //where XX is the IFO name being monitored.
	lockConditions = ""; //By default sets to XX:Both_arms_locked_strict_cm,
						 //where XX is the IFO name being monitored.
	window = new Hanning();
	windowType = "hanning";
	historyLength = 12 * 60 * 60;
	debugMode = false;
	
	//If an IFO is specified but no channel, the channel will default to 
	//XX:LSC-AS_Q, where XX is the specified IFO. If a channel is specified
	//but no IFO, then the channel will be parsed to obtain the IFO. If
	//both an IFO and a channel are specified, they will both be used. If
	//the channel is of the form ":channelName", then the IFO name will
	//be put at the beginning of the channel name. Either an IFO or a channel
	//must be specified.
	ifo = "";
	channel = "";
	
	timeStride = 60;
	numIntervals = 15;
}


//attempt to read arguments from the command line
void StrainbandsMon::parseCommandLine( int argc, const char *argv[] ) {
	if( argc <= 1 ) return; //if there were no arguments, return.

	//put all the arguments into a string
	string arguments = argv[1];
	for( int i = 2; i < argc; i++ ) {
		string add = argv[i];
		arguments += " " + add;
	}

	//send the string to the parameter loader through a stream.
	stringstream ss(arguments);
	string error = loadParams(&ss);
	if( error != "" ) {
		//Error reading parameters from command line. Terminate.
		cerr << "StrainbandsMon usage error on command line arguments. "
			 << "Error occured for argument " << error << endl;
		dumpUsage();
		exit( 1 );
	}
}


//read parameters and bands from config file. 
//See header for format of config file.
void StrainbandsMon::readConfigFile() {
	//open the file for reading
	ifstream fs( configFile.c_str(), ios::in );
	if( !fs.is_open() ) {
		//could not open file. Either the file was not there, we did not
		//have read permissions, or some error occured.
		cerr << "StrainbandsMon fatal error: could not open config file."
			 << endl << "    Tried to use file " << configFile << endl;
	}
	//try to load the parameters from the file.
	string error = loadParams(&fs);
	fs.close();
	if( error != "" ) {
		//Error reading config file. Terminate.
		cerr << "StrainbandsMon error in config file. "
			 << "Error occured for argument " << error << endl;
		dumpUsage();
		exit( 1 );
	}
}


//Load arguments into run parameters
string StrainbandsMon::loadParams( istream *s ) {
	string argName, val;

	//get the name of the argument, and then parse it, storing the
	//following value in the corresponding parameter. If the name is
	//not one of the following if statements, then it is not a
	//recognized parameter, so return it. repeat until we run out of
	//parameters to parse.
	while( *s >> argName ) {
		if( argName == "-verbosity" || argName == "-v" ) {
			if( !(*s >> val) ) return argName;
			verbosity = atoi(val.c_str()); //parse parameter to int (stdlib.h)
			if( verbosity <= 0 ) return argName; //verbosity must be > 0.
		} else if( argName == "-OSCFile" ) {
			if( !(*s >> val) ) return argName;
			oscFile = val;
		} else if( argName == "-config" || argName == "-conf" ) {
			if( !(*s >> val) ) return argName;
			configFile = val;
			readConfigFile();
		} else if( argName == "-calFile" ) {
			if( !(*s >> val) ) return argName;
			refDataFile = val;
		} else if( argName == "-OSCcond" || argName == "-lock") {
			if( !(*s >> val) ) return argName;
			lockConditions = val;
		} else if( argName == "-window" || argName == "-w") {
			if( !(*s >> val) ) return argName;
			windowType = val;

			delete window;
			if( windowType == "hamming" ) window = new Hamming();
			else if( windowType == "hanning" ) window = new Hanning();
			else if( windowType == "blackman" ) window = new Blackman();
			else if( windowType == "flattop" ) window = new FlatTop();
			else return argName; //Window type not supported
		} else if( argName == "-help" || argName == "-h" ) {
			dumpUsage();
			exit( 0 );
		} else if( argName == "-ifo" ) {
			if( !(*s >> val) ) return argName;
			ifo = val;
		} else if( argName == "-channel" ) {
			if( !(*s >> val) ) return argName;
			channel = val;
		} else if( argName == "-stride" ) {
			if( !(*s >> val) ) return argName;
			timeStride = atoi(val.c_str());
			if( timeStride <= 0 ) return argName;
		} else if( argName == "-intervals" ) {
			if( !(*s >> val) ) return argName;
			numIntervals = atoi(val.c_str());
			if( timeStride <= 0 ) return argName;
		} else if( argName == "-bands" ) {
			return readBands( s );
		} else if( argName == "-history" ) {
			if( !(*s >> val) ) return argName;
			historyLength = atoi(val.c_str());
			if( historyLength < timeStride ) return argName;
		} else if( argName == "-debug" ) {
			debugMode = true;
		} else if( argName.at(0)=='#' ) {
			//this line is commented out. Remove it from processing.
			//We limit line lengths to 256 characters.
			char comment[256]; //dummy variable to store comment in
			s->getline( comment, 256 );
		} else if( !isDatEnvArg( argName.c_str() ) ) {
			//Its not one of our args, and its not a DatEnv arg,
			//so there is a usage error.
			return argName;
		}
	}
	
	//Everything loaded successfully.
	return "";
}


//Read in what bands to monitor
string StrainbandsMon::readBands( istream *s ) {
	//keep reading in words until we reach the end of the stream or
	//reach an '-end' flag.
	string word;
	*s >> word;
	while( !s->eof() ) {
		if( word == "-end" ) break;
	
		//create a new band for this band specification, and read
		//in the band limits.
		Band *b = new Band();
		b->channelName = "";
		b->lowerFreq = atof( word.c_str() );
		*s >> word;
		b->upperFreq = atof( word.c_str() );
		
		//look for notches or a specified name. Continue until
		//we reach the next band.
		for( ; ; ) {
			*s >> word;
			if( word == "notch" ) {
				//Insert a notch into the band, reading the notch's
				//lower and upper frequencies from the stream.
				*s >> word;
				b->notches.push_back( atof( word.c_str() ) );
				*s >> word;
				b->notches.push_back( atof( word.c_str() ) );
			} else if( word == "name" ) {
				//Set the channel's name, including the required
				//part specifying the IFO and DMT name
				*s >> word;
				stringstream str;
				str << ":DMT-SBND_AS_Q_" << word;
				str >> b->channelName;
			} else break; //The word is part of the next band, so break
		}
		
		bands.push_back( b );
	}
	
	//all the bands loaded successfully - either the end of the
	//stream was reached, or the '-end' keyword was found
	return "";
}


//Dump usage information to cout
void StrainbandsMon::dumpUsage() {
	cout << "Usage: " << endl;
	ifstream fs("StrainbandsMonUsage.txt");
	
	char s[128];
	fs.getline( s, 128 );
	while( fs.good() ) {
		cout << s << endl;
		fs.getline( s, 128 );
	}
	fs.close();
}


//Cleanup data on monitor termination
StrainbandsMon::~StrainbandsMon() {
	if( verbosity > 1 ) cout << "Terminating." << endl;

	//write out and close the trend file
	trend.close();

	//delete all the dynamically created data that we have made
	delete window;
	for( unsigned int i = 0; i < bands.size(); i++ ) {
		if( bands[i]->history != NULL ) delete bands[i]->history;
		if( bands[i]->historyVals != NULL ) delete bands[i]->historyVals;
		delete bands[i];
	}
	bands.clear();
	delete psd;
	delete mCalibrate;
}


//Process data for one timestride
void StrainbandsMon::ProcessData() {
	//get the time that the data was taken, and the time of the start of
	//our DMT viewer history.
	Time curTime = getDacc().getFillTime();
	Time historyStartTime = curTime; historyStartTime -= historyLength;
	if( verbosity > 2 ) cout << "Processing data at time " << curTime << endl;
	
	//If it is the first time we have processed data, begin serving the data.
	if( isFirstRun ) {
		for( unsigned int i = 0; i < bands.size(); i++ ) {
			bands[i]->history = new TSeries( historyStartTime, 
				timeStride, tserieslength, bands[i]->historyVals );
			serveData( bands[i]->channelName.c_str(), bands[i]->history );
		}
	}
		
	//determine if there were any data dropouts.
	if( !isFirstRun && (curTime.totalS() > lastFillTime + timeStride ) ) {
		int missing_strides = (int) ((curTime.totalS()-lastFillTime)/timeStride)-1;
		if( verbosity > 4 ) 
			cout << "    missed strides: " << missing_strides << endl;
		//set the history values for each band to -1 for the missing data
		for( unsigned int i = 0; i < bands.size(); i++ ) {
			int h = 0;
			//move the previous data back in the array
			for( ; h < tserieslength-missing_strides; h++ ) {
				bands[i]->historyVals[h] 
					= bands[i]->historyVals[h+missing_strides];
			}
			//and set the new points to -1, including in the trend file
			Time trendTime((int)lastFillTime);
			for( ; h < tserieslength; h++ ) {
				trendTime += timeStride;
				trend.trendData( bands[i]->channelName.c_str(), trendTime, -1 );
				bands[i]->historyVals[h] = -1;
			}
		}
	}

	isFirstRun = false;
	
	//Move the data for each band back by one timestride, making room for the
	//new data value to be put into the array and sent to the DMT viewer.
	for( unsigned int i = 0; i < bands.size(); i++ ) {
		int h = 0;
		for( ; h < tserieslength-1; h++ ) {
			bands[i]->historyVals[h] = bands[i]->historyVals[h+1];
		}
	}

	
	//check to see if the operating state conditions are met. If they arent,
	//fill the data for this timestride with -1, ignoring the data we were sent
	if( !mOSC.satisfied( lockConditions ) ) {
		if( verbosity > 4 ) cout << "    OSC not met. Data set to -1." << endl;
		for( unsigned int i = 0; i < bands.size(); i++ ) {
			bands[i]->historyVals[tserieslength-1] = 0;
			trend.trendData( bands[i]->channelName.c_str(), curTime, 0 );
		}
		
		//update the DMT viewer and trend, and then return.
		for( unsigned int i=0; i < bands.size(); i++ ) {
			bands[i]->history->setData( historyStartTime,
				timeStride, bands[i]->historyVals, tserieslength );
		}
		lastFillTime = curTime.totalS();
		trend.Update();
		dumpHTMLFile( curTime, lastDataTime );
		return;
	}
	
	//If we got here, the data is good.
	lastDataTime = curTime;

	//Get the raw data
	const TSeries *ts_asq = getDacc().refData( channel.c_str() );
	
	//Run it through PSD to get an FSpectrum
	FSpectrum PSDed_data;
	psd->generate( PSDed_data, ts_asq );

	//Calibrate the data
	mCalibrate->UpdateResponseFunction();
	PSDed_data = mCalibrate->Apply( PSDed_data );
	
	//get the resulting calibrated values
	float *strainDensity = new float[ PSDed_data.getNStep() + 1 ];
	PSDed_data.getData( PSDed_data.getNStep() + 1, strainDensity );
	
	//go through and calculate strain for each band.
	for( unsigned int i = 0; i < bands.size(); i++ ) {
		//At this point, the data is in units of nm^2/Hz. To convert to
		//strain, integrate along the frequency band we are looking at, 
		//take the square root, and divide by the IFO length.
		double sum = PSDed_data.getSum( bands[i]->lowerFreq,
			bands[i]->upperFreq - bands[i]->lowerFreq ) 
			* PSDed_data.getFStep();
		//remove any notches
		for( unsigned int n = 0; n < bands[i]->notches.size(); n+=2 ) {
			sum -= PSDed_data.getSum( bands[i]->notches[n],
				bands[i]->notches[n+1] - bands[i]->notches[n] ) 
				* PSDed_data.getFStep();
		}
		//and finish the calculation.
		float bandVal = (float)( sqrt( sum ) / arm_length );
		
		//add the data point to the DMT viewer array and trend file
		bands[i]->historyVals[tserieslength-1] = bandVal * SCALE_FACTOR;
		trend.trendData( bands[i]->channelName.c_str(), 
			ts_asq->getStartTime(), bandVal * SCALE_FACTOR );
		bands[i]->history->setData( historyStartTime, timeStride, 
			bands[i]->historyVals, tserieslength );
		
		//print out values
		if( verbosity > 5 ) cout << "    " << bands[i]->lowerFreq
			<< " - " << bands[i]->upperFreq << ": " << bandVal << endl;
	}
	
	delete strainDensity;
	
	//update the trend to include the most recent data
	trend.Update();
	
	//update the last time we got data to this data sample
	lastFillTime = curTime.totalS();
	
	dumpHTMLFile( curTime, lastDataTime );
}


//Write the HTML file to $DMTHTMLOUT/index.html
void StrainbandsMon::dumpHTMLFile( Time curTime, Time lastData ) {
	//Get the directory/file name to write to, and open the file for writing
	if( verbosity > 3 ) cout << "    HTML summary file being written" << endl;
	string htmlName = getenv( "DMTHTMLOUT" );
	if( htmlName.substr( htmlName.length()-1 ) != "/" )
		htmlName += "/";
	htmlName = htmlName + "index.html";
	fstream file( htmlName.c_str(), ios_base::out | ios_base::trunc );
	
	//convert the GPS time to UTC and local times
	char utcStartTime[22];
	char localStartTime[22];
	char utcCurTime[22];
	char localCurTime[22];
	string format = "%H:%N:%S on %m/%d/%y";
	TimeStr( startTime, utcStartTime, format.c_str() );
	LocalStr( startTime, localStartTime, format.c_str() );
	TimeStr( curTime, utcCurTime, format.c_str() );
	LocalStr( curTime, localCurTime, format.c_str() );
	
	//print out the HTML page
	file << "<html><head><title>StrainbandsMon: " << curTime 
		 << "</title></head><body>\n<b>StrainbandsMon</b><br><br>\n";
	
	if( lastData == curTime ) {
		file << "<font color=\"green\">Currently in science mode</font><br>\n";
	} else {
		file << "<font color=\"red\">Not in science mode</font><br>\n";
	}
	
	file << "<table cellpadding=\"5\"><tr><th></th><th>GPS Time</th>"
		 << "<th>UTC Time</th><th>Server Local Time</th></tr>\n"

		 << "<tr><td><b>Current monitor started at:</b></td>" 
		 << "<td>" << (long long)(startTime.totalS()) << "</td>"
		 << "<td>" << utcStartTime << "</td>"
		 << "<td>" << localStartTime << "</td></tr>\n"
		 
		 << "<tr><td><b>Last data processed at:</b></td>"
		 << "<td>" << (long long)(curTime.totalS()) << "</td>"
		 << "<td>" << utcCurTime << "</td>"
		 << "<td>" << localCurTime << "</td></tr>";
	
	if( lastData != curTime && lastData > Time(0) ) {
		char utcLastTime[22];
		char localLastTime[22];
		TimeStr( lastData, utcLastTime, format.c_str() );
		LocalStr( lastData, localLastTime, format.c_str() );

		file << "<tr><td><b>Last time in scince mode:</b></td>"
			 << "<td>" << (long long)(lastData.totalS()) << "</td>"
			 << "<td>" << utcLastTime << "</td>"
			 << "<td>" << localLastTime << "</td></tr>";
	}
	
	file << "</table>\n<br><br>\n"
	
		 << "StrainbandsMon monitors strain within frequency bands. "
		 << "It uses FDEasyCalibrate to compute [strain / sqrt(Hz)] "
		 << "from raw AS_Q data, and then integrates over a frequency "
		 << "band to obtain strain.<br><br>\n"
		 
		 << "<b>Parameter information:</b><br>\n"
		 << "IFO: " << ifo << "<br>\n"
		 << "Channel: " << channel << "<br>\n"
		 << "Timestride: " << timeStride << "<br>\n"
		 << "Number of Intervals per Timestride: " << numIntervals << "<br>\n"
		 << "Lock Conditions: " << lockConditions << "<br><br>\n"
		 
		 << "<b>Frequency bands being monitored on " << ifo << ":</b><br>\n"
		 << "These bands are the bands that DataQual uses: there are narrow "
		 << "bands, broad bands, and narrow bands with artifacts.<br>\n"
		 << "<table><tr><th>Band Frequencies</th><th>Channel Name</th>";
	
	if( lastData == curTime ) {
		file << "<th><font color=\"green\">Current Strain</font></th></tr>";	
	} else if( lastData > Time(0) ) {
		file << "<th><font color=\"red\">Last Valid Strain (GPS "
			 << (long long)(lastData.totalS()) << ")</font></th></tr>";
	} else {
		file << "<th><font color=\"red\">Has Never Been In Science Mode"
			 << "</font></th></tr>";
	}
	
	
	for( unsigned int i = 0; i < bands.size(); i++ ) {
		file << "<tr><td>" << bands[i]->lowerFreq << " - " 
			 << bands[i]->upperFreq;
		if( bands[i]->notches.size() > 0 ) file << " (notched)";	 
		file << "</td><td>"
			 << bands[i]->channelName << "</td><td>";
		
		if( lastData > Time(0) ) {
			//Calculate the position of the last valid data and print it
			int offset = (int)(curTime - lastData) / timeStride + 1;
			file << bands[i]->historyVals[tserieslength-offset] / SCALE_FACTOR;
		} else {
			//has never been in science mode
			file << "N/A";
		}
		file << "</td></tr>\n";
	}
	file << "</table><br><br>"
		 << "<a href=\"" << configFile << ".txt\">Click here</a> to see the "
		 << "config file used.</body></html>";
	file.close();
}


//Something important is happening
void StrainbandsMon::Attention() {
	//Let somebody else take care of it.
	MonServer::Attention();
}
