#ifndef STATION_HH
#define STATION_HH

#include "DatEnv.hh"
#include "MonServer.hh"
#include "Quadetrend.hh"
#include "FixedLenTS.hh"
#include "FSpectrum.hh"
#include "FrWriter.hh"
#include "Histogram1.hh"
#include "OperStateCondList.hh"
#include "autopipe.hh"
#include <vector>
#include <string>

class Pipe;

/**  @name Station
  *  @memo stationarity monitor.
  *  The stationarity monitor measures short-term variations in the noise,
  *  (gain and linearity) of an arbitrary channel. A snapshot PSD is taken
  *  at short time intervals. The variance of each bin is then measured.
  *  Station produces a Time series of the variation in the noise floor
  *  and a spectrum of this series with 10mHz resolution, averaged over 
  *  20 minute strides.
  *  @author J. Zweizig
  *  @version $Id: Station.hh 6016 2009-11-02 22:14:54Z john.zweizig $
  */

//===================================== Plot sigma versus another channel
class xyhist {
public:
    xyhist(const std::string& name, int nBin, double xmin, double xmax);
    ~xyhist(void);
    void calculate(void);
    const char* getChannel(void) const;
    void fill(double x, double y);
    const Histogram1& refXYPlot(void) const;
    void write(void) const;
private:
    std::string mChannel;
    Histogram1 mSum1;
    Histogram1 mSumy;
    Histogram1 mSumyy;
    Histogram1 mXYPlot;
};

inline const char*
xyhist::getChannel(void) const {
    return mChannel.c_str();
}

inline const Histogram1&
xyhist::refXYPlot(void) const {
    return mXYPlot;
}

class stat_class {
public:
    stat_class(const std::string& name, const std::string& chan, 
	       long nsample=128);
    ~stat_class(void);
    void addPlot(const xyhist& xy);
    const char* getChannel(void) const;
    const char* getName(void) const;
    const char* getOscCond(void) const;
    double getVariance(const TSeries& ts);
    int getNXrefChans(void) const;
    const char* getXrefChan(int i) const;
    void  print(void) const;
    void  procStride(const TSeries& ts);
    const TSeries&    refHistory(void) const;
    const FSpectrum&  refPSD(void) const;
    const Histogram1& refSigmaHist(void) const;
    const Histogram1& refXrefHisto(int i) const;
    void reset(void);
    void setDebug(int nLvl);
    void setNSample(long ns);
    void setFilter(const Pipe& p);
    void setFMax(double fmax);
    void setFMin(double fmin);
    void setOscCond(const std::string& cond);
    void synch(void);
    void update(const TSeries& ts);

private:
    typedef std::vector<xyhist> plot_list;
    typedef plot_list::iterator plot_iter;
    typedef plot_list::const_iterator const_plot_iter;

private:
    //---------------------------------  Parameters
    /**  Name of the statistics tool.
      */
    std::string mName;

    /**  Channel name.
      */
    std::string mChannel;

    /**  Osc condition that must be true  if the statistics of the 
      *  channel are to be measured.
      */
    std::string mOscCond;

    /**  Number of samples
     */
    long        mNSample;

    /**  Low edge of frequency band inspected by this tool.
      */
    double      mFMin;

    /**  High edge of frequency band inspected by this tool.
     */
    double      mFMax;

    /**  Length of time to accumulate data to be PSDed.
     */
    Interval    mPsdStride;

    /**  Number of PSDs to be averaged for the dmtviewer PSD.
     */
    FSpectrum::size_type mNPsdAvg;

    /**  Minimum PSDs to be averaged for the dmtviewer PSD (at end of lock).
     */
    FSpectrum::size_type minPsdAvg;

    /**  Quadratic detrend filter.
      */
    Quadetrend  mDetrend;

    /** Filter to be used for pre-processing this channel.
      */
    auto_pipe   mFilter;

    //---------------------------------  Temp storage
    TSeries     mStride;

    //---------------------------------  Statistics
    /**  History of 1-second sigmas. This time series contains the sigma 
      *  of the (typically 64) noise estimates in each 1-second bin.
      */
    FixedLenTS  mHistory;

    /**  Histogram of the individual (1/64 s) noise esimates for the 
      *  specified frequency babnd.
      */
    Histogram1  mSigmaHist;

    /**  Noise vs time in 1/64th second slices. This is accumulated for
      *  an interval defined by mPsdStride (100s by default). The data 
      *  are the ffted the PSD is added to mPSD.
      */
    TSeries     mNoise;

    /**  Internal accumulator for the variance PSD.
     */
    FSpectrum   mPSD;

    /**  Average PSD over the last major time interval (mPsdStride * mNPsdAvg)
      */
    FSpectrum   mAvgPsd;

    /**  List of plots to be made to investigate linearity with .
     */
    plot_list   mPlotList;

    //----------------------------------  high level statistics
    /**  Total power, summed over run time.
     */
    double      mSumPower;
    double      mSumAmpl;
    int         mDebug;
};

//======================================  inline methods
inline const char* 
stat_class::getChannel(void) const {
    return mChannel.c_str();
}

inline const char* 
stat_class::getName(void) const {
    return mName.c_str();
}

inline const char* 
stat_class::getOscCond(void) const {
    return mOscCond.c_str();
}

inline int 
stat_class::getNXrefChans(void) const {
    return mPlotList.size();
}

inline const char* 
stat_class::getXrefChan(int i) const {
    return mPlotList[i].getChannel();
}

inline const TSeries& 
stat_class::refHistory(void) const {
    return mHistory;
}

inline const Histogram1& 
stat_class::refSigmaHist(void) const {
    return mSigmaHist;
}

inline const FSpectrum& 
stat_class::refPSD(void) const {
    return mAvgPsd;
}

inline const Histogram1& 
stat_class::refXrefHisto(int i) const {
    return mPlotList[i].refXYPlot();
}

//======================================  Stationarity monitor class
class Station : public DatEnv, MonServer {
public:
    Station(int argc, const char* argv[]);
    ~Station(void);
    void ProcessData(void);
    void ReadConfig(const std::string& file);
    void write(void);
    void Attention(void);
public:
    typedef std::vector<stat_class> StatList;
    typedef StatList::iterator stat_iter;
private:
    Time     mStartFrame;
    Time     mEndFrame;
    OperStateCondList mOsc;
    std::string mHistoFile;
    StatList mList;
};

#endif // !defined(STATION_HH)
