/* -*- mode: c++; c-basic-offset: 4; -*- */
//
// PlaneMon processing routine
// Author: Evan Goetz (egoetz@umich.edu)
// 16 June 2006
//
/////////////////////////////////////////////////////////

#include "PlaneMon.hh"

//=====================================  Include standard headers
// #include <cstdio>  // included by PlaneMonConfig.hh 
#include <iostream>
#include <fstream>
#include <iomanip>
#include <string>
#include <cstdlib>
#include <cmath>
#include <vector>
#include <sys/stat.h>
#include <unistd.h>

//=====================================  Include LIGO-specific headers
#include "Dacc.hh"
#include "TSeries.hh"
#include "FSeries.hh"
#include "FSpectrum.hh"
#include "Window.hh"
#include "Hanning.hh"
#include "TrigRslt.hh"
#include "TrigClient.hh"
#include "Segment.hh"
#include "html/writer.hh"
#include "html/Attrib.hh"
#include "html/align.hh"
#include "html/color.hh"
#include "html/document.hh"
#include "html/font.hh"
#include "html/hline.hh"
#include "html/image.hh"
#include "html/label.hh"
#include "html/link.hh"
#include "html/size.hh"
#include "html/style.hh"
#include "html/table.hh"
#include "html/text.hh"

//=====================================  Include PlaneMon-specific headers
#include "PlaneMonConfig.hh"

//=====================================  Include headers for basic ROOT graphics
#include "TROOT.h"
#include "TFile.h"
#include "TApplication.h"
#include "TObjArray.h"
#include "TCanvas.h"
#include "TStyle.h"
#include "TH1.h"
#include "TF1.h"
#include "TText.h"
#include "TPaveText.h"
#include "TError.h"

// #include "TMath.h"       // TMath not used for std-c++ > 199801
// #undef isnan             // TMath redefines isnan() using a cpp macro.

#ifndef __CINT__

#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/PlaneMon/PlaneMon.cc 7624 2016-05-13 08:11:59Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "PlaneMon: Airplane Detection Monitor"
#include "ProcIdent.hh"

//======================================  Generate the main routine.
EXEROOT(PlaneMon)
#endif               // !def(__CINT__)

using namespace std;

string
mic_locate(const std::string& chan) {
    const char* lox[] = {"CS", "EX", "EY", 
			 "BSC1", "BSC3", "BSC4", "BSC5", 
			 "BSC6", "BSC9", "BSC10", 0};
    const char* hloc[] = {"LVEA", "ENDX", "ENDY",
			  "LVEA_BSC1", "LVEA_BSC3", "unknown", "MIDX", 
			  "MIDY", "ENDX", "ENDY", "unknown"};
    const char* lloc[] = {"LVEA", "ENDX", "ENDY",
			  "LVEA_BSC1", "LVEA_BSC3", "ENDX", "ENDY", 
			  "unknown",  "unknown", "unknown", "unknown"};
    string::size_type start_inx = chan.find("-");
    string::size_type end_inx = chan.find("_MIC");
    if (start_inx == string::npos || end_inx == string::npos) return "unknown";
    start_inx++;
    if (end_inx <= start_inx) return "unknown";
    int iloc = 0;
    while (lox[iloc] && chan.substr(start_inx, end_inx-start_inx)!=lox[iloc]) {
	iloc++;
    }
    string ret;
    if (chan[0] == 'H')      ret = hloc[iloc];
    else if (chan[0] == 'L') ret = lloc[iloc];
    else                     return "unknown";
    if (iloc >= 3) return ret;
    start_inx = end_inx + 4;
    if (!iloc && chan.substr(start_inx,5) == "_LVEA") start_inx += 5;
    if (start_inx >= chan.size()) return ret;
    end_inx = chan.find("_DQ", start_inx);
    if (end_inx != chan.size() - 3) {
	ret += chan.substr(start_inx);
	return ret;
    }
    ret += chan.substr(start_inx, end_inx - start_inx);
    return ret;
}



//======================================  HTML Text
const char* const kHtmlTitle = "PlaneMon: Airplane Detection Monitor";
const char* const kHtmlDescription = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; PlaneMon detects \
       airplanes from characteristic signals in microphone channels. In order \
       to determine whether there is an airplane flying nearby or not, PlaneMon \
       looks for excess power in microphone channels. The largest of the excess \
       power is marked as an airplane data point. When enough data points in a \
       particular channel have been collected, PlaneMon calls the channel triggered. \
       With enough triggered channels, PlaneMon saves the data as an event depending \
       on the level of verbosity the user has set.";
const char* const kHtmlWarning = "Viewing these pages with Netscape 4.xx can cause data \
       to be displayed incorrectly. Use Firefox, IE, etc.";


//===========================================  Doppler fit for the Fitting routine (1/f)
Double_t fitf(Double_t *x, Double_t *par) {

    Double_t fitval = 0;

    fitval = par[0]*(1+par[1]*par[1]*(x[0]-par[2])*.0029154518950437/sqrt(par[1]*
          (x[0]-par[2])*par[1]*(x[0]-par[2])+par[3]*par[3]));

    return fitval;

}

//============================================  Doppler fit for the Fitting routine (f)
Double_t planeFit(Double_t *x, Double_t *par) {

    Double_t fitval = 0;

    fitval = par[0]*343*sqrt(par[1]*par[1]*(x[0]-par[2])*(x[0]-par[2])+par[3]*par[3])/
        (343*sqrt(par[1]*par[1]*(x[0]-par[2])*(x[0]-par[2])+par[3]*par[3])+par[1]*
         par[1]*(x[0]-par[2]));

    return fitval;
}


//======================================  PlaneMonMulti constructor.
PlaneMon::PlaneMon(int argc, const char *argv[]) 
    : DatEnv(argc, argv), 
      mAlarm("PlaneMon"), 
      syntax(false), 
      mElevAlarmOn(false), 
      mPlAlarmOn(false), 
      mArchiveEvent(false), 
      mMakejpg(false), 
      mMakegif(false), 
      mCandidate(false), 
      mDebug(false), 
      mVerbosity(3),  
      mCounter(0), 
      mArrayPointer(0), 
      mTrgdChans(0), 
      mExtraBumps(0), 
      mT(0.), 
      mChiSq(0.)
{

    //----------------------------------  Look for arguments
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
        if (argi == "-verbosity") {
	    mVerbosity = strtol(argv[++i],0,0);
	} else if (argi == "-loc") {
	    mLoc = argv[++i];
        } else if(argi == "-makejpg") {
            mMakejpg = true;
        } else if(argi == "-makegif") {
            mMakegif = true;
	} else if (argi == "-help") {
	    syntax = true;
	} else if (argi == "--help") {
	    syntax = true;
	} else if (argi == "-h") {
	    syntax = true;
	} else if (argi == "--h") {
	    syntax = true;
	} else if (argi == "-debug") {
	    mDebug = true;
	} else if (isDatEnvArg(argv[i])) {
	    i++;
	} else {
	    syntax = true;
	    cerr << "Unrecognized argument: " << argi << endl;
	}
    }

    //Don't print save file messages or canvases
    gErrorIgnoreLevel=1500;
    gROOT->SetBatch(kTRUE);

    if(mLoc.compare("H")==0) {
	sprintf(configFilename,"PlaneMon_LHO.conf");
    } else if(mLoc.compare("L")==0) {
	sprintf(configFilename,"PlaneMon_LLO.conf");
    } else {
	cerr << "Invalid location! LH0 = ""H"", LLO = ""L""" << endl;
	syntax = true;
    }


    //Set html directory to write to
    const char* htmldir = getenv("DMTHTMLOUT");
    if(htmldir && !mDebug) {
        dirloc = string(htmldir) + "/";
    } else if(mLoc.compare("H")==0 && !htmldir && !mDebug) {
        dirloc = "/data/gds_monitor_reports/PlaneMon/";
    } else if(mLoc.compare("L")==0 && !htmldir && !mDebug) {
        dirloc = "/import/space/apacheFiles/htdocs/gds/monitor_reports/PlaneMon/";
    } else if(mLoc.compare("H")==0 && mDebug) {
	dirloc = "/home/evang/public_html/PlaneMon/debug/";
    }

    //----------------------------------  Bail out if command line not correct
    if (syntax) {
        cerr << "Command syntax:" << endl;
	cerr << argv[0] 
	     << " [-verbosity <Verbosity>]  [-loc <Location (H or L)>]  [-makejpg]"
             << "  [<DatEnv-args>]"
	     << endl;
	finish();
    }
    
    //If the log file already exists, clear it and close.
    ifstream testlog((dirloc+"log.txt").c_str());
    if(testlog) {
        testlog.close();
        ofstream logfile((dirloc+"log.txt").c_str());
        logfile.close();
    } else {
        testlog.close();
    }

    //If there is no stats file, then write new header.
    ifstream teststats((dirloc+"stats.txt").c_str());
    if(teststats) {
        teststats.close();
    } else {
        ofstream statfile((dirloc+"stats.txt").c_str());
        statfile << "PlaneMon Monitor event statistics file." << endl;
        statfile << "File format:" << endl;
        statfile << "GPS time of first bump" << endl;
        statfile << "<Channel name>: <freq.(Hz)>, ";
        statfile << "<t of closest approach (at detector) (GPS sec)>, <distance of closest ";
        statfile << "approach (m)> <vel of airplane (m/s)>, <Chi-sq of fit>, ";
        statfile << "<Total bumps>, <Bumps in fit>" << endl;
        statfile << endl;
        statfile.close();
    }

    //If no epoch list exists, make a new one
    ifstream testelist((dirloc+"list.txt").c_str());
    if(testelist) {
	testelist.close();
    } else {
	ofstream elist((dirloc+"list.txt").c_str(),ios::app);
	elist.close();
    }

    //Write the verbosity to the log file.
    ofstream logfile((dirloc+"log.txt").c_str(),ios::app);
    if(mVerbosity==3) {
	logfile << "Verbosity level is QUIET" << endl;
	cout << "Verbosity is QUIET" << endl;
    } else if(mVerbosity==2) {
	logfile << "Verbosity level is SEMI-VERBOSE" << endl;
	cout << "Verbosity is SEMI-VERBOSE" << endl;
    } else if(mVerbosity==1) {
	logfile << "Verbosity level is VERBOSE" << endl;
	cout << "Verbosity is VERBOSE" << endl;
    } else {
    	cerr << "INVALID VERBOSITY LEVEL! Choose Quiet=3, Semi-verbose=2, or Verbose=1" << endl;
	syntax = true;
	finish();
    }
    logfile.close();

    //See if there is a valid config file. If there is not, exit and complain.
    ifstream testconf(configFilename);
    if(!testconf) {
    	testconf.close();
	printf("ERROR: cannot open %s. PlaneMon has crashed.",configFilename);
	syntax = true;
	finish();
    } else {
    	testconf.close();
	parameters = new PlaneMonConfig(configFilename);
    }

    //Set alarm for an airplane event
    AlarmData planeAlarm("PlaneMon","Airplane_Candidate",Interval(120),
			 7,"Strong evidence of nearby airplane.");
    planeAlarm.setWebFile("Monitors/PlaneMon/index.html");
    planeAlarm.jamFlags(AlarmData::kReTrigger);
    mAlarm.defineAlarm(planeAlarm);

    //Set alarm for elevated acoustic levels
    AlarmData elevatedAlarm("PlaneMon","Elevated_Acoustic_Noise",Interval(120),
			    4,"Elevated acoustic noise. Possible airplane nearby.");
    elevatedAlarm.setWebFile("Monitors/PlaneMon/index.html");
    elevatedAlarm.jamFlags(AlarmData::kReTrigger);
    mAlarm.defineAlarm(elevatedAlarm);

    //Initialize the arrays to be used in PlaneMon
    if(!syntax) {
        int channels = parameters->channelNumber;
        int heldData = (int)(120/(parameters->Stride));
        int dataLength = (parameters->fHigh)-(parameters->fLow);

        mChanTrgd = new bool[channels];
        mTrigOn = new bool[channels];
        mFitted = new bool[channels];

        mlastTS = new TSeries[channels];

        mfirstt0 = new Time[channels];
        mTrigSetTime = new Time[channels];

        mBumpCount = new int[channels];
        mPointer = new int[channels];

        mfirsttime = new double[channels];
        mlasttime = new double[channels];
        mCenterFreq = new double[channels];
        mCenterFreqAmp = new double[channels];
        mCenterFreqStdDev = new double[channels];
        mInvCenterFreqStdDev = new double[channels];
        xarray = new double*[channels];
        yarray = new double*[channels];
        stddevs = new double*[channels];
        invstddevs = new double*[channels];
        mTotalArray = new double**[channels];
        mTotalShape = new double**[channels];

        fittedPlots = new TH1F*[channels];
        planeFits = new TF1*[channels];
        curveInfos = new TPaveText*[channels];
        polfit = new TF1("poly_fit","pol5",0,120);
        fit = new TF1("fit",fitf,0,120,4);

        for(int i=0; i<channels; i++) {
            xarray[i] = new double[heldData];
            yarray[i] = new double[heldData];
            stddevs[i] = new double[heldData];
            invstddevs[i] = new double[heldData];
            mTotalArray[i] = new double*[heldData];
            mTotalShape[i] = new double*[heldData];
            for(int j=0; j<heldData; j++) {
                mTotalArray[i][j] = new double[dataLength];
                mTotalShape[i][j] = new double[dataLength];
            }

            
            planeFits[i] = new TF1("planeFit",planeFit,0,120,4);
            //curveInfos[i] = new TPaveText(.65,.70,.99,.99,"NDC");

        }


        //----------------------------------  set the data accessor
        getDacc().setStride(Interval( parameters->Stride )); // in before calling ProcessData

        for(int i=0; i<channels; i++) {
            getDacc().addChannel(parameters->channelNames[i]);
            mChanTrgd[i] = false;
            mTrigOn[i] = false;
            mFitted[i] = false;
            mBumpCount[i] = 0;
            mPointer[i] = 0;
            mCenterFreq[i] = 0.;
            mCenterFreqAmp[i] = 0.;
            mCenterFreqStdDev[i] = 0.;
            mInvCenterFreqStdDev[i] = 0.;
	    mfirsttime[i] = 0;
        }
    }
}

//======================================  PlaneMonMulti destructor.
PlaneMon::~PlaneMon() {
    if(!syntax) {
        int heldData = (int)(120/(parameters->Stride));

        for(int i=0; i<parameters->channelNumber; i++) {
            delete [] xarray[i];
            delete [] yarray[i];
            delete [] stddevs[i];
            delete [] invstddevs[i];

            //delete planeFits[i];
            //delete curveInfos[i];

            for(int j=0; j<heldData; j++) {
                delete [] mTotalArray[i][j];
                delete [] mTotalShape[i][j];
            }
            delete [] mTotalArray[i];
            delete [] mTotalShape[i];
        }

        delete parameters;
        delete [] mChanTrgd;
        delete [] mTrigOn;
        delete [] mFitted;
        delete [] mlastTS;
        delete [] mfirstt0;
        delete [] mTrigSetTime;
        delete [] mBumpCount;
        delete [] mfirsttime;
        delete [] mlasttime;
        delete [] mCenterFreq;
        delete [] mCenterFreqAmp;
        delete [] mCenterFreqStdDev;
        delete [] mInvCenterFreqStdDev;
        delete [] mTotalArray;
        delete [] mTotalShape;
        delete [] xarray;
        delete [] yarray;
        delete [] stddevs;
        delete [] invstddevs;
        delete [] mPointer;
        delete [] fittedPlots;
        delete [] curveInfos;
        delete [] planeFits;
        delete polfit;
        delete fit;

    }

    gROOT->SetBatch(kFALSE);
    gErrorIgnoreLevel=0;

    cout << "PlaneMon is finished" << endl;
}

//======================================  Frame processing function.
void PlaneMon::ProcessData(void) {
    const double invsqt12(1./sqrt(12.0));
    
    //Setup array parameters
    int heldData = (int)(120/(parameters->Stride));
    int dataLength = (parameters->fHigh)-(parameters->fLow);
    Time t0;

    //Get pointers to the current data. Loop over number of channels
    for(int i=0; i<parameters->channelNumber; i++) {
        mChannel = parameters->channelNames[i];
        const TSeries *tts = getDacc().refData(mChannel.c_str());
	if(!tts || tts->isEmpty()) {
	    cout << "Channel: " << mChannel << " was not found." << endl;
	    return;
    	}

    	TSeries ts = *tts;

    	//For first time stride, store the time series. Then for each following time 
	//stride, append the next second.
    	if(mCounter == 0) {
	    xarray[i][0] = 0.;    	//set the xarray to be zero
    	    mlastTS[i] = ts;		//save last tseries for overlap
    	    t0 = ts.getStartTime();
            char buf[20];
            mGpsNow = TimeStr(t0,buf,"%s");
	    if(i==0) {
		ofstream logfile((dirloc+"log.txt").c_str(),ios::app);
		logfile << mGpsNow << " Learning FFT curve shape\n";
		logfile.close();
	    }
        }
	
	if(mCounter>0) {

	    //Get current time
	    t0 = ts.getStartTime();
            char buf[20];
            mGpsNow = TimeStr(t0,buf,"%s");

            TSeries appendedTS = mlastTS[i];	//load first half of the tseries
	    appendedTS.Append(ts);		//append the most recent tseries
	    appendedTS = Hanning().apply(appendedTS);		//Window the time series
            FSpectrum psd = FSpectrum(FSeries(appendedTS)); //create power spectral density
	    mlastTS[i].Clear();		//clear the tseries in preparation for loading
            mlastTS[i] = ts;		//remember the second half of the time series
	
            //t0 = ts.getStartTime();	//t0 is the start time in the Time type

            //Reset mArrayPointer if it exceeds heldData
            if(mArrayPointer==heldData) {
    	    	mArrayPointer = 0;
            } else if(mArrayPointer>heldData) {
		mArrayPointer = 0;
		ofstream logfile((dirloc+"log.txt").c_str(),ios::app);
		logfile << mGpsNow << " mArrayPointer out of range!\n";
		logfile.close();
	    }

	    //If there is longer than one stride between the start of one and the start of
	    //the next, show an error message
	    if(t0.totalS()-mT>parameters->Stride && mT!=0.) {
                double tloss = t0.totalS()-mT;
		cerr << "[PlaneMon] DATA LOSS! " << tloss << " seconds missed." << endl;
		ofstream logfile((dirloc+"log.txt").c_str(),ios::app);
		logfile << "DATA LOSS! " << tloss << " seconds missed"<< endl;
		logfile.close();
	    }
            mT = t0.totalS();	    //get GPS seconds of t0 in the double type

            //Fill total array and shape array with data. Perform exponential
	    //averaging on the total shape array
	    double expfc = exp(-0.00833*parameters->Stride);
            double pwr = 0.;				  //total power in a bin
            for(int j=0; j<dataLength; j++) {
		pwr = psd.getSum(parameters->fLow+j,1.);  //power in 1 Hz bin into array
		
		//no average first time through
		mTotalArray[i][mArrayPointer][j] = pwr;
		if(mCounter==1) {
		    mTotalShape[i][mArrayPointer][j] = pwr;
		
		    //Now do the exponential averageing
		} else if(mCounter>1) {
		    if(mArrayPointer>0) {
			mTotalShape[i][mArrayPointer][j] = (1-expfc)*pwr
			    +expfc*mTotalShape[i][mArrayPointer-1][j];
		    } else {
			mTotalShape[i][mArrayPointer][j] = (1-expfc)*pwr
			    +expfc*mTotalShape[i][heldData-1][j];
		    }
		}
            }
	}

        //BUMP FINDING, CALCULATING CENTRAL BUMP FREQUENCY AND STANDARD DEVIATION
        if(mCounter >= heldData) {
            if(mT-mfirsttime[i]>120 && mChanTrgd[i]) {
       	        mBumpCount[i] = 0;
                mfirstt0[i] = (Time)0;
                mfirsttime[i] = 0.;
		mChanTrgd[i] = false;
	    }

	    //find the maximum frequency and weighted central frequency of each bump
	    double currentStride[dataLength];
	    double totalWeight = 0.;		//total weight variable
	    double totalsqweight = 0.;		//total weight squared
	    double upperThreshold = parameters->highThreshold;	//upper threshold of bump
	    double lowerThreshold = parameters->lowThreshold;   //lower threshold of bump
	    double weightedfreq = 0.;		//weighted central frequency
	    double freqstddev = 0.;		//weighted standard deviation
	    vector<int> binsover;		//vector of bin numbers above threshold
	
	    //Get bins that are above thresholds.
	    for(int j=0; j<dataLength; j++) {
		//Write the current stride for easy access.
		currentStride[j] = mTotalArray[i][mArrayPointer][j]/
		    mTotalShape[i][mArrayPointer][j];
	    }
	
	    //The FFTs have been "whitened" to ~1 so anytime a bin is greater than 
	    //the upper threshold we note this bin and surrounding bins that 
	    //have bins greater than the lower threshold in a temporary vector.
	    for(int j=0; j<dataLength; j++) {
		if(currentStride[j]>upperThreshold) {
		    if (j>0 && currentStride[j-1]>lowerThreshold &&
			       currentStride[j-1]<upperThreshold &&
			(binsover.empty() || binsover.back() != j-1) ) {
			binsover.push_back(j-1);
		    }
		    binsover.push_back(j);
		    if (j+1 < dataLength && 
			currentStride[j+1]>lowerThreshold &&
			currentStride[j+1]<upperThreshold) {
			binsover.push_back(j+1);
		    }
		}
	    }
	
	    int size = int(binsover.size());   // number of bins over threshold
	
	    //If there is only one bin over the threshold, give central
	    //frequency and standard deviations
	    if(size==1) {
		mCenterFreq[i] = binsover[0]+parameters->fLow+.5;
		mCenterFreqAmp[i] = currentStride[binsover[0]]-1;
		mCenterFreqStdDev[i] = invsqt12;
		mInvCenterFreqStdDev[i] = invsqt12/(mCenterFreq[i]*mCenterFreq[i]);
		
		if(mBumpCount[i]==0) {
		    mfirsttime[i] = mT;			//double type first time
		    mfirstt0[i] = t0;			//Time type first time
		}
		
		mlasttime[i] = mT;
		xarray[i][mPointer[i]] = mT;
		yarray[i][mPointer[i]] = mCenterFreq[i];
		stddevs[i][mPointer[i]] = mCenterFreqStdDev[i];
		invstddevs[i][mPointer[i]] = mInvCenterFreqStdDev[i];

		mPointer[i] += 1;		//increment array pointer
		mBumpCount[i] += 1;		//increment bump pointer
            	if(t0-mArchiveFinish<(Interval)20 && t0>mArchiveFinish) {
            	    mExtraBumps++;              //increment number of extra bumps
            	}

		//Or if there are multiple bins over
	    } else if(size>1) {
		int j = 0;
		int startbin = 0;
		int backbin = 0;
		//Compute weighted central frequency for first cluster of pixels
		while(mCenterFreq[i]==0.) {
		    if(j<size-1 && binsover[j]-binsover[j+1]==-1) {
			weightedfreq += (parameters->fLow+binsover[j])*
			    (currentStride[binsover[j]]-1);
			totalWeight += currentStride[binsover[j]]-1;
			backbin++;
		    } else if((j>0 && j<size-1 && binsover[j]-binsover[j+1]!=-1) || 
			      j==size-1) {
			weightedfreq += (parameters->fLow+binsover[j])*
			    (currentStride[binsover[j]]-1);
			totalWeight += currentStride[binsover[j]]-1;
			weightedfreq = weightedfreq/totalWeight;
			mCenterFreq[i] = weightedfreq;
			mCenterFreqAmp[i] = totalWeight;
			startbin = j-backbin;
			weightedfreq = 0.;
			totalWeight = 0.;
			backbin = 0;
		    } else {
			mCenterFreq[i] = parameters->fLow+binsover[j]+.5;
			mCenterFreqAmp[i] = currentStride[binsover[j]]-1;
			startbin = j-backbin;
			weightedfreq = 0.;
			totalWeight = 0.;
			backbin = 0;
		    }
		    j++;
		}
		//Compute next cluster and keep the larger of the two. Continue until 
		//all clusters have been computed.
		while(j<size) {
		    if(j<size-1 && binsover[j]-binsover[j+1]==-1) {
			weightedfreq += (parameters->fLow+binsover[j])*
			    (currentStride[binsover[j]]-1);
			totalWeight += currentStride[binsover[j]]-1;
			backbin++;
		    } else if((j>0 && j<size-1 && binsover[j]-binsover[j+1]!=-1) || 
			      j==size-1) {
			weightedfreq += (parameters->fLow+binsover[j])*
			    (currentStride[binsover[j]]-1);
			totalWeight += currentStride[binsover[j]]-1;
			weightedfreq = weightedfreq/totalWeight;
			if(totalWeight>mCenterFreqAmp[i]) {
			    mCenterFreqAmp[i] = totalWeight;
			    mCenterFreq[i] = weightedfreq;
			    startbin = j-backbin;
			}
			weightedfreq = 0.;
			totalWeight = 0.;
			backbin = 0;
		    } else {
			if(totalWeight>mCenterFreqAmp[i]) {
			    mCenterFreq[i] = parameters->fLow+binsover[j]+.5;
			    mCenterFreqAmp[i] = currentStride[binsover[j]]-1;
			    startbin = j-backbin;
			}
			weightedfreq = 0.;
			totalWeight = 0.;
			backbin = 0;
		    }
		    j++;
		}
		j = startbin;

		//Now compute the standard deviation for the largest cluster
		while(mCenterFreqStdDev[i]==0.) {
		    int    binj = binsover[j];
		    double valj = currentStride[binj] - 1.0;
		    double df0j = parameters->fLow + binj - mCenterFreq[i];

		    //-----------------------  Handle single bin clusters
		    if ( j == startbin && 
			(j == size-1 || binsover[j+1] != binj+1) ) {
			mCenterFreqStdDev[i] = invsqt12;
			mInvCenterFreqStdDev[i] = invsqt12/
			    (mCenterFreq[i]*mCenterFreq[i]);

		    //-----------------------  Sum non-final bins.
		    } else if(j+1 < size && binsover[j+1] == binj + 1) {
			totalWeight   += valj;
			totalsqweight += valj*valj;
			freqstddev    += valj*df0j*df0j;

		    //-----------------------  Final bin... set the stds
		    } else {
			totalWeight   += valj;
			totalsqweight += valj*valj;
			freqstddev    += valj*df0j*df0j;
			freqstddev = sqrt(totalWeight*freqstddev/
					  (totalWeight*totalWeight-totalsqweight));
			mCenterFreqStdDev[i] = freqstddev;
			mInvCenterFreqStdDev[i] = freqstddev/
			    (mCenterFreq[i]*mCenterFreq[i]);
		    }
		    j++;
		}
		
		//If this is the first bump, then note the position in the array
		//and the time in both formats
		if(mBumpCount[i]==0) {
		    mfirsttime[i] = mT;			//double type first time
		    mfirstt0[i] = t0;			//Time type first time
		}
		
		//put the bump data in arrays
		mlasttime[i] = mT;
		xarray[i][mPointer[i]] = mT;
		yarray[i][mPointer[i]] = mCenterFreq[i];
		stddevs[i][mPointer[i]] = mCenterFreqStdDev[i];
		invstddevs[i][mPointer[i]] = mInvCenterFreqStdDev[i];

		mPointer[i] += 1;		//increment array pointer
		mBumpCount[i] += 1;		//increment bump pointer
		if(t0-mArchiveFinish<(Interval)20 && t0>mArchiveFinish) {
		    mExtraBumps++;              //increment number of extra bumps
		}
		
	    }	//end if binsover.size()>1
	
	    binsover.clear();			//clear binsover vector
	
	    //if the bump reaches the end of the 2 minutes, move the bumps up
	    //and reduce the pointer and bump counter (if necessary)
	    while(mT-xarray[i][0]>120 && mPointer[i]>0) {
		for(int j=1; j<mPointer[i]; j++) {
		    xarray[i][j-1] = xarray[i][j];
		    yarray[i][j-1] = yarray[i][j];
		    stddevs[i][j-1] = stddevs[i][j];
		    invstddevs[i][j-1] = invstddevs[i][j];
		}
		mPointer[i] -= 1;
	    }
	
	    //If still fewer than 6 bumps and the last bump was more than 4 seconds 
	    //ago, reset the bump counter
	    if(mBumpCount[i]>0 && mBumpCount[i]<7 && mT-mlasttime[i]>4.) {
		mBumpCount[i] = 0;
		mChanTrgd[i] = false;
	    } else if(mBumpCount[i]>6 && !mChanTrgd[i] && mT-mfirsttime[i]>=20) {
		mBumpCount[i] = 0;
            	mChanTrgd[i] = false;
	    } else if(mBumpCount[i]>6 && !mChanTrgd[i] && mT-mfirsttime[i]<20) {
		mChanTrgd[i] = true;
	    }
	} //end if mCounter>=heldData
    } //end loop over channels

    //=================================  Process after plots are full
    //Now begin if counter is equal or greater than heldData
    if(mCounter>=heldData) {
    	//Count the number of triggered channels
	mTrgdChans = 0;
	for(int i=0; i<parameters->channelNumber; i++) {

	    //Trigger is off if 120 seconds has passed
            if(t0-mTrigSetTime[i]>(Interval)120 && !mChanTrgd[i]) {
		mTrigOn[i] = false;
	    }

	    //add one to the total number of triggered channels
	    if(mChanTrgd[i]) {
		mTrgdChans++;
	    }
	
	    //Generate triggers for active channels
	    if(mChanTrgd[i] && !mTrigOn[i]) {
		trig::TrigRslt tr("PlaneMon",parameters->channelNames[i]);
		tr.setTime(mfirstt0[i]);
		tr.setDuration((Interval)120);
		tr.setDisposition(trig::d_metaDB);
		tr.setPriority(trig::p_error);
		sendTrigger(tr);
		mTrigOn[i] = true;
		mTrigSetTime[i] = mfirstt0[i];
		char buff[20];
                ofstream logfile((dirloc+"log.txt").c_str(),ios::app);
		logfile << TimeStr(mfirstt0[i],buff,"%s") 
			<< " Sending trigger for channel " 
			<< parameters->channelNames[i] << endl;
                logfile.close();
	    }
	}

	//====================================  More data
        //  If there are more data points because a plane event lasts longer 
	//  than 2 minutes, then find the new time and archive the data again
	Time eventt0 = (Time)0;                              //temporary time storage

	if(mTrgdChans>=mVerbosity && !mArchiveEvent && t0-mArchiveFinish>(Interval)60) {
	    for(int i=0; i<parameters->channelNumber; i++) {
                if(eventt0==(Time)0 && mfirstt0[i]!=(Time)0 && mChanTrgd[i]) {
                    eventt0 = mfirstt0[i];
                } else if(mfirstt0[i]<eventt0 && mfirstt0[i]!=(Time)0 && mChanTrgd[i]) {
                    eventt0 = mfirstt0[i];
                }
            }

	    mArchiveEvent = true;
	    mArchiveFinish = eventt0+(Interval)120;
	    mArchiveTime = eventt0;
	} else if(!mArchiveEvent && mExtraBumps>15 && mTrgdChans>0) {
	    for(int i=0; i<parameters->channelNumber; i++) {
                if(eventt0==(Time)0 && mfirstt0[i]!=(Time)0 && mChanTrgd[i]) {
                    eventt0 = mfirstt0[i];
                } else if(mfirstt0[i]<eventt0 && mfirstt0[i]!=(Time)0 && mChanTrgd[i]) {
                    eventt0 = mfirstt0[i];
                }
            }

	    mArchiveEvent = true;
	    mArchiveFinish = eventt0+(Interval)60;
	    mArchiveTime = eventt0;

	    for(int i=0; i<parameters->channelNumber; i++) {
		if(eventt0==mfirstt0[i]) {
		    mBumpCount[i] = 0;
		    mfirstt0[i] = (Time)0;
		    mfirsttime[i] = 0.;
		    mChanTrgd[i] = false;
		}
	    }
	} else if(mTrgdChans==0 && t0-mArchiveFinish>=(Interval)20) {
	    mArchiveEvent = false;
	    mExtraBumps = 0;
	}


	//===========================================================Alarm flags
	//After 119 seconds, set Elevated or Plane alarm flag off
	if(t0-mElevAlarmSetTime>(Interval)120) {
	    mElevAlarmOn = false;
	}
	if(t0-mPlAlarmSetTime>(Interval)120) {
	    mPlAlarmOn = false;
	}

	//===========================================================Missed Log file
	//Make a log of any events where 2 or more channels are triggered, in case 
	//PlaneMon misses something, it can be checked later.
	if(mTrgdChans>=2 && !(mElevAlarmOn || mPlAlarmOn)) {
            for(int i=0; i<parameters->channelNumber; i++) {
                if(eventt0==(Time)0 && mfirstt0[i]!=(Time)0 && mChanTrgd[i]) {
                    eventt0 = mfirstt0[i];
                } else if(mfirstt0[i]<eventt0 && mfirstt0[i]!=(Time)0 && mChanTrgd[i]) {
                    eventt0 = mfirstt0[i];
                }
            }

            //Write number of channels triggered and the time to the 
            //missed events log
	    ofstream eventlog((dirloc+"missedLog.txt").c_str(),ios::app);
	    eventlog << mGpsNow << " " << mTrgdChans << " channels triggered"<< "\n";
            eventlog.close();
	}

	//============================================================Alarms
	//Generate alarms for airplane or elevated levels if alarms are off
	AlarmHandle han;
	if(mTrgdChans==2 && !mElevAlarmOn && !mPlAlarmOn) {
	    AlarmData al("PlaneMon","Elevated_Acoustic_Noise",0,4,
			 "Elevated acoustic noise. Possible airplane nearby.","");
	    lmsg::error_t rc = mAlarm.setAlarm(al,han);
	    if(rc) {
		cout << "[PlaneMon] sending alarm: " << rc << endl;
	    }
	    mElevAlarmOn = true;
	    mElevAlarmSetTime = t0;
            ofstream logfile((dirloc+"log.txt").c_str(),ios::app);
	    logfile << mGpsNow << " Sending elevated alarm." << endl;
            logfile.close();
	} else if(mTrgdChans>2 && !mPlAlarmOn) {
	    AlarmData al("PlaneMon","Airplane_Candidate",0,7,
			 "Strong evidence of nearby airplane.","");
	    lmsg::error_t rc = mAlarm.setAlarm(al,han);
	    if(rc) {
		cout << "[PlaneMon] sending alarm: " << rc << endl;
	    }
	    mPlAlarmOn = true;
	    mPlAlarmSetTime = t0;
            ofstream logfile((dirloc+"log.txt").c_str(),ios::app);
	    logfile << mGpsNow << " Sending plane candidate alarm." << endl;
            logfile.close();
	}
	
	//Generate summary and current plots webpages every 15 iterations
	if(mCounter%15==0) {
    	    Summary(t0);               //Make summary page

    	    //summary page done
	    ofstream logfile((dirloc+"log.txt").c_str(),ios::app);
            logfile << setprecision (9) << mT << " Summary Done. ";
            logfile.close();

	    CurrentPlots(t0);          //Make current plots
	
	    //Write number of channels triggered and which channels are 
	    //triggered to the detailed log file.
	    ofstream logf((dirloc+"log.txt").c_str(),ios::app);
    	    logf << "Plots Updated. mTrgdChans=" << mTrgdChans << " Chans Trgd: ";
    	    logf.close();
	    for(int i=0; i<parameters->channelNumber; i++) {
		if(mChanTrgd[i]) {
		    ofstream logfi((dirloc+"log.txt").c_str(),ios::app);
		    logfi << "YES ";
		    logfi.close();
		} else {
		    ofstream logfil((dirloc+"log.txt").c_str(),ios::app);
		    logfil << "NO ";
		    logfil.close();
		}
		if(i==parameters->channelNumber-1) {
		    ofstream logs((dirloc+"log.txt").c_str(),ios::app);
		    logs << endl;
		    logs.close();
		}
	    }
	}

	//=====================================  Archiving Events
	if(t0>=mArchiveFinish && mArchiveEvent) {
	    mExtraBumps = 0;
	    mArchiveEvent = false;

	    char buff[20];
            mGpsEvent = TimeStr(mArchiveTime,buff,"%s");
	    mFoldername = TimeStr(mArchiveTime,buff,"%s");
	    string epoch = LocalStr(mArchiveTime,buff,"%02m_%Y");
	        

	    //Open the list of epochs.
	    ifstream infile((dirloc+"list.txt").c_str());
	    string epoch_dir = dirloc + epoch;

	    //If the epoch is not found, then create a new epoch. 
	    //This makes a new folder and a new archive list in the 
	    //new epoch folder.
            if (access(epoch_dir.c_str(), X_OK)) {
               	mEpoch = epoch;
               	if (mkdir(epoch_dir.c_str(), 0755)) {
		    cerr << "failed to make directory " << epoch_dir << endl;
		    epoch_dir=".";
		}

               	ofstream outputfile((epoch_dir+"/list.txt").c_str());
               	outputfile.close();

               	ofstream outfile((dirloc+"list.txt").c_str(),ios::app);
               	outfile << mEpoch << "\n";
               	outfile.close();
                   
               	//Construct the html document object archived events
               	html::document doc ("PlaneMon: Archived Events");

               	//Title
                html::block titleblk ("center");
                titleblk.lineBreak();
                html::text title("PlaneMon: Archived Events");
                title.setFont(html::font ("Helvetica"));
                title.setColor(html::color (0,0,128));
                title.setSize(html::size (+4));
                titleblk.addObject(title);
                titleblk.lineBreak();
                titleblk.addObject(html::text("&nbsp;"));
                doc.addObject(titleblk);
                doc.addObject(html::hline());
                doc.lineBreak();
                doc.lineBreak();

                html::block eventsblk ("left");
                ifstream infile((dirloc+"list.txt").c_str());
                while(infile.good()) {
		    string path;
		    getline(infile, path);
		    if (!path.empty()) {
                       	eventsblk.addObject(
				html::link(path, path+"/archivelist.html"));
                       	eventsblk.lineBreak();
		    }
                }
                infile.close();
                doc.addObject(eventsblk);

                //Write archived events webpage to html file
                ofstream out ((dirloc+"eventarchive.html").c_str());
                if(out) {
		    html::writer htmlout (out);
		    doc.write (htmlout);
		    out.close();
                }
            } else {
               	mEpoch = epoch;
            }

            ofstream eventout((dirloc+mEpoch+"/list.txt").c_str(),ios::app);
            eventout << mFoldername << "\n";
            eventout.close();

	    system(("mkdir -m 755 "+dirloc+mEpoch+"/"+mFoldername).c_str());
    
	    //Write a new line in the stats file
	    ofstream stats((dirloc+"stats.txt").c_str(),ios::app);
	    stats << "\n" << mGpsEvent << "\n";
	    stats.close();

	    TFile file((dirloc+mEpoch+"/"+mFoldername+"/chanHists.root").c_str(),"RECREATE");
	    /*fittedPlots = new TH1F*[parameters->channelNumber];
            cout << "fittedPlots created at location " << fittedPlots << endl;
	    planeFits = new TF1*[parameters->channelNumber];
            cout << "planeFits created at location " << planeFits << endl;
	    curveInfos = new TPaveText*[parameters->channelNumber];
            cout << "curveInfos created at location " << curveInfos << endl;*/
            if (Debug()) cout << "mChanTrgd has memory location " 
			      << &mChanTrgd << endl;
	    for(int i=0; i<parameters->channelNumber; i++) {
                if (Debug()) cout << "Checking channel i = " << i 
				  << " at memory location " << &i 
				  << " with mChanTrgd at " << &mChanTrgd[i] 
				  << endl;
		if(mChanTrgd[i]) {
                    if (Debug()) 
			cout << "Channel i = " << i 
			     << " with memory location " << &mChanTrgd[i] 
			     << " is triggered. Executing fit subroutine" 
			     << endl;
		    PlaneMon::fitcurve(i,t0);        //Execute fitting on triggered chans
		} 
                if (Debug()) cout << "Checked channel i = " << i 
				  << " at memory location " << &i 
				  << " with mChanTrgd at " << &mChanTrgd[i] 
				  << endl;
	    }
	    archiveEvent(mArchiveTime,t0);      //Archive the event

            PlaneMon::genSegment(mTrigC);       //Generate a segment

	    mArchiveEvent = false;              //reset archive event flag
	    mCandidate = false;                  //reset candidate event flag
	    mTimeBuffer = t0+(Interval)60;      //set new time buffer value
	    file.Close();
	    for(int i=0; i<parameters->channelNumber; i++) {
		if(mChanTrgd[i] && mFitted[i]) {
		    mFitted[i] = false;      //reset fitted flag

                    if (Debug()) cout << "2 Deleting planeFits[" << i 
				      << "] at memory location " 
				      << planeFits[i] << endl; 
		    delete planeFits[i];     //delete the fits 

                    if (Debug()) cout << "Deleting curveInfos[" << i 
				      << "] at memory location " 
				      << curveInfos[i] << endl;
		    delete curveInfos[i];    //delete curve infos
		}
	    }

            /* cout << "Deleting fittedPlots, planeFits and curveInfos at locations << fittedPlots << ", " << planeFits << " and " << curveInfos << endl;
	    delete [] fittedPlots;
	    delete [] planeFits;
	    delete [] curveInfos;*/
			
	}
	if(t0-mArchiveFinish>=(Interval)20) {
	    mExtraBumps = 0;
	}


	//=================================================Reset bump parameters
	//reset parameters to zero
	for(int i=0; i<parameters->channelNumber; i++) {
	    mCenterFreq[i] = 0.;
	    mCenterFreqAmp[i] = 0;
	    mCenterFreqStdDev[i] = 0.;
	    mInvCenterFreqStdDev[i] = 0.;
	}
    }  //End if mCounter is at least heldData

    mCounter++;
    mArrayPointer++;		//increment array pointer

}


//===========================================  Write summary to an html file.
void PlaneMon::Summary(const Time& t) {
    
    //Construct the html document object
    html::document doc (kHtmlTitle);
    char buf[128];
    
    //Title
    html::block titleblk ("center");
    titleblk.lineBreak();
    html::text title (kHtmlTitle);
    title.setFont (html::font ("Helvetica"));
    title.setColor (html::color (0,0,128));
    title.setSize (html::size (+4));
    titleblk.addObject(title);
    titleblk.lineBreak();
    titleblk.addObject(html::text ("&nbsp;"));
    doc.addObject(titleblk);
    doc.addObject(html::hline());
    
    //Time updated
    html::block updateblk ("center");
    html::text lasttime ("This page was last updated: ");
    string timeZone;
    if(strncmp(parameters->channelNames[0],"H",1)==0) {
	timeZone = "PT";
    } else {
	timeZone = "CT";
    }
    lasttime << LocalStr(t,buf,"%M %d, %Y at %H:%N:%S");
    lasttime << " " << timeZone << " " << LocalStr(t,buf,"(GPS=%s)");
    updateblk.addObject(lasttime);
    updateblk.lineBreak();
    html::text nexttime ("Next update: ");
    nexttime << LocalStr(t+(Interval)15,buf,"%M %d, %Y at %H:%N:%S");
    nexttime << " " << timeZone << " " << LocalStr(t+(Interval)15,buf,"(GPS=%s)");
    updateblk.addObject(nexttime);
    updateblk.lineBreak();
    updateblk.lineBreak();
    html::text warning (kHtmlWarning);
    warning.setColor(html::color (128,0,0));
    warning.setSize(html::size (+1));
    updateblk.addObject(warning);
    doc.addObject(updateblk);
    
    //Current Status
    html::block statusblk ("left");
    statusblk.lineBreak();
    html::text statustitle ("Global Status: ");
    statustitle.setSize(html::size (+2));
    statusblk.addObject(statustitle);
    if(mTrgdChans<2) {
    	html::text status ("Quiet");
	status.setSize(html::size (+2));
	status.setColor(html::color (0,128,0));
	statusblk.addObject(status);
    } else if(mTrgdChans==2) {
    	html::text status ("Elevated");
	status.setSize(html::size (+2));
	status.setColor(html::color (0,0,255));
	statusblk.addObject(status);
    } else if(mTrgdChans>2) {
    	html::text status ("Airplane Candidate");
	status.setSize(html::size (+2));
	status.setColor(html::color (255,0,0));
	statusblk.addObject(status);
    }
    statusblk.lineBreak();
    html::text localstatustitle ("Local Status:");
    localstatustitle.setSize(html::size (+2));
    statusblk.addObject(localstatustitle);
    statusblk.lineBreak();

    for(int i=0; i<parameters->channelNumber; i++) {
        string chname = parameters->channelNames[i];
        string name = mic_locate(chname);
    	html::text channelname ("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;");
	channelname << name;
	channelname << " (";
	channelname << parameters->channelNames[i];
	channelname << "): ";
	channelname.setSize(html::size (+1));
	statusblk.addObject(channelname);
	
	if(mPointer[i]<=6){
	    html::text localstatus ("Quiet");
	    localstatus.setColor(html::color(0,128,0));
	    localstatus.setSize(html::size (+1));
	    statusblk.addObject(localstatus);
	} else if(mPointer[i]>6 && mBumpCount[i]<7){
	    html::text localstatus ("Elevated");
	    localstatus.setColor(html::color(0,0,255));
	    localstatus.setSize(html::size (+1));
	    statusblk.addObject(localstatus);
	} else {
	    html::text localstatus ("Activity");
	    localstatus.setColor(html::color(255,0,0));
	    localstatus.setSize(html::size (+1));
	    statusblk.addObject(localstatus);
	}
	statusblk.lineBreak();
	statusblk.lineBreak();
    }
    doc.addObject(statusblk);
    
    //Current plots link
    html::block links ("center");
    html::link currentplots ("Current Plots (last 2 minutes)","currentplots.html");
    html::link prevevents ("Events Archive","eventarchive.html");
    html::link statspage ("Monitor Statistics File","stats.txt");
    html::link missedlog ("Semi-verbose Logfile","missedLog.txt");
    html::link log("Monitor Detail Logfile","log.txt");
    links.addObject(currentplots);
    links.lineBreak();
    links.addObject(prevevents);
    links.lineBreak();
    links.addObject(statspage);
    links.lineBreak();
    links.addObject(missedlog);
    links.lineBreak();
    links.addObject(log);
    links.lineBreak();
    doc.addObject(links);
    doc.addObject(html::hline());
    
    //Documentation: summary, indicators and further info link
    html::block document ("left");
    html::text overviewttle ("Overview of PlaneMon");
    overviewttle.setSize(html::size (+2));
    document.addObject(overviewttle);
    document.lineBreak();
    html::text overview (kHtmlDescription);
    overview.setSize(html::size (+1));
    document.addObject(overview);
    document.lineBreak();
    document.lineBreak();
    html::text indicttle ("PlaneMon Status Indicators");
    indicttle.setSize(html::size (+2));
    document.addObject(indicttle);
    document.lineBreak();
    html::table indictbl;
    indictbl.addColumn("Indicator");
    indictbl.addColumn("Status");
    indictbl.addColumn("Meanings");
    indictbl.addRow();
    html::table statuses;
    statuses.addColumn("Quiet");
    statuses.addRow();
    statuses.addRow();
    statuses.insertData(0,0,html::text ("Elevated"));
    statuses.insertData(1,0,html::text ("Airplane Candidate"));
    html::table meanings;
    meanings.addColumn("Fewer than 2 channels triggered with activity");
    meanings.addRow();
    meanings.addRow();
    meanings.insertData(0,0,html::text ("Exactly 2 channels triggered with \
       activity"));
    meanings.insertData(1,0,html::text ("More than 2 channels triggered with \
       activity"));
    indictbl.insertData(0,0,html::text ("Global Status"));
    indictbl.insertData(0,1,statuses);
    indictbl.insertData(0,2,meanings);
    html::table locstatuses;
    locstatuses.addColumn("Quiet");
    locstatuses.addRow();
    locstatuses.addRow();
    locstatuses.insertData(0,0,html::text ("Elevated"));
    locstatuses.insertData(1,0,html::text ("Activity"));
    html::table locmeanings;
    locmeanings.addColumn("Untriggered. Six or fewer bumps in past 2 minutes.");
    locmeanings.addRow();
    locmeanings.addRow();
    locmeanings.insertData(0,0,html::text ("Untriggered. More than 6 bumps but \
       less than 7 localized bumps."));
    locmeanings.insertData(1,0,html::text ("Triggered. More than 6 localized bumps."));
    indictbl.addRow();
    indictbl.insertData(1,0,html::text ("Local Status"));
    indictbl.insertData(1,1,locstatuses);
    indictbl.insertData(1,2,locmeanings);
    indictbl.setBorder(true);
    document.addObject(indictbl);
    document.lineBreak();
    document.lineBreak();
    html::text addinfotle ("Additional Information");
    addinfotle.setSize(html::size (+2));
    document.addObject(addinfotle);
    document.lineBreak();
    document.addObject(html::link ("Documentation",
				   "../../dmt/Monitors/PlaneMon/PlaneMonDoc.pdf"));
    document.lineBreak();
    document.addObject(html::link ("FAQ","FAQ.html"));
    doc.addObject(document);
    doc.lineBreak();
    
    //Author info
    doc.addObject (html::hline());
    html::block authors ("center");
    authors.addObject(html::text ("This monitor was written by "));
    authors.addObject(html::link ("Evan Goetz","mailto:egoetz@umich.edu"));
    authors.addObject(html::text (" and "));
    authors.addObject(html::link ("Keith Riles","mailto:kriles@umich.edu"));
    authors.lineBreak();
    doc.addObject(authors);
    
    //Write webpage to file
    ofstream out ((dirloc+"index.html").c_str());
    if(out) {
    	html::writer htmlout (out);
	doc.write(htmlout);
	out.close();
    }
}


//===========================================  Write current plots to an html file.
void PlaneMon::CurrentPlots(const Time& t) {
    //Construct the html document object
    html::document doc ("PlaneMon: Current Plots");
    char buf[128];
    
    //Title
    html::block titleblk ("center");
    titleblk.lineBreak();
    html::text title("PlaneMon: Plots over past two minutes");
    title.setFont(html::font ("Helvetica"));
    title.setColor(html::color (0,0,128));
    title.setSize(html::size (+4));
    titleblk.addObject(title);
    titleblk.lineBreak();
    titleblk.addObject(html::text("&nbsp;"));
    doc.addObject(titleblk);
    doc.addObject(html::hline());
    
    //Time updated
    html::block updateblk ("center");
    html::text lasttime("This page was last updated: ");
    string timeZone;
    if(strncmp(parameters->channelNames[0],"H",1)==0) {
	timeZone = "PT";
    } else {
	timeZone = "CT";
    }
    lasttime << LocalStr(t,buf,"%M %d, %Y at %H:%N:%S");
    lasttime << " " << timeZone << " " << LocalStr(t,buf,"(GPS=%s)");
    updateblk.addObject(lasttime);
    updateblk.lineBreak();
    html::text firsttime("Plots start at initial time: ");
    firsttime << LocalStr((t-(Interval)120),buf,"%M %d, %Y at %H:%N:%S");
    firsttime << " " << timeZone << " " << LocalStr((t-(Interval)120),buf,"(GPS=%s)");
    updateblk.addObject(firsttime);
    updateblk.lineBreak();
    html::text nexttime ("Next update: ");
    nexttime << LocalStr(t+(Interval)15,buf,"%M %d, %Y at %H:%N:%S");
    nexttime << " " << timeZone << " " << LocalStr(t+(Interval)15,buf,"(GPS=%s)");
    updateblk.addObject(nexttime);
    updateblk.lineBreak();
    updateblk.lineBreak();
    html::text warning (kHtmlWarning);
    warning.setColor(html::color (128,0,0));
    warning.setSize(html::size (+1));
    updateblk.addObject(warning);
    updateblk.lineBreak();
    doc.addObject(updateblk);
    
    //Plots
    html::block plotsblk ("left");
    html::table plottbl;
    plottbl.addColumn("Channel");
    plottbl.addColumn("Time vs Frequency trajectories");
    plottbl.addColumn("Status");
    html::text quiet ("Quiet");
    html::text activity ("Activity");
    html::text elevated ("Elevated");
    quiet.setColor(html::color(0,128,0));
    activity.setColor(html::color(255,0,0));
    elevated.setColor(html::color(0,0,128));
    
    //Make new plots for the current plots page
    plots = new TH1F*[parameters->channelNumber];
    for(int i=0; i<parameters->channelNumber; i++) {
	// -- channel name and location.
	string chname=parameters->channelNames[i];
        string name = mic_locate(chname);
    
    	plottbl.addRow();
	plottbl.insertData(i,0,html::text(name));
	
	//new canvas for the plot
	canvas = new TCanvas("canvas","Output Canvas",500,350);
	
	plots[i] = bumpPlot(i, name);
        plots[i]->Draw("E1P");
	//Write jpg and pdf, eps and gif or just eps.
        if(mMakejpg) {
	    canvas->Print((dirloc+name+".jpg").c_str(),"jpg");
	    canvas->Print((dirloc+name+".pdf").c_str(),"pdf");
        } else if(mMakegif) {
            canvas->Print((dirloc+name+".eps").c_str(),"eps");
            system(("pstopnm -ppm -xborder 0 -yborder 0 -portrait "+dirloc+name+".eps").c_str());
            system(("ppmtogif "+dirloc+name+".eps001.ppm > "+dirloc+name+".gif").c_str());
        } else {
            canvas->Print((dirloc+name+".eps").c_str(),"eps");
        }

	html::image newTFplot;
        if(mMakejpg) {
	    newTFplot.setSource(("./"+name+".jpg").c_str());
        } else {
            newTFplot.setSource((dirloc+name+".gif").c_str());
        }
	newTFplot.setWidth("250");
	html::link imglnk;
        if(mMakejpg) {
	    imglnk.setAddr((name+".jpg").c_str());
        } else if(mMakegif) {
            imglnk.setAddr((name+".gif").c_str());
        }
	imglnk.addObject(newTFplot);
	plottbl.insertData(i,1,imglnk);
        if(mMakejpg) {
	    plottbl.insertData(i, 1, html::link("Click for PDF", name+".pdf"));
        } else {
            plottbl.insertData(i, 1, html::link("Click for EPS", name+".eps"));
        }

	if(mPointer[i]<=6){
	    plottbl.insertData(i,2,quiet);
	} else if(mBumpCount[i]<7 && mPointer[i]>6){
	    plottbl.insertData(i,2,elevated);
	} else {
	    plottbl.insertData(i,2,activity);
	}
	delete canvas;         //delete the canvas image of the plot
	delete plots[i];       //delete the t-f plot
    }
    delete [] plots;           //delete all t-f plots

    plottbl.setBorder(true);
    plotsblk.addObject(plottbl);
    doc.addObject(plotsblk);

    //Write webpage to html file
    ofstream out ((dirloc+"currentplots.html").c_str());
    if(out) {
    	html::writer htmlout (out);
	doc.write (htmlout);
	out.close();
    }

}


//======================================  Archive Events
void PlaneMon::archiveEvent(const Time& t, const Time& tup) {

    if (Debug()) cout << "Starting archive routine" << endl;

    char buf[128];
    string pagename = LocalStr(t,buf,"%02m_%Y");

    if (Debug()) cout << "Generating HTML page" << endl;

    //Construct the html document object archived events
    html::document doc (("PlaneMon: "+pagename+" Archived Events").c_str());

    //Title
    if (Debug()) cout << "Title" << endl;
    html::block titleblk ("center");
    titleblk.lineBreak();
    html::text title(("PlaneMon: "+pagename+" Archived Events").c_str());
    title.setFont(html::font ("Helvetica"));
    title.setColor(html::color (0,0,128));
    title.setSize(html::size (+4));
    titleblk.addObject(title);
    titleblk.lineBreak();
    titleblk.addObject(html::text("&nbsp;"));
    doc.addObject(titleblk);
    doc.addObject(html::hline());

    //Time updated
    if (Debug()) cout << "Time" << endl;
    html::block updateblk ("center");
    html::text lasttime ("This page was last updated: ");
    string timeZone;
    if(strncmp(parameters->channelNames[0],"H",1)==0) {
	timeZone = "PT";
    } else {
	timeZone = "CT";
    }
    lasttime << LocalStr(tup,buf,"%M %d, %Y at %H:%N:%S");
    lasttime << " " << timeZone << " " << LocalStr(tup,buf,"(GPS=%s)");
    updateblk.addObject(lasttime);
    updateblk.lineBreak();
    updateblk.lineBreak();
    doc.addObject(updateblk);

    //Make new plots for the web archive.
    if (Debug()) cout << "Running through the plots" << endl;
    plots = new TH1F*[parameters->channelNumber];
    for(int i=0; i<parameters->channelNumber; i++) {
	string chname = parameters->channelNames[i];
        string name = mic_locate(chname);

        //make new plot picture for the web archive
	canvas = new TCanvas("canvas","Output Canvas",500,350);
	if(!mChanTrgd[i]) {
	    plots[i] = bumpPlot(i, name);
	    plots[i]->DrawCopy("E1P");
	    plots[i]->Write(name.c_str());
	} else {
	    fittedPlots[i]->DrawCopy("E1P");
	    fittedPlots[i]->Write(name.c_str());
	    if(mFitted[i]) {
		planeFits[i]->DrawCopy("SAME");
		curveInfos[i]->Draw("SAME");
	    }
	}
        if (Debug()) cout << "Making image i " << i << endl;
    	if(mMakejpg) {
	    canvas->Print((dirloc+mEpoch+"/"+mFoldername+"/"+name+".jpg").c_str(),"jpg");
	    canvas->Print((dirloc+mEpoch+"/"+mFoldername+"/"+name+".pdf").c_str(),"pdf");
    	} else if(mMakegif) {
	    canvas->Print((dirloc+mEpoch+"/"+mFoldername+"/"+name+".eps").c_str(),"eps");
	    system(("pstopnm -ppm -xborder 0 -yborder 0 -portrait "+dirloc+mEpoch+"/"+
		    mFoldername+"/"+name+".eps").c_str());
	    system(("ppmtogif "+name+".eps001.ppm > "+dirloc+mEpoch+"/"+mFoldername+"/"+
		    name+".gif").c_str());
    	} else {
	    canvas->Print((dirloc+mEpoch+"/"+mFoldername+"/"+name+".eps").c_str(),"eps");
    	}
	delete canvas;      //delete the canvas for the next picture.
	if(!mChanTrgd[i]) {
	    delete plots[i];
	} else {
            if (Debug()) cout << "Deleting fittedPlots[" << i 
			      << "] at memory location " << fittedPlots[i] 
			      << endl;
	    delete fittedPlots[i];
	}
    }
    delete [] plots;        //delete the t-f plots

    //Read in list file and make links on the archived events page.
    html::block eventsblk ("left");
    ifstream infile((dirloc+mEpoch+"/list.txt").c_str());
    while (infile.good()) {
	string path;
	getline(infile, path);
	if (!path.empty()) {
	    eventsblk.addObject(html::link(path, path + "/"));
	    eventsblk.lineBreak();
	}
    }
    infile.close();
    doc.addObject(eventsblk);

    //Write archived events webpage to html file
    ofstream out ((dirloc+mEpoch+"/archivelist.html").c_str());
    if(out) {
    	html::writer htmlout (out);
	doc.write (htmlout);
        out.close();
    }
    //===========================End making events archive page list

    //Make index.html page for particular event
    if (Debug()) cout << "Making index.html page" << endl;
    html::document archiveIndex (("PlaneMon: Event "+mFoldername).c_str());
    html::block archtitleblk ("center");
    archtitleblk.lineBreak();
    html::text archtitle(("PlaneMon: Event "+mFoldername).c_str());
    archtitle.setFont(html::font ("Helvetica"));
    archtitle.setColor(html::color (0,0,128));
    archtitle.setSize(html::size (+4));
    archtitleblk.addObject(archtitle);
    archtitleblk.lineBreak();
    archtitleblk.addObject(html::text("&nbsp;"));
    archiveIndex.addObject(archtitleblk);
    archiveIndex.addObject(html::hline());

    html::block archupdateblk ("center");
    archupdateblk.addObject(lasttime);
    archupdateblk.lineBreak();
    html::text firsttime("Plots start at initial time: ");
    firsttime << LocalStr((tup-(Interval)120),buf,"%M %d, %Y at %H:%N:%S");
    firsttime << " " << timeZone << " " << LocalStr((tup-(Interval)120),buf,"(GPS=%s)");
    archupdateblk.addObject(firsttime);
    archupdateblk.lineBreak();
    archupdateblk.lineBreak();

    archupdateblk.addObject(html::link(mFoldername + ": trajectory", 
				       "./traj.html"));
    archiveIndex.addObject(archupdateblk);

    html::block plotsblk ("left");
    plotsblk.lineBreak();
    plotsblk.lineBreak();
    html::table plottbl;
    plottbl.addColumn("Channel");
    plottbl.addColumn("Time vs Frequency trajectories");
    plottbl.addColumn("Status");
    html::text quiet ("Quiet");
    html::text activity ("Activity");
    html::text elevated ("Elevated");
    quiet.setColor(html::color(0,128,0));
    activity.setColor(html::color(255,0,0));
    elevated.setColor(html::color(0,0,128));
    if (Debug()) cout << "Running through the channels again" << endl;
    for(int i=0; i<parameters->channelNumber; i++) {
	string chname = parameters->channelNames[i];
        string name = mic_locate(chname);

        plottbl.addRow();
        plottbl.insertData(i,0,html::text(name));
	
	html::image newTFplot;
        if(mMakejpg) {
            newTFplot.setSource(("./"+name+".jpg").c_str());
        } else {
            newTFplot.setSource(("./"+name+".gif").c_str());
        }
        newTFplot.setWidth("250");
        html::link imglnk;
        if(mMakejpg) {
            imglnk.setAddr((name+".jpg").c_str());
        } else {
            imglnk.setAddr((name+".gif").c_str());
        }
        imglnk.addObject(newTFplot);
        plottbl.insertData(i,1,imglnk);
        if(mMakejpg) {
            plottbl.insertData(i,1,html::link("Click for PDF", name+".pdf"));
        } else {
            plottbl.insertData(i,1,html::link("Click for EPS", name+".eps"));
        }
	
        if(mPointer[i]<=6){
            plottbl.insertData(i,2,quiet);
        } else if(mBumpCount[i]<7 && mPointer[i]>6){
            plottbl.insertData(i,2,elevated);
        } else {
            plottbl.insertData(i,2,activity);
        }
    }
    plottbl.setBorder(true);
    plotsblk.addObject(plottbl);
    archiveIndex.addObject(plotsblk);
    ofstream archiveout((dirloc+mEpoch+"/"+mFoldername+"/index.html").c_str());
    if(archiveout) {
        html::writer htmlout (archiveout);
        archiveIndex.write (htmlout);
        archiveout.close();
    }

    //Write a temporary file to */traj.html where * is the event time.
    html::document eventTraj (("PlaneMon: Event "+mFoldername+" Trajectory").c_str());
    html::block trajtitleblk ("center");
    trajtitleblk.lineBreak();
    html::text trajtitle(("PlaneMon: Event "+mFoldername+" Trajectory").c_str());
    trajtitle.setFont(html::font ("Helvetica"));
    trajtitle.setColor(html::color (0,0,128));
    trajtitle.setSize(html::size (+4));
    trajtitleblk.addObject(trajtitle);
    trajtitleblk.lineBreak();
    trajtitleblk.addObject(html::text("&nbsp;"));
    eventTraj.addObject(trajtitleblk);
    eventTraj.addObject(html::hline());

    html::block trajupdateblk ("center");
    trajupdateblk.addObject(lasttime);
    trajupdateblk.lineBreak();
    trajupdateblk.addObject(firsttime);
    trajupdateblk.lineBreak();
    trajupdateblk.lineBreak();
    eventTraj.addObject(trajupdateblk);

    //html::block plotsblk ("left");
    html::block fitplotsblk ("center");
    html::text none("Plots and Trajectory are not yet available");
    none.setFont(html::font ("Helvetica"));
    none.setColor(html::color (128,0,0));
    none.setSize(html::size (+3));
    fitplotsblk.addObject(none);
    fitplotsblk.lineBreak();
    eventTraj.addObject(fitplotsblk);

    ofstream traj((dirloc+mEpoch+"/"+mFoldername+"/traj.html").c_str());
    if(traj) {
        html::writer htmlout (traj);
        eventTraj.write (htmlout);
        traj.close();
    }
    if (Debug()) cout << "Finished archive routine" << endl;
}

/* //===========================================  Doppler fit for the Fitting routine (1/f)
Double_t fitf(Double_t *x, Double_t *par) {

    Double_t fitval = 0;

    fitval = par[0]*(1+par[1]*par[1]*(x[0]-par[2])*.0029154518950437/sqrt(par[1]*
	  (x[0]-par[2])*par[1]*(x[0]-par[2])+par[3]*par[3]));

    return fitval;

}

//============================================  Doppler fit for the Fitting routine (f)
Double_t planeFit(Double_t *x, Double_t *par) {

    Double_t fitval = 0;

    fitval = par[0]*343*sqrt(par[1]*par[1]*(x[0]-par[2])*(x[0]-par[2])+par[3]*par[3])/
	(343*sqrt(par[1]*par[1]*(x[0]-par[2])*(x[0]-par[2])+par[3]*par[3])+par[1]*
	 par[1]*(x[0]-par[2]));

    return fitval;
} */



//===========================================  Fitting routine
void PlaneMon::fitcurve(const int& i, const Time& t) {
    string filename = TimeString(mArchiveTime, "%s");
    
    //Output starting fit routine in the detailed log file
    ofstream info((dirloc+"log.txt").c_str(),ios::app);
    info << mGpsNow << " " << parameters->channelNames[i] << 
	" Starting fit routine ";
    info.close();
    
    int heldData = (int)(120/(parameters->Stride));
    
    //Get the name for the channel
    string chname = parameters->channelNames[i];
    string name = mic_locate(chname);

    //The new t-f bump plot
    fittedPlots[i] = new TH1F((name+string("_")+filename).c_str(),
			      "Airplane bump plot", heldData, 0, 120);
    if (Debug()) cout << "Created fittedPlots[" << i << "] at memory location "
		      << fittedPlots[i] << endl;
    
    int bumpcheck = mPointer[i];  // Check to make sure there are enough data points
    
    //Put a space in the log file
    ofstream infor((dirloc+"log.txt").c_str(),ios::app);
    infor << bumpcheck << " ";
    infor.close();

    //  Only data points with positive (1/freq slope on either side are kept. 
    //  First and last points must have negative slopes with the closest two 
    //  data points
    for(int j=0; j<mPointer[i]; j++) {
        int pos = (int)((xarray[i][j]-(mT-120))/parameters->Stride);
        if(j==0 && (((1/yarray[i][j]-3*invstddevs[i][j])-
		     (1/yarray[i][j+1]+3*invstddevs[i][j+1]))/(xarray[i][j]-xarray[i][j+1]))>=0.) {
            fittedPlots[i]->SetBinContent(pos+1,1/yarray[i][j]);
	    fittedPlots[i]->SetBinError(pos+1,invstddevs[i][j]);
        } else if(j>0 && (((1/yarray[i][j]+3*invstddevs[i][j])-
			   (1/yarray[i][j-1]-3*invstddevs[i][j-1]))/(xarray[i][j]-xarray[i][j-1]))>=0.) {
            fittedPlots[i]->SetBinContent(pos+1,1/yarray[i][j]);
	    fittedPlots[i]->SetBinError(pos+1,invstddevs[i][j]);
        }
    }
    
    //Write finished slope check in the detailed log file.
    ofstream inform((dirloc+"log.txt").c_str(),ios::app);
    inform << "Finished slope check " << bumpcheck << " ";
    inform.close();

    /* //The fits, will be called "fit"
    TF1* polfit = new TF1((name+"_fit"+filename).c_str(),"pol5",0,120);
    cout << "Created polfit at memory location " << polfit << endl;
    TF1* fit = new TF1("fit",fitf,0,120,4);
    cout << "Created fit at memory location " << fit << endl; */
    planeFits[i] = new TF1(("planeFit_"+name).c_str(),planeFit,0,120,4);
    if (Debug()) cout << "Created planeFits[" << i << "] at memory location " 
		      << planeFits[i] << endl;
    
    //If there are enough data points, fit the data to a polynomial, find the odd function
    //from the polynomial then cut any point with t less than t0 and which is larger than
    //1/f0 or cut any point with t larger than t0 an is smaller than 1/f0.
    if(bumpcheck>=12) {
        polfit->SetParameters(.005,.0015,.0003,.000005,.000001,.0000003);
        polfit->SetParLimits(0,.001,.1);  
	polfit->SetParLimits(1,.00000001,.02);  
	polfit->SetParLimits(2,-.0025,.0025);  
	polfit->SetParLimits(3,-.00084,.00034);  
	polfit->SetParLimits(4,-.00021,.00021);  
	polfit->SetParLimits(5,-.000034,.000084);  
	fittedPlots[i]->Fit(polfit,"OQNF");
	double b5 = polfit->GetParameter(5);
	double tinfl = -.2*polfit->GetParameter(4)/b5;
	double b3 = polfit->GetParameter(3)-10*b5*tinfl*tinfl;
	double b1 = polfit->GetParameter(1)-3*b3*tinfl*tinfl-5*b5*pow(tinfl,4);
	double b0 = polfit->GetParameter(0)+tinfl*(b1+b3*tinfl*tinfl+b5*pow(tinfl,4));
	for(int j=0; j<mPointer[i]; j++) {
	    int pos = (int)((xarray[i][j]-(mT-120))/parameters->Stride);
	    if(xarray[i][j]-(mT-120)<tinfl && 1/yarray[i][j]>b0) {
		fittedPlots[i]->SetBinContent(pos+1,0.);
		fittedPlots[i]->SetBinError(pos+1,0.);
		bumpcheck--;
	    } else if(xarray[i][j]-(mT-120)>tinfl && 1/yarray[i][j]<b0) {
		fittedPlots[i]->SetBinContent(pos+1,0.);
		fittedPlots[i]->SetBinError(pos+1,0.);
		bumpcheck--;
	    }
	}
    }
    
    //Write finished above/below cut to detailed log file.
    ofstream informa((dirloc+"log.txt").c_str(),ios::app);
    informa << "Finished above/below cut " << bumpcheck << " ";
    informa.close();

    //If there are still enough points, then cut any single point with chi squared statistic 
    //that is larger than 0.5
    if(bumpcheck>=12) {
        polfit->SetParameters(polfit->GetParameter(0),polfit->GetParameter(1),
			      polfit->GetParameter(2),polfit->GetParameter(3),polfit->GetParameter(4),
			      polfit->GetParameter(5));
	polfit->SetParLimits(0,.001,.1);  //fit->SetParLimits(0,.001,.1);
	polfit->SetParLimits(1,.00000001,.02);  //fit->SetParLimits(1,.00000001,.02);
	polfit->SetParLimits(2,-.0025,.0025);  //fit->SetParLimits(2,-.005,.005);
	polfit->SetParLimits(3,-.00084,.00034);  //fit->SetParLimits(3,-.005,.002);
	polfit->SetParLimits(4,-.00021,.00021);  //fit->SetParLimits(4,-.005,.005);
	polfit->SetParLimits(5,-.000034,.000084);  //fit->SetParLimits(5,-.004,.01);
        fittedPlots[i]->Fit(polfit,"OQNF");
	double a0 = polfit->GetParameter(0);
    	double a1 = polfit->GetParameter(1);
    	double a2 = polfit->GetParameter(2);
    	double a3 = polfit->GetParameter(3);
    	double a4 = polfit->GetParameter(4);
    	double a5 = polfit->GetParameter(5);
	for(int j=0; j<mPointer[i]; j++) {
	    int pos = (int)((xarray[i][j]-(mT-120))/parameters->Stride);
	    if(fittedPlots[i]->GetBinContent(pos+1)>0.) {
		double tval = xarray[i][j]-(mT-120);
		double chisq = pow((1/yarray[i][j]-(a0+a1*tval+a2*tval*tval+a3*pow(tval,3)+
						    a4*pow(tval,4)+a5*pow(tval,5)))/(invstddevs[i][j]),2);
		if(chisq>=7.5) {
		    fittedPlots[i]->SetBinContent(pos+1,0.);
		    fittedPlots[i]->SetBinError(pos+1,0.);
		    bumpcheck--;
		}
	    }
	}
    }
    //Write finished chi-squared cut to detailed log file.
    ofstream informat((dirloc+"log.txt").c_str(),ios::app);
    informat << "Finished chi-squared cut " << bumpcheck << " ";
    informat.close();

    //  If there is still enough data, then fit the remaining points to the 
    //  Doppler equation and replot as freq vs time
    if(bumpcheck>=12) {
        polfit->SetParameters(polfit->GetParameter(0), polfit->GetParameter(1),
			      polfit->GetParameter(2), polfit->GetParameter(3),
			      polfit->GetParameter(4), polfit->GetParameter(5));
    	polfit->SetParLimits(0,.001,.1);
	polfit->SetParLimits(1,.00000001,.02);
	polfit->SetParLimits(2,-.0025,.0025);
	polfit->SetParLimits(3,-.00084,.00034);
	polfit->SetParLimits(4,-.00021,.00021);
	polfit->SetParLimits(5,-.000034,.000084);
	fittedPlots[i]->Fit(polfit,"OQNF");
	double c5 = polfit->GetParameter(5);
    	double tinflec = -.2*polfit->GetParameter(4)/c5;
    	double c3 = polfit->GetParameter(3)-10.*c5*tinflec*tinflec;
    	double c1 = polfit->GetParameter(1)-3.*c3*tinflec*tinflec-
	    5.*c5*tinflec*tinflec*tinflec*tinflec;
    	double c0 = polfit->GetParameter(0)+tinflec*(c1+c3*tinflec*tinflec+
						     c5*tinflec*tinflec*tinflec*tinflec);
    	double doca = abs(-3.*(1./c0)*343.*c1*c1/(6.*c3));
    	double vp = sqrt(abs(-3.*(1./c0)*(1./c0)*343.*343.*c1*c1*c1/(6.*c3)));
	
	//cout << "Did last poly fit" << endl;

	//Input doppler fit equation for inverse frequency and fit 1/f vs time plot
	fit->SetParameters(abs(c0),vp,tinflec,doca);
	fit->SetParLimits(0,.5*abs(c0),abs(c0)+.5*abs(c0));
	fit->SetParLimits(1,.5*vp,2.*vp);
	fit->SetParLimits(2,tinflec-30.,tinflec+30.);
	fit->SetParLimits(3,.5*doca,doca+.5*doca);
	fittedPlots[i]->Fit("fit","OQN");
	//invfit = (TF1*)fittedPlots[i]->GetFunction("fit");
	//delete planeFits[i];

	//cout << "Did doppler fit for 1/f" << endl;

	for(int j=0; j<mPointer[i]; j++) {
	    int pos = (int)((xarray[i][j]-(mT-120))/parameters->Stride);
	    if(fittedPlots[i]->GetBinContent(pos+1)!=0) {
		fittedPlots[i]->SetBinContent(pos+1,yarray[i][j]);
		fittedPlots[i]->SetBinError(pos+1,stddevs[i][j]);
	    }
	}
       	//Save the removed bump plot
    	fittedPlots[i]->Write((name+"_"+filename).c_str());
	
	//Now make the plot be f vs time.
	for(int j=0; j<mPointer[i]; j++) {
	    int pos = (int)((xarray[i][j]-(mT-120))/parameters->Stride);
	    fittedPlots[i]->SetBinContent(pos+1,yarray[i][j]);
	    fittedPlots[i]->SetBinError(pos+1,stddevs[i][j]);
    	}

    	//  We will be plotting the doppler fit with found parameters f0, vp, 
	//  t0 and doca.
    	//planeFits[i] = new TF1("planeFit",planeFit,0,120);
    	planeFits[i]->SetParameters((1./fit->GetParameter(0)),fit->GetParameter(1),
				    fit->GetParameter(2),fit->GetParameter(3));
	mFitted[i] = true;
	if (Debug()) {
	    cout << "Plotted doppler fit for f" << endl;
	    cout << "f0=" << planeFits[i]->GetParameter(0) << endl;
	}
	//If the fitting didn't happen, just switch to f vs time.
    } else {

	//Even save the unfitted PlaneMon cut plots
	//fittedPlots[i]->Write((name+"_"+filename).c_str());

    	for(int j=0; j<mPointer[i]; j++) {
	    int pos = (int)((xarray[i][j]-(mT-120))/parameters->Stride);
	    fittedPlots[i]->SetBinContent(pos+1,yarray[i][j]);
	    fittedPlots[i]->SetBinError(pos+1,stddevs[i][j]);
	}
	mFitted[i] = false;

        if (Debug()) cout << "1 Deleting planefits[" << i 
			  << "] at memory location " << planeFits[i] << endl;
	delete planeFits[i];
    }

    //Print the current channel name to the stats file.
    ofstream stats((dirloc+"stats.txt").c_str(),ios::app);
    stats << parameters->channelNames[i];
    stats.close();

    //Set up the plot to look nice.
    if (Debug()) cout << "pretty plots" << endl;
    fittedPlots[i]->SetStats(kFALSE);				//no stats box
    fittedPlots[i]->SetXTitle("Time (sec)");			//x axis title
    fittedPlots[i]->GetXaxis()->CenterTitle(kTRUE);		//centered
    fittedPlots[i]->SetYTitle("Frequency (Hz)");		//y axis title
    fittedPlots[i]->GetYaxis()->CenterTitle(kTRUE);		//centered
    fittedPlots[i]->SetMinimum(parameters->fLow);		//y axis min
    fittedPlots[i]->SetMaximum(parameters->fHigh);	        //y axis max
    fittedPlots[i]->SetMarkerStyle(7);				//dots
    fittedPlots[i]->SetMarkerColor(4);				//blue
    //If the plot was fitted, then print things to the plot and add data to 
    //the stats file.
    if(mFitted[i]) {
	ostringstream infostr;
        mCandidate = true;

	planeFits[i]->SetLineColor(kBlack);
	
	curveInfos[i] = new TPaveText(.65,.70,.99,.99,"NDC");	//Info box
        if (Debug()) cout << "Created curveInfos[" << i 
			  << "] at memory location " << curveInfos[i] << endl;

	if (Debug()) cout << "f0" << endl;

    	infostr << "f0 = " << planeFits[i]->GetParameter(0) << " Hz";
	curveInfos[i]->AddText(infostr.str().c_str());  //add f0 to the info box
    	ofstream stat((dirloc+"stats.txt").c_str(),ios::app);
        stat << " " << setprecision (6) << planeFits[i]->GetParameter(0);
        stat.close();
	infostr.str().clear();

	if (Debug()) cout << "t0" << endl;

    	infostr << "t0 = " << planeFits[i]->GetParameter(2) + mfirsttime[i]; //buffer t0
	curveInfos[i]->AddText(infostr.str().c_str());	 //add t0 to the box
	infostr.str("");
	ofstream statist((dirloc+"stats.txt").c_str(),ios::app);
        statist << " " << setprecision (12) 
		<< planeFits[i]->GetParameter(2)+mfirsttime[i];
        statist.close();

	if (Debug()) cout << "chi-sq" << endl;

    	infostr << "ChiSquared = " << fit->GetChisquare();	//buffer chisq
	curveInfos[i]->AddText(infostr.str().c_str());		//add chisq
	infostr.str("");

	if (Debug()) cout << "doca" << endl;

    	infostr << "DOCA = " << planeFits[i]->GetParameter(3) << " m";	//buffer DOCA
    	curveInfos[i]->AddText(infostr.str().c_str());		     //add DOCA
        ofstream statout((dirloc+"stats.txt").c_str(),ios::app);
        statout << " " << setprecision (8) << planeFits[i]->GetParameter(3);
        statout.close();
	infostr.str("");

	if (Debug()) cout << "vp" << endl;

    	infostr << "vp = " << planeFits[i]->GetParameter(1) << " m/s";	//buffer vp
    	curveInfos[i]->AddText(infostr.str().c_str());		    //add vp
        ofstream statoutp((dirloc+"stats.txt").c_str(),ios::app);
        statoutp << " " << setprecision (8) << planeFits[i]->GetParameter(1) 
		 << " " << fit->GetChisquare();
        statoutp << " " << mPointer[i] << " " << bumpcheck;
        statoutp.close();
	infostr.str("");

	if (Debug()) cout << "points" << endl;

    	infostr << "No. of points = " << bumpcheck;    // buffer bumpcheck
    	curveInfos[i]->AddText(infostr.str().c_str()); // add bumpcheck
    	curveInfos[i]->SetBorderSize(1);	       // set border to be small
	infostr.str("");
    	if (std::isnan(fit->GetParameter(1)) ||        // isnan fails Double_t
	   planeFits[i]->GetParameter(1)>343.) {
            infostr << "ERROR IN FIT!";
            curveInfos[i]->AddText(infostr.str().c_str());
	    infostr.str("");
        }    
    } 
    //  If not fitted then print out in the stats file that it wasn't fitted.
    else {
        ofstream statp((dirloc+"stats.txt").c_str(),ios::app);
        statp << " NOT FITTED";
        statp.close();
    }

    //Endline in the stats file
    ofstream statistics((dirloc+"stats.txt").c_str(),ios::app);
    statistics << "\n";
    statistics.close();
    
    /* cout << "Deleting polfit and fit at memory locations " << polfit << " and " << fit << endl;
    delete polfit;
    delete fit;                 //Delete the fits function */
    
    //Write that the fit routine has finished in the detailed log file.
    ofstream informati((dirloc+"log.txt").c_str(),ios::app);
    informati << "Finished fit routine." << endl;
    informati.close();

    if (Debug()) cout << "Fit routine ended!" << endl;

}       //End fit routine.




//======================================  Generate Segment
void PlaneMon::genSegment(TrigClient& tc) {
    if(mLoc.compare("L")==0 && mCandidate) {
      trig::Segment s("DMT-AIRCRAFT_VERY_LIKELY", 1, mArchiveTime, 
		      mArchiveTime + Interval(120));
      s.setIfos("L1");
      lmsg::error_type rc = tc.sendSegment(s);
      if(rc) {
        lmsg::put_lmsg_error("[PlaneMon] Error sending segment", rc);
      }
    } else if(mLoc.compare("L")==0 && !mCandidate) {
      trig::Segment s("DMT-AIRCRAFT_LIKELY", 1, mArchiveTime, 
		      mArchiveTime + Interval(120));
      s.setIfos("L1");
      lmsg::error_type rc = tc.sendSegment(s);
      if(rc) {
        lmsg::put_lmsg_error("[PlaneMon] Error sending segment", rc);
      }
    } else if(mLoc.compare("H")==0 && mCandidate) {
      trig::Segment s("DMT-AIRCRAFT_VERY_LIKELY", 1, mArchiveTime, 
		      mArchiveTime + Interval(120));
      s.setIfos("H1");
      lmsg::error_type rc = tc.sendSegment(s);
      if(rc) {
        lmsg::put_lmsg_error("[PlaneMon] Error sending segment", rc);
      }
      /* Commenting out H2 segments - no longer needed
      s.setIfos("H2");
      rc = tc.sendSegment(s);
      if(rc) {
        lmsg::put_lmsg_error("[PlaneMon] Error sending segment", rc);
      }
      */
    } else if(mLoc.compare("H")==0 && !mCandidate) {
      trig::Segment s("DMT-AIRCRAFT_LIKELY", 1, mArchiveTime, 
		      mArchiveTime + Interval(120));
      s.setIfos("H1");
      lmsg::error_type rc = tc.sendSegment(s);
      if(rc) {
        lmsg::put_lmsg_error("[PlaneMon] Error sending segment", rc);
      }
      /* Commenting out H2 segments - no longer needed
      s.setIfos("H2");
      rc = tc.sendSegment(s);
      if(rc) {
        lmsg::put_lmsg_error("[PlaneMon] Error sending segment", rc);
      }
      */
    } else {
      cout << "PROBLEM making segment!" << endl;
    }
    cout << "Completed segment with start time = " << mArchiveTime << endl;
}

//======================================  Make a bump plot
TH1F*
PlaneMon::bumpPlot(int i, const std::string& name) const {
    Time t0(Time::ulong_t(mT) - 120);
    Time::ulong_t tOff = t0.getS();
    string ttl = TimeString(t0, "Airplane bump plot %02m/%02d/%Y");
    int heldData = (int)(120/(parameters->Stride));
    TH1F* plot = new TH1F(name.c_str(), ttl.c_str(), heldData, 0, 120);
    plot->GetXaxis()->SetTimeDisplay(1);  // The X axis is a time axis
    plot->GetXaxis()->SetTimeOffset(getUTC(t0), "gmt");
    plot->GetXaxis()->SetTimeFormat("%T");
    for(int j=0; j<mPointer[i]; j++) {
	int pos = int(double(xarray[i][j]-tOff)/parameters->Stride);
	plot->SetBinContent(pos+1,yarray[i][j]);
	plot->SetBinError(pos+1,stddevs[i][j]);
    }

    plot->SetStats(kFALSE);			//no stats box
    plot->SetXTitle("Time (hh:mm:ss UTC)");	//x axis title
    plot->GetXaxis()->CenterTitle(kTRUE);	//centered
    plot->SetYTitle("Frequency (Hz)");		//y axis title
    plot->GetYaxis()->CenterTitle(kTRUE);	//centered
    plot->SetMinimum(parameters->fLow);		//y axis min
    plot->SetMaximum(parameters->fHigh);	//y axis max
    plot->SetMarkerStyle(7);			//dots
    plot->SetMarkerColor(4);			//blue
    return plot;
}


//======================================  Handle Messages
void
PlaneMon::Attention(void) {
    MonServer::Attention();
}

