/* -*- mode: c++; c-basic-offset: 3; -*- */
//======================================  W-pipe headers
#include "wscan.hh"
#include "wcondition.hh"
#include "weventlist.hh" 
#include "wfigure.hh"
#include "wframecache.hh"
#include "wmeasure.hh"
#include "wreaddata.hh"
#include "wresample.hh"
#include "wtile.hh"
#include "wtransform.hh"
#include "matlab_fcs.hh"

//======================================  DMT headers
#include "TSeries.hh"
#include "fSeries/DFT.hh"
#include "Interval.hh"

//======================================  C++ and system headers.
#include <fstream>
#include <sstream>
#include <iostream>
#include <cstdlib>
#include <cstdio>
#include <cmath>
#include <iomanip>
#include <stdexcept>

using namespace std;
using namespace wpipe;

//======================================  Syntax output function
static void
wscan_syntax(void) {
   cerr << "The dmt_wscan command syntax is:" << endl;
   cerr << "  dmt_wscan <event-time> <config> <frame-cache> <out-dir>"
	<< " <plot> <debug>" << endl;
   cerr << "where: " << endl;
   cerr << "  <event-time>  Is the central time to be scanned." << endl;
   cerr << "  <config>      Is a configuration file path." << endl;
   cerr << "  <frame-cache> Is the omega frame-cache file path." << endl;
   cerr << "<out-dir>       is the root directory to receive plots for this run"
	<< endl;
   cerr << " <gen-report>   if non-zero, an html report page is generated [0]." 
	<< endl;
   cerr << " <debug>        debug printout level [1]." << endl;
}

//======================================  Main function.
int 
main(int argc, const char* argv[]) {
   std::string frameCacheFile, outDir;
   bool generateReport = false;
   int debugLevel = 1;
   if (argc < 3) {
      cerr << "Insufficient arguments" << endl;
      wscan_syntax();
      return 1;
   }

   //------------------------------------  Get the start time.
   string eventTimeString = argv[1];

   std::string configFile = argv[2];

   if (argc > 3) frameCacheFile = argv[3];  

   if (argc > 4) outDir = argv[4];

   if (argc > 5) {
      if (string(argv[5]) == "true") generateReport = 1;
      else                           generateReport = strtol(argv[5],0,0) > 0;
   }

   if (argc > 6) {
      debugLevel = strtol(argv[6],0,0);
   }

   cout << "wscan parameters: " <<endl;
   cout << "   event time       = " << eventTimeString << endl;
   cout << "   config file      = " << configFile << endl;
   cout << "   frame cache      = " << frameCacheFile << endl;
   cout << "   output directory = " << outDir << endl;
   cout << "   generate report  = " << boolstr_yn(generateReport) << endl;
   cout << "   debug level      = " << debugLevel << endl;

   try {
      wscan(eventTimeString, configFile, frameCacheFile, outDir,
	    generateReport, debugLevel);
   }
   catch (exception& e) {
      cerr << "caught exception: " << e.what() << endl;
   }
}

static std::ostream&
expfmt(double x, std::ostream& out) {
   double dexp = log10(x);
   int expo = int(dexp);
   if (dexp < 0) expo--;
   return out << fixed << setprecision(1) << x/pow(10.0, expo)
	      << "&times;10<sup>" << expo << "</sup>";
}

//======================================  insert scripts function
static void
insert_scripts(ostream& file) {
   const char* script_text[] = {
      "<script type=\"text/javascript\">",
      "function showImage(gpsTime, channelName, timeRanges, imageType) {",
      "  for (var timeRangeIndex in timeRanges) {",
      "    var imageBaseName =",
      "      gpsTime + \"_\" + channelName + \"_\" + timeRanges[timeRangeIndex];",
      "    document.getElementById(\"a_\" + imageBaseName).href =",
      "      imageBaseName + \"_\" + imageType + \".png\";",
      "    document.getElementById(\"img_\" + imageBaseName).src =",
      "      imageBaseName + \"_\" + imageType + \".thumb.png\";",
      "  }",
      "}",
      "function toggleVisible(division) {",
      "  if (document.getElementById(\"div_\" + division).style.display == \"none\") {",
      "    document.getElementById(\"div_\" + division).style.display = \"block\";",
      "    document.getElementById(\"input_\" + division).checked = true;",
      "  } else {",
      "    document.getElementById(\"div_\" + division).style.display = \"none\";",
      "    document.getElementById(\"input_\" + division).checked = false;",
      "  } ",
      "}",
      "function gotoSection(section) {",
      "  document.getElementById(\"div_\" + section).style.display = \"block\";",
      "  document.getElementById(\"input_\" + section).checked = true;",
      "  window.location.hash = section;",
      "}",
      "</script>",
      0
   };
   for (int i=0; script_text[i]; i++) {
      file << script_text[i] << endl;
   }
}

//======================================  scan function.
void
wpipe::wscan(const std::string& eventTimeString, const std::string& configFile, 
	     const std::string& frameCacheFile, const std::string& outDir, 
	     bool generateReport, int debugLevel) {
   const double dInf(1.0/0.0);

   wfigure figureHandle;

   /////////////////////////////////////////////////////////////////////////////
   //                                   defaults                              //
   /////////////////////////////////////////////////////////////////////////////

   // default configuration parameters
   search_chan defchan;
   defchan.sampleFrequency = 4096;
   defchan.searchTimeRange = 64;
   double fRange[] = {0, dInf};
   defchan.searchFrequencyRange = dble_vect(fRange, fRange+2);
   double qRange[] = {4, 64};
   defchan.searchQRange = dble_vect(qRange, qRange+2);
   defchan.searchMaximumEnergyLoss = 0.2;
   defchan.whiteNoiseFalseRate = 1e-2;
   defchan.searchWindowDuration = 0.1;
   double plotRange[] = {1, 4, 16};
   defchan.plotTimeRanges = dble_vect(plotRange, plotRange+3);
   double plotFRange[]  = {0, dInf};
   defchan.plotFrequencyRange = dble_vect(plotFRange, plotFRange+2);
   //notused double defaultPlotMaximumEnergyLoss = 0.2;
   double plotNormER[] = {0, 25.5};
   defchan.plotNormalizedEnergyRange = dble_vect(plotNormER, plotNormER+2);
   //notused bool defaultAlwaysPlotFlag = 0;

   /////////////////////////////////////////////////////////////////////////////
   //                            hard coded parameters                        //
   /////////////////////////////////////////////////////////////////////////////

   // search parameters
   double transientFactor = 2.0;
   double outlierFactor = 2.0;
   //notused double durationInflation = 1.0;
   //notused double bandwidthInflation = 1.0;

   // display parameters
   int plotHorizontalResolution = 512;
   double plotDurationInflation = 0.5;
   double plotBandwidthInflation = 0.5;

   // limits on number of significant tiles
   int maximumSignificants = 10000;
   int maximumMosaics = 10000;

   /////////////////////////////////////////////////////////////////////////////
   //                        process command line arguments                   //
   /////////////////////////////////////////////////////////////////////////////

   string outputDirectory = outDir;

   // apply default arguments
   string configurationFile = configFile;
   if (configurationFile.empty()) {
      configurationFile = "configuration.txt";
   }

   std::string cacheFile = frameCacheFile;
   if (cacheFile.empty()) {
      cacheFile = "framecache.txt";
   }

   // convert string event time and debug level to numbers
   double time = strtod(eventTimeString.c_str(), 0);
   long sec = long(time);
   Time eventTime = Time(sec, long((time-sec)*1e9));

   // name of html file
   string htmlFile = "index.html";

   // name of text summary file
   string textSummaryFile = "summary.txt";

   // name of xml summary file
   string xmlSummaryFile = "summary.xml";

   // name of context file
   string contextFile = "context.html";

   /////////////////////////////////////////////////////////////////////////////
   //                                 write header                            //
   /////////////////////////////////////////////////////////////////////////////

   // report status
   if (debugLevel >= 1) {
      time_t clock=0;
      cout << "OmegaScan of event at " << eventTimeString << endl;
      cout << "created by " << getenv("USER") << " on " << datestr(clock, 29)
	   <<" at " << datestr(clock, 13) << endl;
   }

   /////////////////////////////////////////////////////////////////////////////
   //                       initialize random number generators               //
   /////////////////////////////////////////////////////////////////////////////

   // set random number generator seeds based on event time
   //rand("state", eventTime);
   //randn("state", eventTime);

   /////////////////////////////////////////////////////////////////////////////
   //                           read configuration file                       //
   /////////////////////////////////////////////////////////////////////////////

   // report status
   if (debugLevel >= 1) {
      cout << "reading configuration file " << configurationFile << endl;
   }
   // enable or disable warnings
   if (debugLevel >= 2) {
      // warning on WSCAN:incompleteConfiguration"
   } else {
      //warning off WSCAN:incompleteConfiguration"
   }

   // initialize configuration structure
   config_vect configuration;
   section_vect sectList;
   scan_config(configurationFile, defchan, configuration, sectList);

   // number of configured channels
   int numberOfChannels = configuration.size();

   // number of sections
   int numberOfSections = sectList.size();

   /////////////////////////////////////////////////////////////////////////////
   //                           load frame cache file                         //
   /////////////////////////////////////////////////////////////////////////////

   // report status
   if (debugLevel >= 1) {
      cout << "reading frame cache file " << cacheFile << endl;
   }

   // load frame file cache
   wframecache frameCache(cacheFile);
   if (debugLevel > 2) frameCache.display();

   /////////////////////////////////////////////////////////////////////////////
   //                         create output directory                         //
   /////////////////////////////////////////////////////////////////////////////

   // if (outputDirectory not specified, make one based on center time
   if (outputDirectory.empty()) {
      outputDirectory = string("scans/") + eventTimeString;
   }

   // report status
   if (debugLevel >= 1) {
      cout << "creating event directory" << endl;
      cout << "  outputDirectory:         " << outputDirectory << endl;
   }

   // create spectrogram directory
   string shell_cmd;
   shell_cmd  = string("/bin/mkdir -p ") + outputDirectory + ";";
   shell_cmd += string("/bin/cp ") + configurationFile + " " + outputDirectory 
      + "/configuration.txt";

   // copy configuration file
   if (system(shell_cmd.c_str())) {
      error(string("shell command: \"") + shell_cmd + " \"failed");
   }

   /////////////////////////////////////////////////////////////////////////////
   //                      initialize text summary report                     //
   /////////////////////////////////////////////////////////////////////////////

   // report status
   wlog(debugLevel, 1, "opening text summary");

   // open text summary file
   string fullTextSummaryFile = outputDirectory + "/" + textSummaryFile;
   FILE* textSummaryFID = fopen(fullTextSummaryFile.c_str(), "w");

   // write column definitions
   fprintf(textSummaryFID, "# channelName\n");
   fprintf(textSummaryFID, "# peakTime\n");
   fprintf(textSummaryFID, "# peakFrequency\n");
   fprintf(textSummaryFID, "# peakQ\n");
   fprintf(textSummaryFID, "# peakSignificance\n");
   fprintf(textSummaryFID, "# peakAmplitude\n");

   /////////////////////////////////////////////////////////////////////////////
   //                      initialize xml summary report                      //
   /////////////////////////////////////////////////////////////////////////////

   // report status
   wlog(debugLevel, 1, "opening xml summary");

   // open xml summary file
   string fullXmlSummary = outputDirectory +  "/" + xmlSummaryFile;
   FILE* xmlSummaryFID = fopen(fullXmlSummary.c_str(), "w");
   if (!xmlSummaryFID) {
      perror("Error opening xml summary file");
      return;
   }

   // write channel definitions
   fprintf(xmlSummaryFID, "<?xml version='1.0' encoding='utf-8' ?>\n");
   fprintf(xmlSummaryFID, "<LIGO_LW>\n");
   fprintf(xmlSummaryFID, "<Table Name=\"wscan:summary:table\">\n");
   fprintf(xmlSummaryFID, 
	   "<Column Name=\"wscan:summary:channelName\" Type=\"lstring\"/>\n");
   fprintf(xmlSummaryFID, 
	   "<Column Name=\"wscan:summary:peakTime\" Type=\"real_8\"/>\n");
   fprintf(xmlSummaryFID, 
	   "<Column Name=\"wscan:summary:peakFrequency\" Type=\"real_8\"/>\n");
   fprintf(xmlSummaryFID, 
	   "<Column Name=\"wscan:summary:peakQ\" Type=\"real_8\"/>\n");
   fprintf(xmlSummaryFID, 
	 "<Column Name=\"wscan:summary:peakSignificance\" Type=\"real_8\"/>\n");
   fprintf(xmlSummaryFID, 
	   "<Column Name=\"wscan:summary:peakAmplitude\" Type=\"real_8\"/>\n");
   fprintf(xmlSummaryFID, 
      "<Stream Name=\"wscan:summary:table\" Type=\"Local\" Delimiter=\",\">\n");
   fflush(xmlSummaryFID);

   /////////////////////////////////////////////////////////////////////////////
   //                          initialize html report                         //
   /////////////////////////////////////////////////////////////////////////////
   ofstream htmlFID;
   if (generateReport) {

      // report status
      wlog(debugLevel, 1, "opening html report");

      // open web page
      string fullHtmlFile = outputDirectory + "/" + htmlFile;
      htmlFID.open(fullHtmlFile.c_str());
      if (!htmlFID.is_open()) {
	 perror("Error opening html file");
	 return;
      }

      // begin html
      htmlFID << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\" \"http://www.w3.org/TR/html4/strict.dtd\">"
	      << endl;
      htmlFID << "<html>" << endl;
      htmlFID << "<head>" << endl;
      htmlFID << "<title>Scan " << eventTimeString << "</title>"<< endl;
      htmlFID << "<link rel=\"icon\" type=\"image/x-icon\" href=\"omega.logo.icon.png\" />" << endl;
      htmlFID << "<link rel=\"stylesheet\" type=\"text/css\" href=\"wstyle.css\" />" 
	      << endl;

      insert_scripts(htmlFID);

      htmlFID << "</head>" << endl;
      htmlFID << "<body>" << endl;
      htmlFID << endl;

      htmlFID << "<h1 class=\"title\">"
	      << "<a href=\"https://wiki.ligo.org/detchar/Omega_c#dmt_wscan\""
	      << " class=\"title\">Omega_c Pipeline</a> Scan</h1>" << endl;
      htmlFID << "<h1>Scan " << eventTimeString << "</h1>" << endl;
      htmlFID << endl;
      htmlFID << "<div class=\"main\">" << endl;
      htmlFID << endl;

      /////////////////////////////////////////////////////////////////////
      //                           add index to html report              //
      /////////////////////////////////////////////////////////////////////

      htmlFID << "<h2>Index</h2>" << endl;
      htmlFID << "<div class=\"section\">" << endl;
      htmlFID << "<ul>" << endl;
      for (int sectionNumber=0; 
	   sectionNumber<numberOfSections; 
	   sectionNumber++) {
	 htmlFID << "<li><a href=\"javascript:gotoSection('" 
		 << sectList[sectionNumber].Index << "')\">" 
		 << sectList[sectionNumber].Index << "</a></li>" << endl;
      }
      htmlFID << "<li><a href=\"log.txt\">Scan Log</a></li>" << endl;
      htmlFID << "<li><a href=\"" << textSummaryFile 
	      << "\">Text Summary</a></li>" << endl;
      htmlFID << "<li><a href=\"" << xmlSummaryFile 
	      << "\">XML Summary</a></li>" << endl;
      htmlFID << "<li><a href=\"configuration.txt\">Configuration</a></li>" 
	      << endl;
      htmlFID << "</ul>" << endl;
      htmlFID << "</div>" << endl;
      htmlFID << endl;
      htmlFID.flush();

      //--------------------------------  Initialize figure handle
      figureHandle.open();
   } // generateReport

   ////////////////////////////////////////////////////////////////////////////
   //                       loop over configuration channels                 //
   ////////////////////////////////////////////////////////////////////////////
  
   resampler resample;

   // initialize section status
   int inSectionFlag = 0;

   // begin loop over channel configurations
   for (int channelNumber=0; channelNumber<numberOfChannels; channelNumber++) {
      resample.reset();
      dble_vect emptyRange;

      if (debugLevel > 1) configuration[channelNumber].display(cout);

      /////////////////////////////////////////////////////////////////////////
      //                        add section to html report                   //
      /////////////////////////////////////////////////////////////////////////

      if (generateReport) {

	 // begin loop over new sections
	 for (int sectionNumber=0; 
	      sectionNumber < numberOfSections; 
	      sectionNumber++) {

	    if (sectList[sectionNumber].Start > channelNumber) break;
	    if (sectList[sectionNumber].Start != channelNumber) continue;
	  
	    // end any previously existing section
	    if (inSectionFlag == 1) {
	       htmlFID << "</div>" << endl;
	       htmlFID << endl;
	    }

	    // label section
	    htmlFID << "<a name=\"" << sectList[sectionNumber].Index <<"\"></a>"
		    << endl; 
	    htmlFID << "<h2 class=\"" << sectList[sectionNumber].Index << "\">"
		    << endl; 
	    htmlFID << "<input id=\"input_" << sectList[sectionNumber].Index 
		    << "\" type=\"checkbox\" " 
		    << sectList[sectionNumber].Checked << endl; 
	    htmlFID << "       onclick=\"toggleVisible('" 
		    << sectList[sectionNumber].Index << "');\" />" << endl; 
	    htmlFID << sectList[sectionNumber].Name << endl;
	    htmlFID << "</h2>" << endl;
	    htmlFID << endl;
	    htmlFID << "<div class=\"section\" id=\"div_" 
		    << sectList[sectionNumber].Index 
		    << "\" style=\"display: " 
		    << sectList[sectionNumber].Display << ";\">"  << endl; 
	    htmlFID << endl;

	    // record start of section
	    inSectionFlag = 1;

	    // transcribe context information
	    if (sectList[sectionNumber].Index == "Context" ||
		sectList[sectionNumber].Index == "Timing") {
	       contextFile = outputDirectory + "/" + contextFile;
	       if (exist(contextFile,"file") ) {
		  wlog(debugLevel, 1, "  adding context file");
	    
		  htmlFID << "<!-- BEGIN CONTEXT -->" << endl;
		  htmlFID << endl;
		  ifstream ctxstream(contextFile.c_str());
		  string ctxstring;
		  while (!ctxstream.eof()) {
		     getline(ctxstream, ctxstring);
		     htmlFID << ctxstring << endl;
		  }
		  ctxstream.close();
		  htmlFID << endl;
		  htmlFID << "<!-- END CONTEXT -->" << endl;
		  htmlFID << endl;
	       } else {
		  wlog(debugLevel, 1, "  context file not found. skipping");
	       }
	    }  else if (sectList[sectionNumber].Index == "Parameters") {
	       htmlFID << "<!-- BEGIN PARAMETERS -->" << endl;
	       htmlFID << endl;
	       htmlFID << "<!-- END PARAMETERS -->" << endl;
	       htmlFID << endl;
	    } else if (sectList[sectionNumber].Index == "Notes") {
	       htmlFID << "<!-- BEGIN NOTES -->" << endl;
	       htmlFID << endl;
	       htmlFID << "<!-- END NOTES -->" << endl;
	       htmlFID << endl;
	    }

	 }	// end loop over new sections

      } // generateReport
  
      /////////////////////////////////////////////////////////////////////////
      //               identify statistically significant channels           //
      /////////////////////////////////////////////////////////////////////////

      // extract search configuration for channel
      str_vect  channelName = configuration[channelNumber].channelName;
      string    channelNameString = strrep(channelName[0], ";", ":");
      str_vect  frameType = configuration[channelNumber].frameType;
      dble_vect timeShifts = configuration[channelNumber].timeShifts;
      double    sampleFrequency = configuration[channelNumber].sampleFrequency;
      double    timeRange = configuration[channelNumber].searchTimeRange;
      dble_vect frequencyRange=configuration[channelNumber].searchFrequencyRange;
      dble_vect qRange = configuration[channelNumber].searchQRange;
      double maximumEnergyLoss = configuration[channelNumber].searchMaximumEnergyLoss;
      double searchWindowDuration = configuration[channelNumber].searchWindowDuration;
      bool alwaysPlotFlag = configuration[channelNumber].alwaysPlotFlag;

      // Plot parameters
      dble_vect plotTimeRanges;
      dble_vect plotFrequencyRange;
      dble_vect plotNormalizedEnergyRange;
      if (generateReport) {
	 plotTimeRanges = configuration[channelNumber].plotTimeRanges;
	 plotFrequencyRange = configuration[channelNumber].plotFrequencyRange;
	 plotNormalizedEnergyRange = 
	    configuration[channelNumber].plotNormalizedEnergyRange;
      }

      // display status
      if (debugLevel >= 1) {
	 cout << "processing channel " << channelNameString << endl;
      }
  
      // find closest sample time to event time
      Time centerTime = Time(eventTime.getS());
      centerTime += round(double(eventTime-centerTime)*sampleFrequency)
	            / sampleFrequency;

      // determine segment start and stop times
      Time startTime = centerTime - Interval(timeRange / 2.);
      if (startTime-Time(startTime.getS()) < Interval(0.5)) {
	 startTime = Time(startTime.getS());
      } else {
	 startTime = Time(startTime.getS()+1);
      }
      Time stopTime = startTime + timeRange;

      // generate search tiling
      wlog(debugLevel, 2, "  tiling for search");
      double highPassCutoff = -1;
      double lowPassCutoff = -1;
      double whiteningDuration = -1;

      wtile tiling(timeRange, qRange, frequencyRange, sampleFrequency, 
		   maximumEnergyLoss, highPassCutoff, lowPassCutoff, 
		   whiteningDuration, transientFactor, debugLevel);


      //  The threshold is set to yield the specified false event rate when 
      //  applied to all available frequencies and Qs of a white noise 
      //  signal. It is not modified to account for restricted f, Q ranges. 
      //  It is also only a rough estimate, and the result false event rate
      //  may vary significantly depending on the quality of the data.
      //
      double whiteNoiseFalseRate = configuration[channelNumber].whiteNoiseFalseRate;
      double eventThreshold = configuration[channelNumber].eventThreshold;
      if (eventThreshold == 0) {
	 eventThreshold = tiling.threshold_from_rate(whiteNoiseFalseRate);
      }

      //========================================================================
      //  This has to be fixed for reasonable data I/O
      // read data from frame file
      wlog(debugLevel, 2, "  reading data");
      tser_vect rawData;
      wreaddata(frameCache, channelName, frameType, 
		startTime, stopTime, timeShifts, debugLevel,
		rawData);

      // check for read error
      if (rawData.empty()) {
	 wlog(debugLevel, 1, "  ERROR: cannot load frame data");
	 continue;
      }

      // test multiple channels for consistency
      if (rawData.size() > 1) {
	 wlog(debugLevel, 2, "  subtracting data");
	 rawData[0] -= rawData[1];
	 rawData.erase(rawData.begin()+1, rawData.end());
      }

      //--------------------------------  Check for all zeros
      if (rawData[0].getMinimum() == rawData[0].getMaximum()) {
	 wlog(debugLevel, 1, "  WARNING: channel contains no information");
	 continue;
      }

      //--------------------------------  Resample data
      wlog(debugLevel, 2, "  resampling data");
      tser_vect rsamp;
      rsamp = resample.wresample(rawData, sampleFrequency);

      // high pass filter and whiten data
      wlog(debugLevel, 2, "  conditioning data");

      wcondition conditioner(rsamp, tiling);
      containers::DFT whitenedData  = conditioner.whitenedDFT();
      containers::DFT whitenedCoefs;

      // q transform whitened data
      wlog(debugLevel, 2, "  transforming whitened data");
      wtransform whitenedTransform(whitenedData, tiling, whitenedCoefs, 
				   outlierFactor, channelNameString);
      //whitenedTransform.dump(cout);

      // identify most significant whitened transform tile
      wlog(debugLevel, 2, "  measuring peak significance");
      Time thresholdReferenceTime = centerTime;
      dble_vect thresholdTimeRange(2);
      thresholdTimeRange[0] = -0.5*searchWindowDuration;
      thresholdTimeRange[1] =  0.5*searchWindowDuration;
      dble_vect thresholdQRange;
      dble_vect thresholdFrequencyRange;
      wmeasure whitenedProperties(whitenedTransform, tiling, startTime, 
				  thresholdReferenceTime, thresholdTimeRange, 
				  thresholdFrequencyRange, thresholdQRange, 
				  debugLevel);
      //whitenedProperties.dump(cout);

      //--------------------------------  If channel not significant
      if (whitenedProperties[0].peak.NormalizedEnergy < eventThreshold) {

	 //-----------------------------  If always plot flag is not set
	 if (!alwaysPlotFlag) {

	    //--------------------------  report status, skip to next channel
	    if (debugLevel >= 1) {
	       cout << "  channel not significant at white noise false rate " 
		    << whiteNoiseFalseRate << endl;
	    }
	    fprintf(textSummaryFID, 
		    "%-30s %#020.9f %#09.3e %#09.3e %#09.3e %#09.3e\n", 
		    channelNameString.c_str(), 0.0, 0.0, 0.0, 0.0, 0.0);
	    fprintf(xmlSummaryFID, 
		    "\"%s\",%#020.9f,%#09.3e,%#09.3f,%#09.3e,%#09.3e,\n", 
		    channelNameString.c_str(), 0.0, 0.0, 0.0, 0.0, 0.0);
	    continue;
	 }   // end test for always plot flag
      } // end test of channel significance

      /////////////////////////////////////////////////////////////////////////
      //               analyze statistically significant channels            //
      /////////////////////////////////////////////////////////////////////////
 
      //--------------------------------  Q transform raw data
      wlog(debugLevel, 2, "  transforming raw data");
      wtransform rawTransform(conditioner.rawDFT(), tiling, whitenedCoefs, 
			      outlierFactor, channelNameString);
      //rawTransform.dump(cout);

      weventstack rawEvents, whitenedEvents;
      if (generateReport) {

	 // identify significant whitened q transform tiles
	 wlog(debugLevel, 2, "  thresholding whitened transform");
	 double maxPlotTRange = plotTimeRanges[0];
	 for (size_t i=1; i<plotTimeRanges.size(); i++) {
	    if (maxPlotTRange < plotTimeRanges[i]) 
	       maxPlotTRange = plotTimeRanges[i];
	 }
	 thresholdTimeRange[0] = -0.5 * (maxPlotTRange 
					 + tiling.end_plane().row(0).duration 
					 * plotDurationInflation);
	 thresholdTimeRange[1] = -thresholdTimeRange[0];
	 thresholdFrequencyRange = plotFrequencyRange;
	 thresholdQRange = emptyRange;

	 //  Identify significant tiles using event threshold calculated to 
	 //  give a specified false rate for white noise.
	 //
	 weventstack whitenedSignificants;
	 whitenedSignificants.wthreshold(
		     whitenedTransform, tiling, eventThreshold, 
		     thresholdReferenceTime, thresholdTimeRange, 
		     thresholdFrequencyRange, thresholdQRange, 
		     maximumSignificants, "", 0, 0, 0, debugLevel);
	 //whitenedSignificants.dump(cout);

	 // identify significant raw q transform tiles
	 wlog(debugLevel, 2, "  thresholding raw transform");
	 weventstack rawSignificants;
	 rawSignificants.wthreshold(rawTransform, tiling, 
				    eventThreshold, thresholdReferenceTime,
				    thresholdTimeRange, thresholdFrequencyRange,
				    thresholdQRange, maximumSignificants, "", 0,
				    0, 0, debugLevel);
	 //rawSignificants.dump(cout);

	 // select non-overlapping whitened significant tiles
	 wlog(debugLevel, 2, "  selecting whitened events");
	 whitenedEvents.wselect(whitenedSignificants, plotDurationInflation, 
				plotBandwidthInflation,  maximumMosaics, 
				debugLevel);

	 // select non-overlapping raw significant tiles
	 wlog(debugLevel, 2, "  selecting raw events");
	 rawEvents.wselect(rawSignificants, plotDurationInflation, 
			   plotBandwidthInflation, maximumMosaics, debugLevel);

	 // recover time series data
	 //rawData = wifft(rawData);
	 //highPassedData = wifft(highPassedData);
	 //whitenedData = wifft(whitenedData);

      } // generateReport

      //////////////////////////////////////////////////////////////////////////
      //                     add channel to summary report                    //
      //////////////////////////////////////////////////////////////////////////

      // identify most significant raw transform tile
      wlog(debugLevel, 2, "  measuring peak properties");
      thresholdReferenceTime = whitenedProperties[0].peak.centerTime();
      thresholdTimeRange[0] = -whitenedProperties[0].peak.Duration;
      thresholdTimeRange[1] = -thresholdTimeRange[0];
      thresholdFrequencyRange.resize(2);
      thresholdFrequencyRange[0] = whitenedProperties[0].peak.Frequency 
	 - whitenedProperties[0].peak.Bandwidth;
      thresholdFrequencyRange[1] = whitenedProperties[0].peak.Frequency 
	 + whitenedProperties[0].peak.Bandwidth;
      thresholdQRange = emptyRange;
      wmeasure rawProperties(rawTransform, tiling, startTime, 
			     thresholdReferenceTime, thresholdTimeRange, 
			     thresholdFrequencyRange, thresholdQRange, 
			     debugLevel);
      //rawProperties.dump(cout);
 
      // most significant tile properties
      Time mostSignificantTime =  whitenedProperties[0].peak.centerTime();
      double mostSignificantFrequency = whitenedProperties[0].peak.Frequency;
      double mostSignificantQ =  whitenedProperties[0].peak.Q;
      double mostSignificantNormalizedEnergy =
	 whitenedProperties[0].peak.NormalizedEnergy;

      //----------------------------------------------------------------------
      //   This was changed to use signalAmplitude[0]
      // double mostSignificantAmplitude = rawProperties{1}.peakAmplitude;
      // double mostSignificantAmplitude = rawProperties[0].signalAmplitude;
      double mostSignificantAmplitude = rawProperties[0].signal.Amplitude;

      // write most significant tile properties to text summary file
      wlog(debugLevel, 2, "  writing text summary");
      fprintf(textSummaryFID, 
	      "%-30s %#020.9f %#09.3e %#09.3e %#09.3e %#09.3e\n", 
	      channelNameString.c_str(), mostSignificantTime.totalS(), 
	      mostSignificantFrequency, mostSignificantQ, 
	      mostSignificantNormalizedEnergy, mostSignificantAmplitude);
      fflush(textSummaryFID);

      // write most significant tile properties to xml summary file
      wlog(debugLevel, 2, "  writing xml summary");
      fprintf(xmlSummaryFID, 
	      "\"%s\",%#020.9f,%#09.3e,%#09.3e,%#09.3e,%#09.3e,\n", 
	      channelNameString.c_str(), mostSignificantTime.totalS(), 
	      mostSignificantFrequency, mostSignificantQ, 
	      mostSignificantNormalizedEnergy, mostSignificantAmplitude);

      //////////////////////////////////////////////////////////////////////////
      //                       add channel to html report                     //
      //////////////////////////////////////////////////////////////////////////

      if (generateReport) {
	 ostringstream oss;

	 // html for most significant tile properties
	 oss << fixed << setprecision(3) << mostSignificantTime.totalS();
	 string mostSignificantTimeHtml = oss.str();
	 oss.str("");

	 oss << fixed << setprecision(1) << mostSignificantFrequency;
	 string mostSignificantFrequencyHtml =  oss.str();
	 oss.str("");

	 oss << fixed << setprecision(1) << mostSignificantQ;
	 string  mostSignificantQHtml =   oss.str();
	 oss.str("");

	 expfmt(mostSignificantNormalizedEnergy, oss);
	 string mostSignificantNormalizedEnergyHtml =  oss.str();
	 oss.str("");   Time eventTime = Time(sec, long((time-sec)*1e9));

   // name of html file
   string htmlFile = "index.html";

   // name of text summary file
   string textSummaryFile = "summary.txt";

   // name of xml summary file

	 expfmt(mostSignificantAmplitude, oss);
	 string mostSignificantAmplitudeHtml =  oss.str();
	 oss.str("");

#ifdef OMEGA_MATLAB_BUGS
	 double snrsq  = 2 * mostSignificantNormalizedEnergy;
#else
	 double snrsq  = 2 * mostSignificantNormalizedEnergy - 1;
	 if (snrsq < 0) snrsq = 0;
#endif
	 oss << fixed << setprecision(1) << sqrt(snrsq);
	 string mostSignificantSNRHtml =  oss.str();
	 oss.str("");

	 // begin html channel entry
	 wlog(debugLevel, 2, "  writing html report");
	 htmlFID << "<a name=\"" << channelNameString << "\"></a>" << endl;
	 htmlFID << "<h3>" << endl;
	 htmlFID << "<input id=\"input_" << channelNameString 
		 << "\" type=\"checkbox\" checked" << endl;
	 htmlFID << "       onclick=\"toggleVisible('" << channelNameString 
		 << "');\" />" << endl; 
	 htmlFID << "<a href=\"https://cis.ligo.org/channel/byname/" 
		 << channelNameString << "\">" << endl;
	 htmlFID << channelNameString << endl;
	 htmlFID << "</a>" << endl;
	 htmlFID << "</h3>" << endl;

	 htmlFID << "<div id=\"div_" << channelNameString 
		 << "\" style=\"display: block;\">" << endl;

	 htmlFID << "most significant tile:" << endl
		 << " t = " << mostSignificantTimeHtml << " s," << endl
		 << " f = " << mostSignificantFrequencyHtml << " Hz," << endl
		 << " Q = " <<mostSignificantQHtml << "," << endl
		 << " Z = " <<mostSignificantNormalizedEnergyHtml << "," << endl
		 << " X = " << mostSignificantAmplitudeHtml 
		 << " Hz<sup>-&frac12;</sup>," << endl 
		 << " SNR = " << mostSignificantSNRHtml << endl
		 << "<br />" << endl; 

	 oss << "'" << fixed << setprecision(2) << plotTimeRanges[0] << "'";
	 for (size_t i=1; i < plotTimeRanges.size(); i++) {
	    oss << ", '" << fixed << setprecision(2) << plotTimeRanges[i] << "'";
	 }
	 string timeRangeString = oss.str();

	 htmlFID << "  time series:" << endl;
	 htmlFID << "<a href=\"javascript:showImage('" << eventTimeString 
		 << "', '" << channelNameString << "'," << endl;
	 htmlFID << "[" << timeRangeString << "], 'timeseries_raw');\">raw</a>,"
		 << endl;
	 htmlFID << "<a href=\"javascript:showImage('" << eventTimeString 
		 << "', '" << channelNameString << "'," << endl;
	 htmlFID << "[" << timeRangeString 
		 << "], 'timeseries_highpassed');\">high passed</a>," << endl;
	 htmlFID << "<a href=\"javascript:showImage('" << eventTimeString 
		 << "', '" << channelNameString << "'," << endl;
	 htmlFID << "[" << timeRangeString 
		 << "], 'timeseries_whitened');\">whitened</a>" << endl;
	 htmlFID << "| spectrogram:" << endl;
	 htmlFID << "<a href=\"javascript:showImage('" << eventTimeString 
		 << "', '" << channelNameString << "'," << endl;
	 htmlFID << "[" << timeRangeString 
		 << "], 'spectrogram_raw');\">raw</a>," << endl;
	 htmlFID << "<a href=\"javascript:showImage('" << eventTimeString 
		 << "', '" << channelNameString << "'," << endl;
	 htmlFID << "[" << timeRangeString 
		 << "], 'spectrogram_whitened');\">whitened</a>," << endl;
	 htmlFID << "<a href=\"javascript:showImage('" << eventTimeString 
		 << "', '" << channelNameString << "'," << endl;
	 htmlFID << "[" << timeRangeString 
		 << "], 'spectrogram_autoscaled');\">autoscaled</a>" << endl;
	 htmlFID << "| eventgram:" << endl;
	 htmlFID << "<a href=\"javascript:showImage('" << eventTimeString 
		 << "', '" << channelNameString << "'," << endl;
	 htmlFID << "[" << timeRangeString 
		 << "], 'eventgram_raw');\">raw</a>," << endl;
	 htmlFID << "<a href=\"javascript:showImage('" << eventTimeString 
		 << "', '" << channelNameString << "'," << endl;
	 htmlFID << "[" << timeRangeString 
		 << "], 'eventgram_whitened');\">whitened</a>,";
	 htmlFID << "<a href=\"javascript:showImage('" << eventTimeString 
		 << "', '" << channelNameString << "'," << endl;
	 htmlFID << "[" << timeRangeString 
		 << "], 'eventgram_autoscaled');\">autoscaled</a>" << endl;
	 htmlFID << "<br />" << endl;

	 ///////////////////////////////////////////////////////////////////////
	 //                     begin loop over time ranges                   //
	 ///////////////////////////////////////////////////////////////////////

	 //  Set figure style parameters
	 if (!configuration[channelNumber].colorMap.empty()) {
	    figureHandle.set_palette(configuration[channelNumber].colorMap);
	 }
	 
	 
	 // begin loop over time ranges
	 for (size_t iPlTiR=0; iPlTiR < plotTimeRanges.size(); iPlTiR++) {
	    double plotTimeRange =  plotTimeRanges[iPlTiR];
	    dble_vect autoNormalizedEnergyRange;

	    // report status
	    if (debugLevel >= 2) {
	       cout << "  processing " << plotTimeRange << " second time range" 
		    << endl;
	    }

	    // determine start and stop times of figures
	    //notused double plotStartTime = timeRange / 2 - plotTimeRange / 2;
	    //notused double plotStopTime = timeRange / 2 + plotTimeRange / 2;

	    ////////////////////////////////////////////////////////////////////
	    //                      plot raw time series                      //
	    ////////////////////////////////////////////////////////////////////

	    // plot raw time series
	    wlog(debugLevel, 2, "    plotting raw time series");
	    figureHandle.clear();
	    figureHandle.wtimeseries(rawData[0], centerTime, -plotTimeRange/2,
				     plotTimeRange/2, channelNameString);

	    // print raw time series to file
	    wlog(debugLevel, 2, "    printing raw time series");
	    ostringstream figName;
	    figName << eventTimeString << "_" << channelNameString << "_" 
		    << fixed << setprecision(2) << plotTimeRange 
		    << "_timeseries_raw";
	    string figBasePath = outputDirectory + "/" + figName.str();
	    figureHandle.wprintfig(figBasePath);
	    figName.str("");

	    ////////////////////////////////////////////////////////////////////
	    //              plot high pass filtered time series               //
	    ////////////////////////////////////////////////////////////////////

	    // plot high pass filtered time series
	    wlog(debugLevel, 2, "    plotting high pass filtered time series");
	    figureHandle.clear();
	    figureHandle.wtimeseries(conditioner.highPassedData(), centerTime, 
				     -plotTimeRange/2, plotTimeRange/2, 
				     channelNameString);

	    // print high pass filtered time series to file
	    wlog(debugLevel, 2, "    printing high passed time series");
	    figName << eventTimeString << "_" << channelNameString << "_" 
		    << setprecision(2) << plotTimeRange << "_timeseries_highpassed";
	    figBasePath = outputDirectory + "/" + figName.str();
	    figureHandle.wprintfig(figBasePath);
	    figName.str("");

	    ////////////////////////////////////////////////////////////////////
	    //                   plot whitened time series                    //
	    ////////////////////////////////////////////////////////////////////

	    //--------------------------  Plot whitened time series
	    wlog(debugLevel, 2, "    plotting whitened time series");
	    figureHandle.clear();
	    figureHandle.wtimeseries(conditioner.whitenedData(), centerTime, 
				     -plotTimeRange/2, plotTimeRange/2, 
				     channelNameString);

	    // print whitened time series to file
	    wlog(debugLevel, 2, "    printing whitened time series");
	    figName << eventTimeString << "_" << channelNameString << "_" 
		    << fixed << setprecision(2) << plotTimeRange 
		    << "_timeseries_whitened";
	    figBasePath = outputDirectory + "/" + figName.str();
	    figureHandle.wprintfig(figBasePath);
	    figName.str("");

	    ////////////////////////////////////////////////////////////////////
	    //                      plot raw spectrogram                      //
	    ////////////////////////////////////////////////////////////////////
	    dble_vect tRange(2);
	    tRange[0] = -plotTimeRange/2;
	    tRange[1] =  plotTimeRange/2;

	    dble_vect qRange(1, mostSignificantQ);

	    // plot raw spectrogram
	    wlog(debugLevel, 2, "    plotting raw spectrogram");
	    figureHandle.clear();
	    figureHandle.wspectrogram(rawTransform[0], tiling, centerTime, 
				      tRange, plotFrequencyRange, 
				      qRange, plotNormalizedEnergyRange, 
				      plotHorizontalResolution);

	    // print autoscaled spectrogram to file
	    wlog(debugLevel, 2, "    printing raw spectrogram");
	    figName << eventTimeString << "_" << channelNameString << "_" 
		    << fixed << setprecision(2) << plotTimeRange 
		    << "_spectrogram_raw";
	    figBasePath = outputDirectory + "/" + figName.str();
	    figureHandle.wprintfig(figBasePath);
	    figName.str("");
    
	    ////////////////////////////////////////////////////////////////////
	    //                   plot whitened spectrogram                    //
	    ////////////////////////////////////////////////////////////////////
	    wlog(debugLevel, 2, "    plotting whitened spectrogram");
	    figureHandle.clear();
	    figureHandle.wspectrogram(whitenedTransform[0], tiling,
				      centerTime, tRange, 
				      plotFrequencyRange, 
				      qRange, plotNormalizedEnergyRange, 
				      plotHorizontalResolution);

	    // print whitened spectrogram to file
	    wlog(debugLevel, 2, "    printing whitened spectrogram");
	    figName << eventTimeString << "_" << channelNameString << "_" 
		    << fixed << setprecision(2) << plotTimeRange 
		    << "_spectrogram_whitened";
	    figBasePath = outputDirectory + "/" + figName.str();
	    figureHandle.wprintfig(figBasePath);
	    figName.str("");

	    ////////////////////////////////////////////////////////////////////
	    //                  plot autoscaled spectrogram                   //
	    ////////////////////////////////////////////////////////////////////
	    wlog(debugLevel, 2, "    plotting autoscaled spectrogram");
	    figureHandle.clear();
	    figureHandle.wspectrogram(whitenedTransform[0], tiling, 
				      centerTime, tRange,
				      plotFrequencyRange, qRange, 
				      autoNormalizedEnergyRange, 
				      plotHorizontalResolution);

	    // print autoscaled spectrogram to file
	    wlog(debugLevel, 2, "    printing autoscaled spectrogram");
	    figName << eventTimeString << "_" << channelNameString << "_" 
		    << fixed << setprecision(2) << plotTimeRange 
		    << "_spectrogram_autoscaled";
	    figBasePath = outputDirectory + "/" + figName.str();
	    figureHandle.wprintfig(figBasePath);
	    figName.str("");

	    ////////////////////////////////////////////////////////////////////
	    //                         plot raw eventgram                     //
	    ////////////////////////////////////////////////////////////////////

	    // plot raw eventgram
	    wlog(debugLevel, 2, "    plotting raw eventgram");
	    figureHandle.clear();
	    if (rawEvents.numberOfChannels()) {
	       figureHandle.weventgram(rawEvents[0], tiling, centerTime, tRange,
				       plotFrequencyRange, plotDurationInflation,
				       plotBandwidthInflation, 
				       plotNormalizedEnergyRange);
	    }
	    // print autoscaled eventgram to file
	    wlog(debugLevel, 2, "    printing raw eventgram");
	    figName << eventTimeString << "_" << channelNameString << "_"
		    << fixed << setprecision(2) << plotTimeRange 
		    << "_eventgram_raw";
	    figBasePath = outputDirectory + "/" + figName.str();
	    figureHandle.wprintfig(figBasePath);
	    figName.str("");

	    ////////////////////////////////////////////////////////////////////
	    //                    plot whitened eventgram                     //
	    ////////////////////////////////////////////////////////////////////

	    // plot whitened eventgram
	    wlog(debugLevel, 2, "    plotting whitened eventgram");
	    figureHandle.clear();
	    if (whitenedEvents.numberOfChannels()) {
	       figureHandle.weventgram(whitenedEvents[0], tiling, centerTime, 
				       tRange, plotFrequencyRange, 
				       plotDurationInflation, 
				       plotBandwidthInflation, 
				       plotNormalizedEnergyRange);
	    }

	    // print whitened eventgram to file
	    wlog(debugLevel, 2, "    printing whitened eventgram");
	    figName << eventTimeString << "_" << channelNameString << "_"
		    << fixed << setprecision(2) << plotTimeRange 
		    << "_eventgram_whitened";
	    figBasePath = outputDirectory + "/" + figName.str();
	    figureHandle.wprintfig(figBasePath);
	    figName.str("");

	    ////////////////////////////////////////////////////////////////////
	    //                   plot autoscaled eventgram                    //
	    ////////////////////////////////////////////////////////////////////

	    // plot autoscaled eventgram
	    wlog(debugLevel, 2, "    plotting autoscaled eventgram");
	    figureHandle.clear();
	    if (whitenedEvents.numberOfChannels()) {
	       figureHandle.weventgram(whitenedEvents[0], tiling, centerTime, 
				       tRange, plotFrequencyRange, 
				       plotDurationInflation, 
				       plotBandwidthInflation, 
				       autoNormalizedEnergyRange);
	    }

	    // print autoscaled eventgram to file
	    wlog(debugLevel, 2, "    printing autoscaled eventgram");
	    figName << eventTimeString << "_" << channelNameString << "_"
		    << fixed << setprecision(2) << plotTimeRange 
		    << "_eventgram_autoscaled";
	    figBasePath = outputDirectory + "/" + figName.str();
	    figureHandle.wprintfig(figBasePath);
	    figName.str("");

	    ////////////////////////////////////////////////////////////////////
	    //                   add images to html report                    //
	    ////////////////////////////////////////////////////////////////////

	    // create html link
	    wlog(debugLevel, 2, "    linking images");
	    htmlFID << fixed << setprecision(2);
	    htmlFID << "<a id=\"a_" << eventTimeString << "_" 
		    << channelNameString << "_" << plotTimeRange << "\""<< endl;
	    htmlFID << "   href=\"" << eventTimeString << "_" 
		    << channelNameString << "_" << plotTimeRange 
		    << "_spectrogram_whitened.png\">" << endl;
	    htmlFID << "<img id=\"img_" <<eventTimeString  << "_" 
		    << channelNameString << "_" << plotTimeRange << "\""<< endl;
	    htmlFID << " src=\"" << eventTimeString << "_" << channelNameString
		    << "_" << plotTimeRange 
		    << "_spectrogram_whitened.thumb.png\"" << endl;
	    htmlFID << "     alt=\"" << eventTimeString << "_" 
		    << channelNameString << "_" << plotTimeRange << "\" /></a>"
		    << endl;
	 }   // end loop time ranges

	 // end html table row
	 htmlFID << "<br />" << endl;
	 htmlFID << "</div>" << endl;
	 htmlFID << endl;
	 htmlFID.flush();

      } // if (generateReport)

   }   // end loop over configuration channels

   //-----------------------------------  Close html report
   if (generateReport) {

      // report status
      wlog(debugLevel, 1, "closing html report");

      // end any previously existing section
      if (inSectionFlag == 1) {
	 htmlFID << "</div>" << endl;
	 htmlFID << endl;
      }

      //--------------------------------  End html
      htmlFID << "</div>" << endl;
      htmlFID << endl;
      htmlFID << "<div class=\"footer\">" << endl;
      htmlFID << "Created by user " << getenv("USER") << " on " 
	      << datestr(0, 29) << " at " << datestr(0, 13) << "<br />" << endl;
      htmlFID << "</div>" << endl;
      htmlFID << endl;
      htmlFID << "</body>" << endl;
      htmlFID << "</html>" << endl;

      // close html file
      htmlFID.close();

   } // if (generateReport)


   //-----------------------------------  Close text summary report
   wlog(debugLevel, 1, "closing text summary");
   fclose(textSummaryFID);

   //-----------------------------------  Close xml summary report
   wlog(debugLevel, 1, "closing xml summary");
   fprintf(xmlSummaryFID, "</Stream>\n");
   fprintf(xmlSummaryFID, "</Table>\n");
    fprintf(xmlSummaryFID, "</LIGO_LW>\n");
   fclose(xmlSummaryFID);

   //-----------------------------------  Exit
   if (debugLevel >= 1) {
      cout << "finished on " << datestr(0, 29) 
	   << " at " << datestr(0, 13) << endl;
   }
   //figureHandle.close();
}

//======================================  Search channel constructor.
search_chan::search_chan(void) {
   clear();
}

//======================================  Clear search channel descriptor.
void
search_chan::clear(void) {
   channelName.clear();
   frameType.clear();
   sampleFrequency = 0;
   searchTimeRange = 0;
   searchFrequencyRange.clear();
   searchQRange.clear();
   searchMaximumEnergyLoss = 0;
   whiteNoiseFalseRate = 0;
   eventThreshold = 0;
   searchWindowDuration = 0;
   plotTimeRanges.clear();
   plotFrequencyRange.clear();
   plotNormalizedEnergyRange.clear();
   alwaysPlotFlag = false;
   colorMap.clear();
   zUnits.clear();
}

//======================================  Scan the configuration.
void
wpipe::scan_config(const std::string& file, const search_chan& def, 
		   config_vect& configuration, section_vect& sectList) {
   // initialize configuration structure
   int channelNumber = 0;

   // open configuration file for reading
   ifstream configurationFID(file.c_str());
   if (!configurationFID.is_open()) {
      throw runtime_error(string("Unable to open configuration file: ") + file);
   }

   search_chan temp_chan;

   // begin loop over configuration file
   while (!configurationFID.eof()) {

      // read one line from configuration file
      string configurationLine;
      getline(configurationFID, configurationLine);
    
      // remove any comments
      string::size_type commentIndices = configurationLine.find("#");
      if (commentIndices != string::npos) {
	 configurationLine.erase(commentIndices);
      }

      // remove leading and trailing blanks
      configurationLine = deblank(configurationLine);

      // if (empty line, skip to the next line
      if (configurationLine.empty()) {
	 continue;
      }

      // check for new section
      if (configurationLine[0] == '[') {

	 // locate field separator
	 string::size_type commaIndex = configurationLine.find(",");

	 // if (field separator not located, report syntax error
	 if (commaIndex == string::npos) {
	    error(string("syntax error processing configuration file ")
		  + file + ":\n" + configurationLine);
	 }

	 //--------------------------------  Define a section...
	 section sect;

	 //-------------------------------- extract section index
	 sect.Index=configurationLine.substr(1, commaIndex-1);

	 //-------------------------------- extract section name
	 sect.Name=configurationLine.substr(commaIndex+1);

	 // remove leading and trailing blanks
	 sect.Index = deblank(sect.Index);
	 sect.Name = deblank(sect.Name);
	 sect.Name.erase(sect.Name.size()-1, 1);

	 // standardize section names
	 sect.Index = strrep(sect.Index, ";", ":");
	 sect.Name  = strrep(sect.Name, ";", ":");

	 // determine initial visibility
	 if (sect.Index == "Context") {
	    sect.Checked = "checked";
	    sect.Display = "block";
	 } else if (sect.Index == "Gravitational") {
	    sect.Checked = "checked";
	    sect.Display = "block";
	 } else {
	    sect.Checked = "checked";
	    sect.Display = "block";
	 }
      
	 // record first channel in section
	 sect.Start = configuration.size();

	 // increment section number
	 sectList.push_back(sect);

	 // continue to next line
	 continue;

      }

      // check for beginning of new channel configuration
      if (configurationLine == "{") {

	 // increment channel number
	 temp_chan.clear();
	 channelNumber = configuration.size();

	 // continue to next line
	 continue;
      }

      // check for end of existing channel configuration
      if (configurationLine == "}") {

	 // validate channel configuration
	 size_t nChan = temp_chan.channelName.size();
	 if (!nChan) {
	    cerr << "channel name not specified for channel number " 
		 << channelNumber << endl;
	    error("channel name not specified");
	 }
	 if (temp_chan.frameType.size() < nChan) {
	    cerr << "frame type(s) not specified for channel number " 
		 << channelNumber << endl;
	    error("frame type(s) not specified");
	 }
	 if (temp_chan.timeShifts.size() < nChan) {
	    temp_chan.timeShifts.resize(nChan, 0.0);
	 }
	 if (temp_chan.sampleFrequency == 0) {
	    cerr << "WSCAN:incompleteConfiguration" 
		 << "sample frequency not specified for channel number"
		 << channelNumber << endl;
	    temp_chan.sampleFrequency = def.sampleFrequency;
	 }
	 if (temp_chan.searchTimeRange == 0) {
	    cerr << "WSCAN:incompleteConfiguration"
		 << "search time range not specified for channel number "
		 << channelNumber << endl;
	    temp_chan.searchTimeRange = def.searchTimeRange;
	 }
	 if (fmod(temp_chan.searchTimeRange, 1) != 0) {
	    cerr << "WSCAN:invalidConfiguration: " 
		 << "search time range non integer for channel number "
		 << channelNumber << endl;
	    error("WSCAN:invalidConfiguration");
	 }
	 if (temp_chan.searchFrequencyRange[1] < 0) {
	    cout << "WSCAN:incompleteConfiguration "
		 << "search frequency range not specified for channel number "
		 << channelNumber << endl;
	    temp_chan.searchFrequencyRange[0] = def.searchFrequencyRange[0];
	    temp_chan.searchFrequencyRange[1] = def.searchFrequencyRange[1];
	 }
	 if (temp_chan.searchQRange[1] == 0) {
	    cout << "WSCAN:incompleteConfiguration "
		 <<	"search Q range not specified for channel number "
		 << channelNumber << endl;
	    temp_chan.searchQRange[0] = def.searchQRange[0];
	    temp_chan.searchQRange[1] = def.searchQRange[1];
	 }
	 if (temp_chan.searchMaximumEnergyLoss == 0) {
	    cerr << "WSCAN:incompleteConfiguration"
		 << "search maximum energy loss not specified for channel "
		 << "number " << channelNumber << endl;
	    temp_chan.searchMaximumEnergyLoss = def.searchMaximumEnergyLoss;
	 }
	 if (!temp_chan.whiteNoiseFalseRate && !temp_chan.eventThreshold) {
	    cerr << "WSCAN:incompleteConfiguration"
		 << "event threshold / false rate not specified "
		 << "for channel number " << channelNumber << endl;
#ifdef OMEGA_MATLAB_BUGS
	    temp_chan.searchMaximumEnergyLoss = def.whiteNoiseFalseRate;
#else
	    temp_chan.whiteNoiseFalseRate = def.whiteNoiseFalseRate;
	    temp_chan.eventThreshold      = def.eventThreshold;
#endif
	 }
	 if (temp_chan.searchWindowDuration == 0) {
	    cerr << "WSCAN:incompleteConfiguration"
		 << "search window duration not specified for channel number "
		 << channelNumber << endl;
	    temp_chan.searchWindowDuration =  def.searchWindowDuration;
	 }
	 if (temp_chan.plotTimeRanges.empty()) {
	    cerr << "WSCAN:incompleteConfiguration" 
		 <<	"plot time ranges not specified for channel number "
		 << channelNumber << endl;
	    temp_chan.plotTimeRanges = def.plotTimeRanges;
	 }
	 if (temp_chan.plotFrequencyRange.empty()) {
	    //cerr << "WSCAN:incompleteConfiguration "
	    //     << "plot frequency range not specified for channel number "
	    //     << channelNumber << endl;
	    temp_chan.plotFrequencyRange = def.plotFrequencyRange;
	 }
	 if (temp_chan.plotNormalizedEnergyRange.empty()) {
	    cerr << "WSCAN:incompleteConfiguration "
		 <<"plot normalized energy range not specified for channel number "
		 << channelNumber << endl;
	    temp_chan.plotNormalizedEnergyRange = def.plotNormalizedEnergyRange;
	 }
	 //if (temp_chan.alwaysPlotFlag == false) {
	 //cerr << "WSCAN:incompleteConfiguration"
	 //     << "always plot flag not specified for channel number "
	 //     << channelNumber << endl;
	 //temp_chan.alwaysPlotFlag = defaultAlwaysPlotFlag;
	 //}

	 configuration.push_back(temp_chan);
	 channelNumber = configuration.size();
	 // continue to next line
	 continue;
      }

      // locate field separator
      string::size_type colonIndex = configurationLine.find(":");

      // if (field separator not located, report syntax error
      if (colonIndex == string::npos) {
	 cerr << "syntax error processing configuration file "
	      << file << ":" << endl << configurationLine << endl;
	 error("syntax error processing configuration file");
      }
    
      // parse configuration line
      string parameterName = configurationLine.substr(0, colonIndex);
      string parameterValue = configurationLine.substr(colonIndex + 1);
      parameterName = deblank(parameterName);
      parameterValue = deblank(parameterValue);
      const char* parValue = parameterValue.c_str();

      // assign parameters based on name
      if (parameterName == "channelName") {
	 temp_chan.channelName = eval_str(parameterValue);
      } else if (parameterName == "frameType") {
	 temp_chan.frameType = eval_str(parameterValue);
      } else if (parameterName == "timeShifts") {
	 temp_chan.timeShifts = eval(parameterValue);
      } else if (parameterName ==  "sampleFrequency") {
	 temp_chan.sampleFrequency = strtod(parValue, 0);
      } else if (parameterName ==  "searchTimeRange") {
	 temp_chan.searchTimeRange = strtod(parValue, 0);
      } else if (parameterName ==  "searchFrequencyRange") {
	 temp_chan.searchFrequencyRange = eval(parameterValue);
      } else if (parameterName ==  "searchQRange") {
	 temp_chan.searchQRange = eval(parameterValue);
      } else if (parameterName ==  "searchMaximumEnergyLoss") {
	 temp_chan.searchMaximumEnergyLoss = strtod(parValue, 0);
      } else if (parameterName ==  "whiteNoiseFalseRate") {
	 temp_chan.whiteNoiseFalseRate = strtod(parValue, 0);
      } else if (parameterName ==  "eventThreshold") {
	 temp_chan.eventThreshold = strtod(parValue, 0);
      } else if (parameterName ==  "searchWindowDuration") {
	 temp_chan.searchWindowDuration = strtod(parValue, 0);
      } else if (parameterName ==  "plotTimeRanges") {
	 temp_chan.plotTimeRanges = eval(parameterValue);
      } else if (parameterName ==  "plotFrequencyRange") {
	 temp_chan.plotFrequencyRange = eval(parameterValue);
      } else if (parameterName ==  "plotNormalizedEnergyRange") {
	 temp_chan.plotNormalizedEnergyRange = eval(parameterValue);
      } else if (parameterName ==  "alwaysPlotFlag") {
	 if (parameterValue == "true") {
	    temp_chan.alwaysPlotFlag = true;
	 } else if (parameterValue == "false") {
	    temp_chan.alwaysPlotFlag = false;
	 } else {
	    temp_chan.alwaysPlotFlag = strtol(parValue,0,0)!=0;
	 }
      }
      else if (parameterName ==  "colorMap") {
	 temp_chan.colorMap=parameterValue;
      }
      else if (parameterName ==  "zUnits") {
	 temp_chan.zUnits=parameterValue;
      }
      else {
	 cerr << "unknown configuration parameter " << parameterName << endl;
	 error("unknown configuration parameter");
      }

      // end loop over channel configuration file
   }

   // close configuration file
   configurationFID.close();
}

//======================================  Diaplay channel parameters.
void
search_chan::display(std::ostream& out) const {
   out << "channelName:               ";
   wpipe::display(channelName, out) << endl;
   out << "frameType:                 ";
   wpipe::display(frameType, out) << endl;
   out << "timeShifts:                ";
   wpipe::display(timeShifts, out) << endl;
   out << "sampleFrequency:           " << sampleFrequency << endl;
   out << "searchTimeRange:           " << searchTimeRange << endl;
   out << "searchFrequencyRange:      "; 
   wpipe::display(searchFrequencyRange, out) << endl;
   out << "searchQRange:              ";
   wpipe::display(searchQRange, out) << endl;
   out << "searchMaximumEnergyLoss:   " << searchMaximumEnergyLoss << endl;
   out << "whiteNoiseFalseRate:       " << whiteNoiseFalseRate << endl;
   out << "searchWindowDuration:      " << searchWindowDuration << endl;
   out << "plotTimeRange:             ";
   wpipe::display(plotTimeRanges, out) << endl;
   out << "plotFrequencyRange:        ";
   wpipe::display(plotFrequencyRange, out) << endl;
   out << "plotNormalizedEnergyRange: ";
   wpipe::display(plotNormalizedEnergyRange, out) << endl;
   out << "alwaysPlotFlag:            " << boolstr_tf(alwaysPlotFlag) << endl;
   out << "colorMap:                  " << colorMap << endl;
   out << "zUnits:                    " << zUnits << endl;
}
