/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "wtypes.hh"
#include "wframecache.hh"
#include "ParseLine.hh"
#include "matlab_fcs.hh"
#include "scandir.hh"
#include <sstream>
#include <iostream>
#include <algorithm>

using namespace wpipe;
using namespace std;

//=====================================  frame cache constructor.
wframecache::wframecache(const std::string& file) 
   : debugLevel(0)
{
   string path = file;
   while (! path.empty()) {
      string::size_type ipos = path.find(',');
      string nx_path = path.substr(0, ipos);
      cout << "parse file: " << nx_path << endl;
      if (scandir::test_directory(nx_path)) parse_directory(nx_path);
      else                                  parse_cacheFile(nx_path);
      if (ipos != string::npos) ipos++;
      path.erase(0, ipos);
   }
   coalesce();
}

//=====================================  Coalesce the cache contents.
void
wframecache::add_group(const string& prefix, gps_type start, gps_type stop, 
		       gps_type delta, const string& dir) {
   if (debugLevel) {
      cout << "frame group prefix: " << prefix << " start: " << start
	   << " stop: " << stop << " delta: " << delta << " dir: " << dir
	   << endl;
   }
   frame_group fg(prefix, start, stop, delta, dir);
   int adjid = find_adjacent(fg);
   if (adjid < 0) frameCache.push_back(fg);
   else           frameCache[adjid].combine(fg);
}

//=====================================  Coalesce the cache contents.
void
wframecache::coalesce(void) {
   size_t N=frameCache.size();
   if (N < 2) return;

   //----------------------------------  Sort the cache vector
   std::sort(frameCache.begin(), frameCache.end());

   size_t i=0;
   for (size_t j=1; j<N; j++) {
      if (frameCache[i].adjacent(frameCache[j])) {
	 frameCache[i].combine(frameCache[j]);
      } else {
	 i++;
	 if (i < j) frameCache[i] = frameCache[j];
      }
   }

   //---------------------------------  Erase coalesced entries.
   i++;
   if (i != N) frameCache.erase(frameCache.begin()+i, frameCache.begin()+N);
}

//=====================================  Display the cache contents.
void
wframecache::display(void) const {
   cout << "prefix start stop dt dir" << endl;
   int N = frameCache.size();
   for (int i=0; i<N; i++) {
      frameCache[i].display();
   }
}

//=====================================  Find an adjacent frame group.
int 
wframecache::find_adjacent(const frame_group& grp) const {
   int N = frameCache.size();
   for (int i=0; i<N; i++) {
      if (frameCache[i].adjacent(grp)) return i;
   }
   return -1;
}

//=====================================  frame cache constructor.
void
wframecache::parse_directory(const std::string& dir) {
   scandir sd(dir);
   while (sd.next_entry()) {
      if (sd.is_frame()) {
	 gps_type gps = sd.gps();
	 gps_type dt  = sd.duration();
	 add_group(sd.prefix(), gps, gps+dt, dt, dir);
      }
   }
}

//=====================================  Parse a frame cache file.
void
wframecache::parse_cacheFile(const std::string& file) {
   ParseLine pl(file.c_str());
   if (!pl.isOpen()) {
      cerr << "Unable to open frame cache file: " << file << endl;
      error("Can not open frame cache file");
   }
   while (pl.getLine() >= 0) {
      int nArg = pl.getCount();
      if (!nArg) continue;

      //--------------------------------  Parse ligo_data_find cache files
      if (nArg == 5) {
	 //-----------------------------  Extract the directory path
	 string path=pl[4];
	 if (path.substr(0, 17) != "file://localhost/") {
	    cerr << "cache path in " << file << "  is not 'file://localhost/'" 
		 << endl;
	    break;
	 }
	 path.erase(0, 16);
	 size_t inx = path.find_last_of('/');
	 if (inx == string::npos) inx = 0;

	 //-----------------------------  Calculate the other parameters.
	 string prefix = pl[0];
	 prefix += string("-") + pl[1];
	 gps_type start = pl.getInt(2);
	 gps_type delta = pl.getInt(3);
	 gps_type stop  = start + delta;
	 add_group(prefix, start, stop, delta, path.substr(0,inx));
      }

      //--------------------------------  Parse prefix, start, stop, delta, dir
      else if (nArg == 6) {
	 string prefix = pl[0];
	 prefix += string("-") + pl[1];
	 add_group(prefix, pl.getInt(2), pl.getInt(3), pl.getInt(4), pl[5]);
      }

      else {
	 cerr << "Invalid format line in: " << file << endl;
	 break;
      }
   }
}

//=====================================  get a list of frames with specified 
//                                       data
int
wframecache::get_list(const std::string& prefix, gps_type start, 
		      gps_type stop, str_vect& frames) const {
   frames.clear();
   size_t N = frameCache.size();
   for (size_t i=0; i<N ; i++) {
      if (frameCache[i].valid(prefix, start, stop)) {
	 frameCache[i].get_frames(start, stop, frames);
      }
   }
   return frames.size();
}

//======================================  Set the debug level;
void
wframecache::set_debug(int lvl) {
   debugLevel = lvl;
}

//=====================================  construct a frame group
wframecache::frame_group::frame_group(const std::string& prefix,
				      gps_type start, 
				      gps_type stop,
				      gps_type frame_len, 
				      const std::string& dir) 
   : _prefix(prefix), _directory(dir), _start(start), _length(frame_len),
     _end(stop)
{}

//=====================================  test for adjacent frame group
bool 
wframecache::frame_group::adjacent(const frame_group& grp) const {
   return (grp._start == _end || grp._end == _start) 
          && grp._prefix == _prefix && grp._directory == _directory;
}

//=====================================  test for adjacent frame group
void
wframecache::frame_group::combine(const frame_group& grp) {
   if (grp._start == _end)      _end   = grp._end;
   else if (grp._end == _start) _start = grp._start;
}

//=====================================  Display the frame group contents.
void
wframecache::frame_group::display(void) const {
   cout << _prefix << " " << _start << " " << _end << " " << _length 
	<< " " << _directory << endl;
}

//=====================================  Test if requested data in group
bool 
wframecache::frame_group::valid(const string& prefix, gps_type start, 
				gps_type stop) const {

   bool rc = (prefix == _prefix && start < _end && stop > _start);
   return rc;
}

//=====================================  Get a list of frames
int
wframecache::frame_group::get_frames(gps_type start, gps_type stop, 
				     str_vect& frames) const {
   int count = 0;

   //-----------------------------------  Handle special entries.
   if (_length == 0) {
      frames.push_back(_directory);
      count++;
   } 

   //-----------------------------------  Build frame list.
   else {
      for (gps_type gps=_start; gps<_end; gps += _length) {
	 if (gps < stop && gps + _length > start) {
	    frames.push_back(frame(gps));
	    count++;
	 }
      }
   }
   return count;
}

//======================================  Get name os frame
std::string 
wframecache::frame_group::frame(gps_type gps) const {
   ostringstream ostr;
   ostr << _directory << "/" << _prefix << "-" << gps << "-" 
	<< _length << ".gwf";
   return ostr.str();
}
