/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "GDSPlot.hh"
#include "TCanvas.h"
#include "TColor.h"
#include "TError.h"
#include "TH1.h"
#include "TH2.h"
#include "TList.h"
#include "TBox.h"
// #include "TPaletteAxis.h"
#include "TStyle.h"
#include "fSeries/PSD.hh"
#include "ParseLine.hh"
#include "TSeries.hh"
#include "DVector.hh"
#include "lcl_array.hh"
#include "WelchPSD.hh"
#include "Hanning.hh"

#include <cmath>
#include <iostream>
#include <stdexcept>
using namespace std;

static const char* defaultPaletteName="jet";

//======================================  Constructor
GDSPlot::GDSPlot(TCanvas* tc) 
    : mCanvas(tc), myCanvas(false), mTrace(0), mXmin(0), mXmax(0), mYmin(0),
      mYmax(0), mZmin(0), mZmax(0), mLogX(false), mLogY(false), mPalletID(0),
      mNextColor(1), mNSmooth(0)
{
    gErrorIgnoreLevel=1500;
    set_style(0);
    if (!mCanvas) {
	if (gPad) set_canvas(gPad->GetCanvas());
	else      set_canvas(new TCanvas("GDSPlot"), true);
    }
}

//======================================  Constructor
GDSPlot::~GDSPlot(void) {
    if (mCanvas && myCanvas) delete mCanvas;
}

//======================================  Plot a data vector.
void
GDSPlot::boxes(size_t nbox, const double* x0, const double* x1, 
	       const double* y0, const double* y1, const double* z) {
    TH2D slate("slate", mTitle.c_str(), 10, mXmin, mXmax, 10, mYmin, mYmax);
    set_palette();
    slate.SetMinimum(mZmin);
    slate.SetMaximum(mZmax);
    for (size_t i=0; i<10; i++) {
	double x = mXmin + (i+0.5)/10.0 * (mXmax - mXmin);
	for (size_t j=0; j<10; j++) {
	    double y = mYmin + (j+0.5)/10.0 * (mYmax - mYmin);
	    slate.Fill(x, y, 0.0001);
	}
    }

    set_axes(slate);
    Int_t ncolors = TColor::GetNumberOfColors();
    slate.SetContour(ncolors, 0);
    if (!mTitle.empty())  slate.SetTitle(mTitle.c_str());
    TH1* cpy = slate.DrawCopy("colz");
    Int_t ndivz = abs(cpy->GetContour());
    if (ndivz == 0) throw runtime_error("GDSPlot:boxes No contours");
    Double_t scale = ndivz / (mZmax - mZmin);
    mCanvas->Update();
    //    TPaletteAxis* zax=(TPaletteAxis*)(cpy->GetListOfFunctions()
    //				      ->FindObject("palette"));
    // if (!zax) {
    //	cerr << "Yikes! Can't find the z-axis palette" << endl;
    //	throw runtime_error("GDSPlot:boxes No Z palette");
    // }
    for (size_t i=0; i<nbox; i++) {
	double z0 = z[i];
	TBox* box = new TBox(x0[i], y0[i], x1[i], y1[i]);
	box->SetBit(kCanDelete);
	//------------------------------ Note no log z values!
	if (z0 >= mZmax) z0 = mZmin + 0.99 * (mZmax - mZmin);
	if (z0 <  mZmin) z0 = mZmin;

	// box->SetFillColor(zax->GetValueColor(z0));
	//------------------------------  Replacement code stolen from 
	//                                TPaletteAxis::GetValueColor
	Int_t color = Int_t(0.01 + (z0 - mZmin) * scale);

	Int_t theColor = Int_t((color + 0.99) * Double_t(ncolors) 
			       / Double_t(ndivz));
	Int_t colorIndex = gStyle->GetColorPalette(theColor);
	box->SetFillColor(colorIndex);
	//-------------------------------------------------------
	box->Draw();
    }
    mTrace++;
}

//======================================  Plot a data vector.
void
GDSPlot::new_plot(void) {
    if (mCanvas) mCanvas->Clear();
    mTrace = 0;
    mXmin = 0;
    mXmax = 0;
    mYmin = 0;
    mYmax = 0;
    mZmin = 0;
    mZmax = 0;
    mLogX = false;
    mLogY = false;
    mNextColor = 1;
}

//======================================  Plot a PSD (log-log)
void 
GDSPlot::plot(const containers::PSD& psd) {
    double fLow = psd.getLowFreq();
    double dF   = psd.getFStep();
    xlog(3 * max(fLow, dF) < psd.getHighFreq());
    ylog(true);
    plot(psd.refDVect(), fLow, dF);
}

//======================================  Plot a PSD (log-log)
void 
GDSPlot::plot(const TSeries& ts) {
    xlog(false);
    plot(*ts.refDVect(), 0.0, ts.getTStep());
}

//======================================  Plot a data vector.
void 
GDSPlot::plot(const DVector& dvec, double xmin, double dx,
	      const string& title) {
    int bMin = 0;
    int bMax = dvec.size();
    if (mLogX && xmin <= 0) {
	bMin = int(xmin / dx) + 1;
    }

  //------------------------------------  Set mXmin, mXmax, bMin, bMax
  if (mXmax <= mXmin) {
    mXmin = xmin + bMin * dx;
    mXmax = xmin + bMax * dx;
  } 
  else {
    if (mXmin > xmin + bMin * dx) bMin = int((mXmin - xmin) / dx + 0.5);
    if (mXmax < xmin + bMax * dx) bMax = int((mXmax - xmin) / dx + 0.5);
  }

  //------------------------------------  get the data to be plotted
  int nBin = bMax - bMin;
  lcl_array<double> ytemp(nBin);
  nBin = dvec.getData(bMin, nBin, ytemp);
  bMax = nBin + bMin;

  //------------------------------------  Get the minimum and maximum
  double ymin  = ytemp[0];
  double yminp = ymin;
  double ymax  = ytemp[0];
  for (int i=1; i<nBin; i++) {
      double y = ytemp[i];
      if (yminp <= 0 || (y>0 && y<yminp)) yminp = y;
      if (y < ymin) ymin = y;
      if (y > ymax) ymax = y;
  }

  //------------------------------------  Fill a histogram
  TH1D histo("dvect", title.c_str(), nBin, xmin+dx*bMin, xmin+dx*bMax);
  for (int i=0; i<nBin; ++i) {
    histo.SetBinContent(i+1, ytemp[bMin + i]);
  }

  //------------------------------------  Optional smoothing
  if (mNSmooth) histo.Smooth(mNSmooth, mSmoothOpt.c_str());

  //------------------------------------  Draw it
  histo.SetLineColor(mNextColor);
  if (!mTrace) {
    if (mLogX) mCanvas->SetLogx(1);
    else       mCanvas->SetLogx(0);

    if (mLogY) {
	mCanvas->SetLogy(1);
	if (mYmin <= 0 && yminp > 0) mYmin = 0.8 * yminp;
	mYmax = 1.25 * ymax;
	if (mYmin < mYmax * 1e-20) mYmin = mYmax * 1e-20;
	histo.SetMinimum(mYmin);
	histo.SetMaximum(mYmax);
    } else {
	mCanvas->SetLogy(0);
    }
    mCanvas->SetTicks(1);

    set_axes(histo);
    histo.DrawCopy();
  }
  else {
    histo.DrawCopy("Same");
  }
  mTrace++;
  mNextColor = mTrace + 1;
}

//======================================  Print canvas to the specified file
void
GDSPlot::print(const std::string& file) const {
    if (!mTrace) return;
    mCanvas->Print(file.c_str());
}

//======================================  Read a palette file
void
GDSPlot::read_palette(const string& palette) {
    std::vector<double> x, red, green, blue;

    //----------------------------------  Open the file
    ParseLine pl(palette.c_str());
    if (!pl.isOpen()) {
	throw runtime_error("GDSPlot::read_palette can not read palette file");
    }

    //----------------------------------- Loop until the end
    while (pl.getLine() >= 0) {
	int nArg = pl.getCount();
	if (nArg == 0) continue;
	if (nArg > 4) {
	    throw runtime_error("GDSPlot::read_palette invalid file format");
	}
	x.push_back(pl.getDouble(0));
	red.push_back(pl.getDouble(1));
	green.push_back(pl.getDouble(2));
	blue.push_back(pl.getDouble(3));
    }

    //---------------------------------- Set the pallette
    set_palette(x.size(), &x.at(0), &red.at(0), &green.at(0), &blue.at(0));
    mPalletID = 99;
}

//======================================  Set the canvas
void 
GDSPlot::set_axes(TH1& slate) {
    if (!mXlabel.empty()) {
	TAxis* axis = slate.GetXaxis();
	axis->CenterTitle(kTRUE);
	slate.SetXTitle(mXlabel.c_str());
    }

    if (!mYlabel.empty()) {
	TAxis* axis = slate.GetYaxis();
	axis->CenterTitle(kTRUE);
	slate.SetYTitle(mYlabel.c_str());
    }

    if (!mZlabel.empty()) {
	TAxis* axis = slate.GetZaxis();
	axis->CenterTitle(kTRUE);
	slate.SetZTitle(mZlabel.c_str());
    }
}

//======================================  Set the canvas
void 
GDSPlot::set_canvas(TCanvas* tc, bool koe) {
    mCanvas  = tc;
    myCanvas = koe;
}

//======================================  Set color of the next plotted trace
void 
GDSPlot::set_color(int tc) {
    mNextColor = tc;
}

//======================================  Set the color pallette
void
GDSPlot::set_palette(void) {
    if (!mPalletID) {
	set_palette(defaultPaletteName);
    }
}

//======================================  Set the color pallette
void
GDSPlot::set_palette(const string& p_spec) {
    int rev = 0;
    string palette(p_spec);
    if (palette.empty()) palette=defaultPaletteName;
    size_t psufx = p_spec.rfind("_r");
    if (p_spec.size() > 2 && psufx == p_spec.size() - 2) {
	rev = 100;
	palette.erase(psufx, 2);
    }
    
    //---------------------------------- Jet color map
    if (palette == "jet") {
	if (mPalletID == 1 + rev) return;
	UInt_t Number = 6;
	Double_t Red[6]    = {0.000, 0.000, 0.000, 1.000, 1.000, 0.500};
	Double_t Green[6]  = {0.000, 0.000, 1.000, 1.000, 0.000, 0.000};
	Double_t Blue[6]   = {0.500, 1.000, 1.000, 0.000, 0.000, 0.000};
	Double_t Length[6] = {0.000, 0.125, 0.375, 0.625, 0.875, 1.000};
	set_palette(Number, Length, Red, Green, Blue, rev);
	mPalletID = 1 + rev;
    }

    //---------------------------------- Hot color map
    else if (palette == "hot") {
	if (mPalletID == 2 + rev) return;
	UInt_t Number = 4;
	Double_t Red[4]    = {0.000, 1.000, 1.000, 1.000};
	Double_t Green[4]  = {0.000, 0.000, 1.000, 1.000};
	Double_t Blue[4]   = {0.000, 0.000, 0.000, 1.000};
	Double_t Length[4] = {0.000, 0.375, 0.750, 1.000};
	set_palette(Number, Length, Red, Green, Blue, rev);
	mPalletID = 2 + rev;
    }

    //---------------------------------- Copper color map
    else if (palette == "copper") {
	if (mPalletID == 3 + rev) return;
	UInt_t Number = 3;
	Double_t Red[3]    = {0.000, 1.000, 1.000};
	Double_t Green[3]  = {0.000, 0.640, 0.800};
	Double_t Blue[3]   = {0.000, 0.400, 0.500};
	Double_t Length[3] = {0.000, 0.800, 1.000};
	set_palette(Number, Length, Red, Green, Blue, rev);
	mPalletID = 3 + rev;
    }

    //---------------------------------- Bone color map
    else if (palette == "bone") {
	if (mPalletID == 4 + rev) return;
	UInt_t Number = 4;
	Double_t Red[4]    = {0.000, 0.320, 0.656, 1.000};
	Double_t Green[4]  = {0.000, 0.320, 0.782, 1.000};
	Double_t Blue[4]   = {0.000, 0.444, 0.782, 1.000};
	Double_t Length[4] = {0.000, 0.375, 0.750, 1.000};
	set_palette(Number, Length, Red, Green, Blue, rev);
	mPalletID = 4 + rev;
    }

    //---------------------------------- Viridis color map
    else if (palette == "viridis") {
	if (mPalletID == 5 + rev) return;
	UInt_t Number = 16;
	Double_t Red[16] =
	    {0.2670, 0.2832, 0.2730, 0.2431, 0.1856, 0.1287, 0.1206, 0.1433,
	     0.1966, 0.2965, 0.4494, 0.6889, 0.8146, 0.8761, 0.9359, 0.9932};
	Double_t Green[16]  =
	    {0.0049, 0.1157, 0.2045, 0.2921, 0.4186, 0.5633, 0.6258, 0.6695,
	     0.7118, 0.7616, 0.8138, 0.8654, 0.8834, 0.8911, 0.8986, 0.9062};
	Double_t Blue[16]   =
	    {0.3294, 0.4361, 0.5017, 0.5385, 0.5568, 0.5512, 0.5335, 0.5112,
	     0.4792, 0.4242, 0.3354, 0.1827, 0.1103, 0.0953, 0.1081, 0.1439};
	Double_t Length[16] = 
	    {0.0000, 0.0784, 0.1490, 0.2235, 0.3451, 0.4980, 0.5647, 0.6118,
	     0.6588, 0.7176, 0.7882, 0.8823, 0.9294, 0.9529, 0.9764, 1.0000};
	set_palette(Number, Length, Red, Green, Blue, rev);
	mPalletID = 5 + rev;
    }

    //---------------------------------- Unrecognized palette
    else {
	throw runtime_error("GDSPlot:set_palette Unrecognized palette name");
    }
}

//======================================  Internal palette setter
void
GDSPlot::set_palette(size_t Number, const double* Length, const double* Red,
		     const double* Green, const double* Blue, bool reverse) {
    lcl_array<double> x(Number, Length);
    lcl_array<double> r(Number, Red);
    lcl_array<double> g(Number, Green);
    lcl_array<double> b(Number, Blue);
    if (reverse) {
	size_t half=Number / 2;
	for (size_t i=0; i<half; i++) {
	    double tmp = x[i];
	    x[i] = 1.0 - x[Number - 1 - i];
	    x[Number - 1 - i] = 1.0 - tmp;
	    std::swap<double>(r[i], r[Number - 1 - i]);
	    std::swap<double>(g[i], g[Number - 1 - i]);
	    std::swap<double>(b[i], b[Number - 1 - i]);
	}
	if ((Number%2) != 0) x[half+1] = 1.0 - x[half+1];
    }
    Int_t nb=255;
    TColor::CreateGradientColorTable(Number, x, r, g, b, nb);
}

//======================================  Set the Axis ranges
void 
GDSPlot::set_range(double xmin, double xmax, double ymin, double ymax) {
    mXmin = xmin;
    mXmax = xmax;
    mYmin = ymin;
    mYmax = ymax;
}

//======================================  Set the print size
void 
GDSPlot::set_size(double xsize, double ysize) const {
    mCanvas->Size(xsize, ysize);
}

//======================================  Set the smoothing options
void 
GDSPlot::set_smooth(int nSmooth, const std::string& smoothOpt) {
    mNSmooth   = nSmooth;
    mSmoothOpt = smoothOpt;
}

//======================================  Set the plot style
void 
GDSPlot::set_style(int id) {
    switch (id) {

    //----------------------------------  Standard plot style
    case 0:
	//------------------------------  Turn off plot statistics
	gStyle->SetOptStat(0);

	//------------------------------  Canvas
	gStyle->SetCanvasBorderMode(0); //  No borders, white background
	gStyle->SetCanvasColor(0);      //  white background

	//------------------------------  Pads:
	gStyle->SetPadBorderMode(0);    //  No borders, white background
	gStyle->SetPadColor(0);         //  White background
	gStyle->SetPadRightMargin(0.15);//  Leave space for z-axis title.

	//------------------------------  Plot Title
	gStyle->SetTitleX(0.5);         //  Centered title
	gStyle->SetTitleAlign(23);      //  
	gStyle->SetTitleBorderSize(0);  //  No borders/shadows
	gStyle->SetTitleFillColor(0);   //  White background.
	gStyle->SetTitleSize(0.08, "u");// Size = 0.08 for title

	//------------------------------  Label text
	gStyle->SetTitleSize(0.05, "xyz"); // Size = 0.05 for all labels
	break;
    default:
	cerr << "Unrecognized graphics style: " << id << endl;
	break;
    }
}

//======================================  Set the Axis ranges
void 
GDSPlot::set_zrange(double zmin, double zmax) {
    mZmin = zmin;
    mZmax = zmax;
}

//======================================  Plot a surface
void 
GDSPlot::surf(int nx, const double* x, int ny, const double* y, 
	      const double* z) {
    TH2D xyplot("xyplot", mTitle.c_str(), nx, x, ny, y);
    double zmin = mZmin + 0.001 * (mZmax - mZmin);
    int inx = 0;
    for (int j=0; j<ny; j++) {
	double y0 = 0.5 * (y[j] + y[j+1]);
	for (int i=0; i<nx; i++) {
	    double x0 = 0.5 * (x[i] + x[i+1]);
	    double z0 = z[inx++];
	    //-------------------------------------
	    // keep the data value on the color map
	    if (z0 < zmin) z0 = zmin;
	    //-------------------------------------
	    xyplot.Fill(x0, y0, z0);
	}
    }

    //---------------------------------- Optional smoothing
    if (mNSmooth) xyplot.Smooth(mNSmooth, mSmoothOpt.c_str());

    if (mLogX) mCanvas->SetLogx(1);
    else       mCanvas->SetLogx(0);
    if (mLogY) mCanvas->SetLogy(1);
    else       mCanvas->SetLogy(0);
    mCanvas->SetTicks(1, 1);

    set_palette();
    if (mZmax > mZmin) {
	xyplot.SetMinimum(mZmin);
	xyplot.SetMaximum(mZmax);
    }
    if (!mTitle.empty())  xyplot.SetTitle(mTitle.c_str());
    set_axes(xyplot);
    xyplot.SetContour(TColor::GetNumberOfColors(), 0);
    xyplot.DrawCopy("colz");
    mTrace++;
}

//======================================  Set canvas title
void 
GDSPlot::title(const std::string& titl) {
    mTitle = titl;
}

//======================================  Set canvas title
void 
GDSPlot::welch(const TSeries& ts, double fMin, double fMax) {
    Hanning hw;
    WelchPSD w(Interval(1.0), 0.50, &hw);
    w.add(ts);
    if (fMin == 0.0 && fMax == 0.0) {
	plot(w.get_psd());
    } else {
	if (fMax == 0.0) fMax = 2.0/ts.getTStep();
	plot(w.get_psd().extract_psd(fMin, fMax-fMin));
    }
}

//======================================  Set the X-axis label
void 
GDSPlot::xlabel(const std::string& titl) {
    mXlabel = titl;
}

//======================================  Set X-axis log scale
void 
GDSPlot::xlog(bool logx) {
    mLogX = logx;
}

//======================================  Time unit ranges
static const double millisecondThreshold = 0.5;
static const double secondThreshold = 3 * 60;
static const double minuteThreshold = 3 * 60 * 60;
static const double hourThreshold = 3 * 24 * 60 * 60;
static const double dayThreshold = 365.25 * 24 * 60 * 60;

//======================================  Set x time scale in reasonable units
double 
GDSPlot::xTimeScale(double dT, const std::string& xttl) {
    double tScale = 1.0;
    string tUnits;
    if (dT < millisecondThreshold) {
      tScale = 0.001;
      tUnits = " [milliseconds]";
    }
    else if (dT < secondThreshold) {
      tScale = 1.0;
      tUnits = " [seconds]";
    }
    else if (dT < minuteThreshold) {
      tScale = 60;
      tUnits = " [minutes]";
    }
    else if (dT < hourThreshold) {
      tScale = 3600;
      tUnits = " [hours]";
    }
    else if (dT < dayThreshold) {
      tScale = 86400;
      tUnits = " [days]";
    }
    else {
      tScale = 31557600;
      tUnits = "years";
    }
    xlabel(xttl + tUnits);
    return tScale;
}


//======================================  Set the Y-axis label
void 
GDSPlot::ylabel(const std::string& titl) {
    mYlabel = titl;
}

//======================================  Set Y-axis log scale
void 
GDSPlot::ylog(bool logy) {
    mLogY = logy;
}

//======================================  Set the Z-axis label
void 
GDSPlot::zlabel(const std::string& titl) {
    mZlabel = titl;
}
