/***************************************************************************
    File        : RandomMersenne.cpp
    Description : Implements the andom number generator of type
                  Mersenne twister
 ---------------------------------------------------------------------------
    Begin       : Fri Jan 2 2004
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#include "RandomMersenne.h"

#define MERS_M   397
#define MERS_R   31
#define MERS_U   11
#define MERS_S   7
#define MERS_T   15
#define MERS_L   18
#define MERS_A   0x9908B0DF
#define MERS_B   0x9D2C5680
#define MERS_C   0xEFC60000


void
gwd::RandomMersenne::DetectComputerArchitecture()
{
  // detect computer architecture
  union {double f; unsigned long int i[2];} convert;
  convert.f = 1.0;
  // Note: Old versions of the Gnu g++ compiler may make an error here,
  // compile with the option  -fenum-int-equiv  to fix the problem
  if (convert.i[1] == 0x3FF00000) mArchitecture = LITTLEENDIAN;
  else if (convert.i[0] == 0x3FF00000) mArchitecture = BIGENDIAN;
  else mArchitecture = NONIEEE;
}

void
gwd::RandomMersenne::RandomInit(unsigned long int seed)
{
  // re-seed generator
  mMT[0]= seed;
  for (mMTIdx=1; mMTIdx < MERS_N; mMTIdx++)
  {
    mMT[mMTIdx] = (1812433253UL * (mMT[mMTIdx-1] ^ (mMT[mMTIdx-1] >> 30)) + mMTIdx);
  }
}


void
gwd::RandomMersenne::RandomInitByArray(std::vector<unsigned long int> seeds)
{
  int length = static_cast<int>(seeds.size());
  // seed by more than 32 bits
  int i, j, k;
  RandomInit(19650218UL);
  if (length <= 0) return;
  i = 1;  j = 0;
  k = (MERS_N > length ? MERS_N : length);
  for (; k; k--)
  {
    mMT[i] = (mMT[i] ^ ((mMT[i-1] ^ (mMT[i-1] >> 30)) * 1664525UL)) + seeds[j] + j;
    i++; j++;
    if (i >= MERS_N) {mMT[0] = mMT[MERS_N-1]; i=1;}
    if (j >= length) j=0;
  }
  for (k = MERS_N-1; k; k--)
  {
    mMT[i] = (mMT[i] ^ ((mMT[i-1] ^ (mMT[i-1] >> 30)) * 1566083941UL)) - i;
    if (++i >= MERS_N)
    {
      mMT[0] = mMT[MERS_N-1];
      i=1;
    }
  }
  mMT[0] = 0x80000000UL;
} // MSB is 1; assuring non-zero initial array


unsigned long int
gwd::RandomMersenne::BRandom()
{
  // generate 32 random bits
  unsigned long int y;

  if (mMTIdx >= MERS_N)
  {
    // generate MERS_N words at one time
    const unsigned long int LOWER_MASK = (1LU << MERS_R) - 1; // lower MERS_R bits
    const unsigned long int UPPER_MASK = -1L  << MERS_R;      // upper (32 - MERS_R) bits
    static const unsigned long int mag01[2] = {0, MERS_A};

    int kk;
    for (kk=0; kk < MERS_N-MERS_M; kk++)
    {
      y = (mMT[kk] & UPPER_MASK) | (mMT[kk+1] & LOWER_MASK);
      mMT[kk] = mMT[kk+MERS_M] ^ (y >> 1) ^ mag01[y & 1];
    }

    for (; kk < MERS_N-1; kk++)
    {
      y = (mMT[kk] & UPPER_MASK) | (mMT[kk+1] & LOWER_MASK);
      mMT[kk] = mMT[kk+(MERS_M-MERS_N)] ^ (y >> 1) ^ mag01[y & 1];
    }

    y = (mMT[MERS_N-1] & UPPER_MASK) | (mMT[0] & LOWER_MASK);
    mMT[MERS_N-1] = mMT[MERS_M-1] ^ (y >> 1) ^ mag01[y & 1];
    mMTIdx = 0;
  }

  y = mMT[mMTIdx++];

  // Tempering (May be omitted):
  y ^=  y >> MERS_U;
  y ^= (y << MERS_S) & MERS_B;
  y ^= (y << MERS_T) & MERS_C;
  y ^=  y >> MERS_L;
  return y;
}


double
gwd::RandomMersenne::Random()
{
  // output random float number in the interval 0 <= x < 1
  union {double f; unsigned long int i[2];} convert;
  unsigned long int r = BRandom(); // get 32 random bits
  // The fastest way to convert random bits to floating point is as follows:
  // Set the binary exponent of a floating point number to 1+bias and set
  // the mantissa to random bits. This will give a random number in the
  // interval [1,2). Then subtract 1.0 to get a random number in the interval
  // [0,1). This procedure requires that we know how floating point numbers
  // are stored. The storing method is tested in function RandomInit and saved
  // in the variable Architecture. The following switch statement can be
  // omitted if the architecture is known. (A PC running Windows or Linux uses
  // LITTLEENDIAN architecture):
  switch (mArchitecture)
  {
    case LITTLEENDIAN:
      convert.i[0] =  r << 20;
      convert.i[1] = (r >> 12) | 0x3FF00000;
      return convert.f - 1.0;
    case BIGENDIAN:
      convert.i[1] =  r << 20;
      convert.i[0] = (r >> 12) | 0x3FF00000;
      return convert.f - 1.0;
    case NONIEEE: default:
      ;
  }
  // This somewhat slower method works for all architectures, including
  // non-IEEE floating point representation:
  return (double)r * (1./((double)(unsigned long int)(-1L)+1.));
}


int
gwd::RandomMersenne::IRandom(int min, int max)
{
  // output random integer in the interval min <= x <= max
  int r;
  r = int((max - min + 1) * Random()) + min; // multiply interval with random and truncate
  if (r > max) r = max;
  if (max < min) return 0x80000000;
  return r;
}



