/***************************************************************************
    File        : Parameter.h
    Description : Contains all the control parameters for the Noise Floor
                  Monitor
 ---------------------------------------------------------------------------
    Begin       : Tue Mar 22 2005
    Author(s)   : Roberto Grosso
 ***************************************************************************/
 
#ifndef __PARAMETER_H
#define __PARAMETER_H

// Libs
#include <iostream>
#include <fstream>
#include <sstream>
#include <algorithm>
#include <string>

// Project files
#include "Types.h"

namespace gwd {
  
  class Parameter {
  public:
    //! Constructor.
    Parameter() : mOrderNotchFilter(0),mUpdateWhiteFlt(0),mOrderWhiteFilter(0),mNoOfFreqPsd(0),
                  mSizePsd(0),mRMSizeForWhiteningFlt(0),mWindowSizeRunningMedian(0),mFltOrder(0),
                  mUpperCutoffFreq(0),mBandFltOrder(0),mTimeStride(0) { }
                  
    //! Desctructor.
    ~Parameter() { }
    
    //! Read parameters from file.
    void ReadParameters(const std::string& filename);
    
    //! Get order of Notch filter.
    unsigned int OrderNotchFilter() { return mOrderNotchFilter; }
    
    //! Get no. of time strides to update whitening filter.
    unsigned int UpdateWhiteFilter() { return mUpdateWhiteFlt; }
    
    //! Get order of whitening filter.
    unsigned int OrderWhiteFilter() { return mOrderWhiteFilter; }
    
    //! Get number of frequencies in the Power Spectrum Density.
    unsigned int NoOfFreqPSD() { return mNoOfFreqPsd; }
    
    //! Get the size is sec. of the time series used for PSD estimation.
    double SizeForPSD() { return mSizePsd; }
    
    //! Get size in Hz. of running median whitening filter.
    double RunningMedianSizeForWhiteningFilter() { return mRMSizeForWhiteningFlt; }
    
    //! Get window size in sec for running median filter.
    double WindowSizeRunningMedian() { return mWindowSizeRunningMedian; }
    
    //! Get filter order for low-pass and band-pass filters.
    unsigned int FltOrder() { return mFltOrder; }
    
    //! Get the frequency for band limited data filtering.
    double UpperCutoffFreq()  { return mUpperCutoffFreq; }

    //! Get filter order for signal bands processing.
    unsigned int BandFltOrder() { return mBandFltOrder; }

    //! Get the number of frequency bands to be computed.
    int NoFreqBands() { return  mNoFreqBands; }
    //! Band frequencies.
    double FreqBand(int ii) { return mFreqBand[ii]; }
    
    //! Get the time stride.
    double TimeStride() { return mTimeStride; }

    //! Get name of detector.
    const char* DetectorSite() { return mDetectorSite.c_str(); }
    //! Get no. of main channels.
    const int NoOfMainChannels() { return mMainChannel.size(); }
    //! Get name of main channel.
    const char* MainChannel(const int chn) { return mMainChannel[chn].c_str(); }
    //! Get no. of channels.
    unsigned int NoChannels()  { return mNoChannels; }
    //! Get detector channel name.
    const char* ChannelName(int ch) { return mChannel[ch].c_str(); }
    //! Get channel sample rate.
    const double ChannelSampleRate(int ch) { return mChannelSampleRate[ch]; }
    
    //! Get the over sampling factor for signal decimation.
    const double OverSamplingFactor() { return mOverSamplingFactor; }
  private:
    //! Order of the Notch Filter.
    unsigned int mOrderNotchFilter;
    
    //! Update of Whitening Filter.
    unsigned int mUpdateWhiteFlt;
    
    //! Order of the Whitening Filter.
    unsigned int mOrderWhiteFilter;
    
    //! Number of frequencies in the Power Spectrum Density.
    unsigned int mNoOfFreqPsd;
    
    //! Size in sec. of ts. to compute the pds.
    double mSizePsd;
    
    //! Size is Hz. of the running median filter to compute Whitening filter.
    double mRMSizeForWhiteningFlt;
    
    //! Size in sec. of the Running median filter.
    double mWindowSizeRunningMedian;
    
    //! Filter order for low-pass and band-pass filters.
    unsigned int mFltOrder;
    
    //! Upper cutoff freuquency
    double mUpperCutoffFreq;

    //! Filter order for signal bands processing.
    unsigned int mBandFltOrder;

    //! No. of frequency bands.
    int mNoFreqBands;
    //! Band frequencies.
    Vector mFreqBand;
    
    //! Time Stride.
    double mTimeStride;

    //! Name of detector, HA for Hanford and LA forLivingstone.
    std::string mDetectorSite;
    //! Name of detector's main channel.
    std::vector<std::string> mMainChannel;
    //! Save nr. of channels.
    unsigned int mNoChannels;
    //! Name of channels to be processed.
    std::vector<std::string> mChannel;
    //! Channel sampling rate
    Vector mChannelSampleRate;
    
    //! Oversampling factor for sampling rate decimation
    double mOverSamplingFactor;
  };
    
} // namespace gwd
#endif // __PARAMETER_H

