/***************************************************************************
    File        : Convolution.h
    Description : Fast Convolution and Deconvolution based on the FFT
                  Implements the algorithms given in the Numericl Receipies
 ---------------------------------------------------------------------------
    Begin       : Fri Jul 27 2001
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __CONVOLUTION_H
#define __CONVOLUTION_H


/*! \file Convolution.h
 *  Fast (cyclic) convolution and deconvolution of two discrete signals
 *  based on the FFT.
 *  A direct covolution is also implemented. This is non-cyclic and
 *  should be used only for small problems.
 *  \brief Fast (cyclic) convolution based on the FFT.
 */

// Libs
#include <iostream>
#include <vector>
#include <cmath>

// Project files
#include "Singleton.h"
#include "Types.h"
#include "Fourier.h"


namespace gwd {

  //! Convolution of two discrete signals.
  class Convolution {
  public:
    // Constructors
    /** Default constructor */
    Convolution () {}

    /**
     * Constructor.
     * It initializes the fftw for better performance in case of repeated
     * computations with signals of the same size.
     */
    Convolution(unsigned int sz,Vector& s,Vector& r)
    {
      Init(sz,s,r);
    }

    /** Destructor */
    virtual ~Convolution() {}

    /**
     * Computes the cyclic convolution using the FFT. The signal and the impulse
     * response are transformed into frequency space and multiplied. Rearrangement
     * of the arrays and zero padding have to be done by the calling function.
     * The class was initialized,
     * i.e. fft of the impulse response filter was already computed and
     * the fftw plan was initialized, see FFTW 3.0.1.
     * @return true in case of success otherwise false.
     */
    inline bool ComputeConvolution();

    /**
     * Computes the cyclic convolution of a signal with an impulse response filter.
     * The signal and the impulse response are transformed into frequency space
     * and multiplied. Rearrangement of the arrays and zero padding have to be
     * done by the calling function.
     * The system was not initialized, i.e. the fourier transform of the
     * signal and the impulse response have to be computed, and the fftw plan
     * was not initialized. The convolution is done in-place, i.e. signal is
     * overwritten.
     * @param signal  the signal to be convolved.
     * @param response the impulse response filter.
     * @return true in case of success otherwise false.
     */
    inline bool ComputeConvolution(Vector& signal,Vector& response);

    /**
     * Computes the cyclic convolution of a signal with an impulse response filter.
     * The signal and the impulse response are transformed into frequency space
     * and multiplied. Rearrangement of the arrays and zero padding have to be
     * done by the calling function.
     * The system was not initialized, i.e. the fourier transform of the
     * signal and the impulse response have to be computed, and the fftw plan
     * was not initialized. The convolution is not done in-place, i.e. the result
     * is written in an extra array.
     * @param signal  the signal to be convolved.
     * @param response the impulse response filter.
     * @param answ the answer containing the result of the convolution.
     * @return true in case of success otherwise false.
     */
    inline bool ComputeConvolution(Vector& signal,Vector& resp,Vector& answ);

    /**
     * Computes the convolution in time domain, thus zero padding is
     * not necessary. The convolutionn is done in-place, i.e. signal
     * is overwritten.
     * The size of the answer is signal size + response size - 1.
     *
     * @param signal  the signal to be convolved.
     * @param response the impulse response filter.
     * @return true in case of success otherwise false.
     */
    inline bool Direct(Vector& signal,Vector& response, Vector& answer);

    /**
     * This method initialize the FFT for better performance
     * if the convolution has to be repeated many times. The
     * impuls response filter is fft transformed only once and
     * FFTW 3 plan is initialized. The subsequent convolutions
     * assume same signal sizes.
     *
     * @param size  size of the fft.
     * @param s the time domain signal.
     * @param r the inpulse response filter.
     */
    inline void Init(unsigned int size,Vector& s,Vector& r);

  private:
    unsigned int mSize;
    std::vector<Complex> mSS;
    std::vector<Complex> mFF;
    Fourier mFFTS;
    Fourier mFFTF;
    Fourier mFFTC;
  };




  // Intialize the fftw for better performance.
  inline void
  Convolution::Init(unsigned int size,Vector& s,Vector& r)
  {
    // set problem size
    mSize = size;
    // Process signal
    if (s.size() != mSize)
      s.resize(mSize);
    // Initialize the FFT
    mFFTS.Init(mSize,s,mSS);
    mFFTC.Init(mSize,mSS,s);

    // Process filter
    if (r.size() != mSize)
    {
      Vector filter(mSize);
      std::copy(r.begin(),r.end(),filter.begin());
      std::fill(filter.begin()+r.size(),filter.end(),double(0));
      // The filter is computed only once
      mFFTF.dft(filter,mFF);
    }
    else
    {
      mFFTF.dft(r,mFF);
    }
  }

  // Computes the cyclic convolution of two signals using the FFT.
  // The systme has to be initialized, signal and response are known.
  inline bool
  Convolution::ComputeConvolution()
  {
    // Compute the FFT of the signal
    mFFTS.dft();

    // Compute the product in frequency domain
    typedef std::vector<Complex>::size_type SizeType;
    const SizeType sigSize = mFF.size();
    for (SizeType nn = 0; nn < sigSize; nn++) mSS[nn] *= mFF[nn];

    // Normalize
    const double factor = static_cast<double>(mSize);
    for (SizeType nn = 0; nn < sigSize; nn++) mSS[nn] /= factor;

    // Transform to time domain
    mFFTC.dft();

    return true;
  }


  // Computes the convolution of two signals (signal and response).
  // The system was not initialized. The computation is done in-place,
  // i.e. the vector signal is overwritten.
  inline bool
  Convolution::ComputeConvolution(Vector& signal,Vector& response)
  {
    if (signal.empty() || response.empty())
    {
      Singleton* single = Singleton::exemplar();
      single->AppendMessage("gwd::Convolution::ComputeConvolution(): empty signal or impulse response");
      return false;
    }
    
    typedef Vector::size_type SizeType;
    SizeType anSize = signal.size() + response.size() - 1;
    SizeType NN = anSize / 2 + 1;

    // Answer is larger than signal, then zero-padd signal
    for (SizeType nn = signal.size(); nn < anSize; nn++)
      signal.push_back(double(0));

    Vector h(anSize);
    std::copy(response.begin(),response.end(),h.begin());
    std::fill(h.begin()+response.size(),h.end(),double(0));

    // If signal size equals response size, just do convolution
    std::vector<Complex> cSignal(NN);
    std::vector<Complex> cFilter(NN);
    // Fourier Transform
    Fourier fourier;
    fourier.dft(signal,cSignal);
    fourier.dft(h,cFilter);

    // Computes convolution
    for (SizeType nn = 0; nn < cSignal.size(); nn++) cSignal[nn] *= cFilter[nn];
    // Normalize
    const double factor = static_cast<double>(anSize);
    for (SizeType nn = 0; nn < cSignal.size(); nn++) cSignal[nn] /= factor;
    // Transform to time domain
    fourier.dft(cSignal,signal);

    return true;
  }


  // Computes the convolution of two signals (signal and response). The
  // result is written in the vector answer.
  // The system was not initialized.
  inline bool
  Convolution::ComputeConvolution(Vector& signal,Vector& response,Vector& answer)
  {
    if (signal.empty() || response.empty())
    {
      Singleton* single = Singleton::exemplar();
      single->AppendMessage("gwd::Convolution::ComputeConvolution(): empty signal or impulse response");
      return false;
    }
    // copy signal into answer array
    answer = signal;
    // Compute convolution, answer will have right size
    if (ComputeConvolution(answer,response))
      return true;
    else
      return false;

  }


  // Computes the convlution in time domain. No zero padding is needed.
  // The size of the output is signal size + response size - 1, this is
  // not a cyclic convolution.
  inline bool
  Convolution::Direct(Vector& signal,Vector& response, Vector& answer)
  {
    if (signal.empty() || response.empty())
    {
      Singleton* single = Singleton::exemplar();
      single->AppendMessage("gwd::Convolution::Direct(): empty signal or impulse response");
      return false;
    }
    
    typedef Vector::size_type SizeType;
    const SizeType sSize = signal.size();
    const SizeType rSize = response.size();
    const SizeType aSize = sSize + rSize - 1;
    if (sSize < rSize)
    {
      Singleton* single = Singleton::exemplar();
      single->AppendMessage("gwd::Convolution::Direct(): signal shorter than impulse response filter!");
      return false;
    }
    if (answer.size() != aSize)
      answer.resize(aSize);

    // First segment overlap with segment
    const SizeType rLim = rSize - 1;
    for (SizeType kk = 0; kk < rLim; kk++)
    {
      answer[kk] = 0.;
      for (SizeType nn = 0; nn <= kk; nn++)
      {
        answer[kk] += response[kk - nn]*signal[nn];
      }
    }

    for (SizeType jj = rLim; jj < sSize; jj++)
    {
      answer[jj] = 0.;
      for (SizeType kk = 0; kk < rSize; kk++)
      {

        answer[jj] += response[kk]*signal[jj - kk];
      }
    }

    // The sSize + rSize - 1 part
    for (SizeType jj = sSize; jj < aSize; jj++)
    {
      answer[jj] = 0.;
      for (SizeType kk = jj - sSize + 1; kk < rSize; kk++)
      {
        answer[jj] += signal[jj - kk]*response[kk];
      }
    }

    // Return value
    return true;
  }
} // namespce gwd

#endif // __CONVOLUTION_H
