/*  File LineMonitor.cc      
    Sergey Klimenko, University of Florida
*/

#ifndef _USE_DMT
#define _USE_DMT
#endif

//#ifndef __CINT__

//  The next three lines are needed if to generate triggers.
//  The descriptive title in PIDTITLE should the monitor function.
// #define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/LineMonitor/LineMonitor.cc 7467 2015-09-15 09:49:52Z john.zweizig@LIGO.ORG $"
// #define PIDTITLE  "Data monitor environment template"
// #include "ProcIdent.hh"

#include <fstream>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <iostream>
#include <sstream>
#include "TSeries.hh"
#include "Dacc.hh"
#include "Interval.hh"
#include "Time.hh"
#include "DVector.hh"

#include "LineFilter.hh"
#include "LineMonitor.hh"
#include "LineTable.hh"
#include "Biorthogonal.hh"

#include "xsil/Xwriter.hh"
#include "xsil/xsil.hh"

using namespace std;

EXECDAT(LineMonitor)        // Generate the main routine.

  //#endif               // !def(__CINT__)


//--- LineMonitor constructor -------------

LineMonitor::LineMonitor(int argc, const char *argv[]) 
   : DatEnv(argc, argv), maxDump(0), frequency(60.), timeStride(1.), dumpLM(false),
     fid(1), nT(1), nDump(1), mDump(0), dmtDump(0), nFirst(1), nLast(1), 
     nSkip(0), masterRate(0.), nBand(5), fScan(1.), SNR(2.), fBand(0.45), nScan(10), 
     nBuffer(1024), mOSC(getDacc()), appendMode(0), al_ca("LineMonitor"), al_ha(),
     al_cb("LineMonitor"), al_hb(), debug(1), ifolock(0)
{ 
   FILE* out;
   LineFilter *lf;
   Biorthogonal<double> B(10);
   WSeries<double>* pw;
  
   char *fname;
   char *ftag;
   char *w;
   char *pc;

   char inFile[256];
   char offFile[256];
   char serverName[256];
   char htmlDir[256];
   char trendDir[256];
   char outputDir[256];
   char nameTag[256];
   char gps[3];
   char c[256];
   
   double al_snr=0.0, al_sns=0.0; //jason 
   bool   al_a         = false;   //jason
   int n,i,fig=0;
   alarmEntry *al_e;

   bool   configFile   = false;
   bool   setfilter    = true;
   bool   offline      = false;
   double maxStride    = 0.;
   
   plow[0]=false;
   plow[1]=false;
   al_set[0]=false;
   al_set[1]=false;

//   sprintf(channelName,"%s","");

   sprintf(a_namea,    "%s","");
   sprintf(lockStatus, "%s","");
   sprintf(filePath,   "%s","");
   sprintf(htmlFile,   "%s","");
   sprintf(htmlDir,    "%s","");
   sprintf(trendDir,   "%s","");
   sprintf(trendFile,  "%s","");
   sprintf(indexFile,  "%s","");
   sprintf(indexHTML,  "%s","");
   sprintf(suffix,     "%s","");
   sprintf(nameTag,    "%s","");
   sprintf(gps,        "%s","%s");

   Filter.resize(0);             // LineFilter vector   
   fileName.resize(0);           // LineMonitor dump file name
   freqTag.resize(0);            // LineFilter frequency tag
   channelName.resize(0);        // array of data channels
   channelIndex.resize(0);       // channel index in the channelName array
   refFilter.resize(0);          // reference Filter index
   channelRate.resize(0);        // decimated data sampling rate
   ts.resize(0);                 // TSeries data pointers
   ws.resize(0);                 // WSeries data pointers

/******************************************************************
 * The LM parameters can be set either from command line or from  *
 * configuration file (option -i fileName)                        *
 ******************************************************************/

// command line input of LineMonitor parameters

   if(argc < 2) DumpHelp();
   else {
      for(int ac = 1; ac < argc; ++ac) {

	 if( strcmp(argv[ac],"-i") == 0 ){
            sprintf(inFile,"%s",argv[++ac]);
	    configFile = true;
	 }

	 else if( strcmp(argv[ac],"-c") == 0 ){
	    pc = new char[100];
            sprintf(pc,"%s",argv[++ac]);
	    sprintf( c,"%s",pc);
	    channelName.push_back(pc);
	 }

	 else if( strcmp(argv[ac],"-H") == 0 ){
            sprintf(htmlFile,"%s",argv[++ac]);
	 }

	 else if( strcmp(argv[ac],"-l") == 0 ){
            sprintf(lockStatus,"%s",argv[++ac]);
	    cout<<"lock condition: "<<lockStatus<<endl;
	 }

	 else if( strcmp(argv[ac],"-infile") == 0 || strcmp(argv[ac],"-inlist") == 0 ){
            sprintf(offFile,"%s",argv[++ac]);
	    offline = true;
	 }

	 else if( strcmp(argv[ac],"-suffix") == 0 ){
	    sprintf(suffix,"%s",argv[++ac]);
	 }

	 else if ( strcmp(argv[ac],"-f") == 0 )
            sscanf(argv[++ac],"%lf",&frequency);

	 else if ( strcmp(argv[ac],"-t") == 0 )
            sscanf(argv[++ac],"%lf",&timeStride);

	 else if ( strcmp(argv[ac],"-I") == 0 )
	    sscanf(argv[++ac],"%d",&fid);
	 
	 else if ( strcmp(argv[ac],"-debug") == 0 )
	    sscanf(argv[++ac],"%d",&debug);

	 else if ( strcmp(argv[ac],"-n") == 0 )
            sscanf(argv[++ac],"%d",&nT);

         else if ( strcmp(argv[ac],"-d") == 0 )
            sscanf(argv[++ac],"%d",&nDump);

         else if ( strcmp(argv[ac],"-F") == 0 )
            sscanf(argv[++ac],"%d",&nFirst);

         else if ( strcmp(argv[ac],"-L") == 0 )
            sscanf(argv[++ac],"%d",&nLast);

         else if ( strcmp(argv[ac],"-S") == 0 )
            sscanf(argv[++ac],"%d",&nSkip);

         else if ( strcmp(argv[ac],"-R") == 0 )
            sscanf(argv[++ac],"%lf",&masterRate);

         else if ( strcmp(argv[ac],"-N") == 0 )
            sscanf(argv[++ac],"%d",&nBand);

         else if ( strcmp(argv[ac],"-s") == 0 )
	    fScan = 0;

	 else if(strcmp(argv[ac],"-lm") == 0) 
	    dumpLM = true;

         else if ( strcmp(argv[ac],"-K") == 0 )
            sscanf(argv[++ac],"%lf",&SNR);

         else if ( strcmp(argv[ac],"-B") == 0 )
            sscanf(argv[++ac],"%lf",&fBand);

         else if ( strcmp(argv[ac],"-m") == 0 )
            sscanf(argv[++ac],"%d",&nScan);

         else if ( strcmp(argv[ac],"-b") == 0 )
            sscanf(argv[++ac],"%d",&nBuffer);
	 
/*
	 else if ( strcmp(argv[ac],"-a") == 0)
	   {
	       if(!configFile)
		   for(i=1;i< argc; ++i)
		       if(strcmp(argv[i],"-i") == 0 )
			   configFile=true;
	       al_a=true;
	       sscanf(argv[++ac],"%lf",&al_snr);
	       if(!configFile)
	       {
		   al_e     = new alarmEntry;
		   al_e->snr=float(al_snr);
		   al_e->fid=0;
		   al_e->b  =false;
		   al_e->c  =true;
		   al_E.push_back(al_e);
		   if (al_e->snr >= 0) plow[0]=true;
		   if (al_e->snr < 0)  plow[1]=true;
	       }
	   }
*/
	 else cout<<"unknown argument: "<<argv[ac]<<endl;
      }
   }
   
// get OUTPUT directories
   if((w = getenv("DMTHTMLOUT")))
     if(strlen(w)){
       if(w[strlen(w)-1] == '/')
	 sprintf(htmlDir,"%s",w);
       else
	 sprintf(htmlDir,"%s/",w);
     }
   if((w = getenv("DMTOUTPUT")))
     if(strlen(w)){
       if(w[strlen(w)-1] == '/')
	 sprintf(outputDir,"%s",w);
       else
	 sprintf(outputDir,"%s/",w);
     }
   if((w = getenv("DMTRENDOUT")))
     if(strlen(w)){
       if(w[strlen(w)-1] == '/')
	 sprintf(trendDir,"%s",w);
       else
	 sprintf(trendDir,"%s/",w);
     }

// get GPS time
   if(!offline) sprintf(gpsTime,"%d",int(Now().getS()));

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

   if(!configFile) // set LM from the command line
     {      
       if(!channelName.size())  DumpHelp(); // no channel was specified
       if(channelName.size()>1) DumpHelp(); // no more then 1 channel
       
// set filter
       lf = new LineFilter(fabs(frequency),timeStride,fid,nT); // initialize LF
       lf->setFilter(nFirst,nLast,nSkip,0,nBand);              // set LF parameters    
       lf->Stride = timeStride;
       double x_fScan = (fScan!=0.) ? fabs(frequency) : fScan;
       lf->setFScan(x_fScan,SNR,fBand,nScan);            // set FScan      
       Filter.push_back(lf);         // add filter to the list
       channelIndex.push_back(0);    // add channel index to the list
       refFilter.push_back(0);       // add reference index to the list
       channelRate.push_back(masterRate);
       
// set frequency tag
       ftag = new char[100];
       double x_frequency = -fabs(frequency);
       n = int(x_frequency);
       sprintf(ftag,"%d.%02d",n,int(100.*(n - x_frequency)));
       freqTag.push_back(ftag);

// set file name for LineMonitor trends
       fname = new char[100];
       sprintf(fname,"%s%s%s-%s.lm",outputDir,channelName[0],ftag,gps);
       while((w = strchr(fname,':'))) w[0] = '-';
       
       if(dumpLM) cout <<"output LineFilter file name: "<< fname << endl;
       fileName.push_back(fname);   // add file name to the list

       maxStride = timeStride/nT;
     }

   else
     {                   // set LM from configuration file
       cout<<"configuration file: "<<inFile<<endl;
       if( (out=fopen(inFile,"r")) == NULL ){ DumpHelp(); exit(0); } 
       
       double x_frequency  = frequency;
       double x_timeStride = timeStride;
       int    x_fid        = fid;
       int    x_nT         = nT;
       int    x_nFirst     = nFirst;
       int    x_nLast      = nLast;
       int    x_nSkip      = nSkip;
       int    x_nBand      = nBand;
       int    x_nR         = 8;
       double x_rate       = masterRate;
       double x_fScan      = fScan;
       double x_SNR        = SNR;
       double x_fBand      = fBand;
       int    x_nScan      = nScan;
       int    x_refFilter  = -1;
       
// read LineMonitor parameters from configuration file
       
       char s[100];
       
       while(fgets(s,100,out) != NULL)
	 {
	   pString = s;
	   cout << endl << s;
	   if(!strlen(s)) break;
	   if(strcmp(w=getToken(pString),"-f") == 0)
	     {
	       x_frequency  = frequency;
	       x_timeStride = timeStride;
	       x_fid        = fid;
	       x_nT         = nT;
	       x_nFirst     = nFirst;
	       x_nLast      = nLast;
	       x_nSkip      = nSkip;
	       x_rate       = 0.;
	       x_nBand      = nBand;
	       x_nR         = 8;
	       x_fScan      = fScan;
	       x_SNR        = SNR;
	       x_fBand      = fBand;
	       x_nScan      = nScan;      
	       x_refFilter  = -1;
	       setfilter    = true;
	       
	       while(strlen(w)>1)
		 {
		   getParameter(w,"-f",x_frequency);  
		   getParameter(w,"-t",x_timeStride);  
		   getParameter(w,"-I",x_fid);         
		   getParameter(w,"-n",x_nT);         
		   getParameter(w,"-F",x_nFirst);      
		   getParameter(w,"-L",x_nLast);      
		   getParameter(w,"-S",x_nSkip);       
		   getParameter(w,"-R",x_rate);       
		   getParameter(w,"-N",x_nBand);       
		   getParameter(w,"-K",x_SNR);       
		   getParameter(w,"-B",x_fBand);      
		   getParameter(w,"-m",x_nScan); 
		   getParameter(w,"-r",x_refFilter);
		   getParameter(w,"-lrf",x_nR);        
 
		   if(strcmp(w,"-a") == 0)  // to set an alarm
		   {
		       getParameter(w,"-a",al_sns);
		       al_e     = new alarmEntry;
		       al_e->snr=float(al_sns);
		       al_e->fid=fig;
		       al_e->b  =false;
		       al_e->c  =true;
		       al_E.push_back(al_e);
		       if(al_e->snr >= 0 ) plow[0]=true;
		       if(al_e->snr < 0 )  plow[1]=true;
		   }

		   if(strcmp(w,"-s") == 0) 
		     x_fScan = 0;

		   else if(strcmp(w,"-tag") == 0)
		     sprintf(nameTag,"%s",w=getToken(pString,true));

		   w = getToken(pString);

		 }
	     }
	   else
	     {
	       setfilter = false;
	       while(strlen(w)>1)
		 {
		   getParameter(w,"-t",timeStride); 
		   getParameter(w,"-I",fid);       
		   getParameter(w,"-n",nT);        
		   getParameter(w,"-F",nFirst);     
		   getParameter(w,"-L",nLast);         
		   getParameter(w,"-S",nSkip);         
		   getParameter(w,"-R",masterRate);          
		   getParameter(w,"-N",nBand);         
		   getParameter(w,"-K",SNR);           
		   getParameter(w,"-B",fBand);         
		   getParameter(w,"-m",nScan);         
		   getParameter(w,"-b",nBuffer);       
		   getParameter(w,"-d",nDump);
		   getParameter(w,"-debug",debug);
		   if(strcmp(w,"-a") == 0)
		     {
		       getParameter(w,"-a",al_snr);
		       al_a=true;
		     }
		   if(strcmp(w,"-s") == 0) fScan = 0;
		   if(strcmp(w,"-lm") == 0) dumpLM = true;
		   if(strcmp(w,"-suffix") == 0)
		     sprintf(suffix,"%s",w=getToken(pString, true));
		   
		   else if(strcmp(w,"-c") == 0){
		     pc = new char[100];
		     sprintf(pc,"%s",w=getToken(pString, true));
		     channelName.push_back(pc);
		   }

		   else if(strcmp(w,"-H") == 0)
		     sprintf(htmlFile,"%s",w=getToken(pString, true));
		   
		   else if(strcmp(w,"-l") == 0)
		     sprintf(lockStatus,"%s",w=getToken(pString, true));
		   
		   else if(strcmp(w,"-o") == 0)
		     {
		       sprintf(filePath,"%s/",w=getToken(pString, true));
		       n = strlen(filePath);
		       if(n>0)
			 if(filePath[n-1] == '/')
			   filePath[n] = '\0';
		     }
		   w = getToken(pString);
		 }
	     }
	   
	   if(setfilter)
	     {
	       if(!channelName.size()) DumpHelp();
	       sprintf( c,"%s",channelName[channelName.size()-1]);
	       
	       cout << "Line Monitor parameters for channel   " << c            << endl
		    << "  seed frequency (Hz) [60.]           " << x_frequency  << endl
		    << "  time stride (s) [1.]                " << x_timeStride << endl
		    << "  filter ID                           " << x_fid        << endl
		    << "  number of stride subdivisions       " << x_nT         << endl
		    << "  frequency scan flag                 " << x_fScan      << endl
		    << "  first harmonic                      " << x_nFirst     << endl
		    << "  last harmonic                       " << x_nLast      << endl
		    << "  skip harmonics                      " << x_nSkip      << endl
		    << "  limit on signal to noise ratio      " << x_SNR        << endl
		    << "  # of fft bins to estimate noise PSD " << x_nBand      << endl
		    << "  scan bandwidth in units of fft bins " << x_fBand      << endl
		    << "  decimated sampling rate             " << x_rate       << endl 
		    << "  length of resampling filter         " << x_nR         << endl 
		    << "  max number of F scan iterations     " << x_nScan      << endl;

	       if(x_refFilter) 
		 cout<<"  reference channel index             " << x_refFilter  << endl;

	       if(!(al_E.empty()))
		   if(fig == int(al_E[al_E.size()-1]->fid))
		     cout<< "  alarm SNR threshold                " << al_E[al_E.size()-1]->snr<<endl;
	       if(maxStride < x_timeStride/x_nT) maxStride = x_timeStride/x_nT;
// keep track of fid for alarm
	       fig+=1;
// set filter
	       lf = new LineFilter(fabs(x_frequency),x_timeStride,x_fid,x_nT); // initialize LF
	       lf->setFilter(x_nFirst,x_nLast,x_nSkip,0,x_nBand,x_nR);   // set LF parameters    
	       lf->Stride = timeStride;
	       x_fScan = (x_fScan!=0.) ? fabs(x_frequency) : x_fScan;
	       lf->setFScan(x_fScan,x_SNR,x_fBand,x_nScan);     // set F scan 
	       Filter.push_back(lf);                            // add filter to the list
	       n = int(channelName.size()-1); 
	       channelIndex.push_back(n);                       // add channel index to the list
	       refFilter.push_back(x_refFilter);
	       channelRate.push_back(x_rate);
	   
// set frequency tag & name tag
	       ftag = new char[100];
	       x_frequency = -fabs(x_frequency);
	       n = int(x_frequency);
	       sprintf(ftag,"%d.%02d%s",n,int(100.*(n - x_frequency)),nameTag);
	       freqTag.push_back(ftag);
	       sprintf(nameTag,    "%s","");
	   
// set file name for LineMonitor trends
	       fname = new char[100];
	       sprintf(fname,"%s%s%s-%s.lm",outputDir,c,ftag,gps);
	       while((w = strchr(fname,':'))) w[0] = '-';
	       
	       if(dumpLM) cout <<"output LineFilter file name: "<< fname << endl;
	       fileName.push_back(fname);   // add file name to the list
	       
	     }
	 }
     }
   
   //+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++=

// check useage of initialization function

// no channel is specified
   if(!channelName.size()) {
     cout << "No channel is specified. " << endl;
     DumpHelp();
   }
   getDacc().setIgnoreMissingChannel(true);

// check if directory for output file exists

   char dir[100];
   if (dumpLM) {
     sprintf(dir,"%s.",outputDir);
     if( (out=fopen(dir,"r"))==NULL ) {
       cout << "Error: can't open output trend  file " << dir << endl;
       exit(0);
     }
     fclose(out);
   }

   sprintf(dir,"%s.",trendDir);
   if( (out=fopen(dir,"r"))==NULL ) {
      cout << "Error: can't open output frame  file " << dir << endl;
      exit(0);
   }
   fclose(out);

   sprintf(dir,"%s.",htmlDir);
   if( (out=fopen(dir,"r"))==NULL ) {
      cout << "Error: can't open output html file " << dir << endl;
      exit(0);
   }
   fclose(out);

// set server Name
   string ifo(channelName[0], 2);

   if(strlen(suffix))
     sprintf(serverName,"LineMon_%s_%s",ifo.c_str(),suffix);
   else
     sprintf(serverName,"LineMon_%s",channelName[0]);

   while((w = strchr(serverName,':'))) w[0] = '-';
   setServerName(serverName);

// set name for frame file
   sprintf(trendFile,"%s%s-%s.gwf",trendDir,serverName,gps);
   while((w = strchr(trendFile,':'))) w[0] = '-';
//   cout <<"output trend file name: "<< trendFile<< endl;
     
// set html file name
   sprintf(indexHTML,"%s",htmlDir);
   if(strlen(htmlFile)){
     sprintf(dir,"%s%s",htmlDir,htmlFile);
     sprintf(htmlFile,"%s",dir);
   }
   else{
     sprintf(htmlFile,"%s%s_%s.html",htmlDir,serverName,gpsTime);
   }
   while((w = strchr(htmlFile,':'))) w[0] = '-';
   cout <<"output html file name: "<< htmlFile << endl;
 

// set time stride
   float tStep(timeStride);
   getDacc().setStride(tStep);

// set channel
   for(i=0; i<int(channelName.size()); i++){
     getDacc().addChannel(channelName[i]);
     ts.push_back(NULL);
     pw = new WSeries<double>(B);
     ws.push_back(pw);
   }
// read OperStateCondition file
   if(strlen(lockStatus)) {
     mOSC.setStride(int(timeStride));
     if( (out=fopen("/export/home/ops/pars/LockLoss.conf","r"))==NULL ) {
       cout << "Error: can't open /export/home/ops/pars/LockLoss.conf "<< endl;
       fclose(out);

       if( (out=fopen("LockLoss.conf","r"))==NULL ) {
	 cout << "Error:         can't open LockLoss.conf file"<< endl;
	 cout << "copy LockLoss.conf file into local directory"<< endl;
	 exit(0);
       }
       else{
	 fclose(out);
	 mDict.addPar("OSCFile",    string("LockLoss.conf"));
       }
     }
     
     else{
       fclose(out);
       mDict.addPar("OSCFile",    string("/export/home/ops/pars/LockLoss.conf"));
     }
   }
     
   if (!mDict.getString("OSCFile").empty()) {
     cout<<"read lock loss file: "<<mDict.getString("OSCFile").c_str()<<endl;
     mOSC.readConfig(mDict.getString("OSCFile").c_str());
   }

// set LineTable
   if(maxStride > 8) maxStride = 8.;
   if(maxStride > timeStride || maxStride == 0.)
     maxStride = timeStride;
   n = int(timeStride/maxStride);
   if(n<1) n = 1;
   lmonTable = new LineTable(1./maxStride, n, SNR*SNR*8./sqrt(double(n)));

// setup frame trend data
   while((w = strchr(serverName,'-'))) w[0] = 'x';
   while((w = strchr(serverName,'-'))) w[0] = 'x';
   while((w = strchr(serverName,'-'))) w[0] = 'x';
   mTrend.setName(serverName);
   mTrend.setType(Trend::kMinute);
   mTrend.setFrameCount(1);
   mTrend.setIFO(ifo.c_str());

// dynamically allocate memory to store the current timeseries for
// three line parameters. Make names of channels to be displayed on dmt viewer. 
// Setup data server from three timeseries for amplitude, frequency, phase

   size_t nH,nV;
   TSeries *pts;
   double str=1.;

   for(size_t i=0; i<Filter.size(); i++){
     
     nH = Filter[i]->nLast - Filter[i]->nFirst;
     nH/= abs(Filter[i]->nStep);
     nH = (nH>0) ? nH+2 : 2;
     str= Filter[i]->Stride;

     string pch = channelName[channelIndex[i]];
     pch.erase(0, pch.find(":")+1);
     string::size_type ix;
     while ((ix=pch.find("-")) != string::npos) pch[ix]='_';

     for(size_t n=0; n<nH; n++){

       pts = new TSeries(Time(0),Interval(str),nBuffer);
       amplitude_tseries.push_back(pts);

       if(n) pts = new TSeries(Time(0),Interval(str),nBuffer);
       else  pts = new TSeries(Time(0),Interval(str),nBuffer);
       frequency_tseries.push_back(pts);

       if(n) pts = new TSeries(Time(0),Interval(str),nBuffer);
       else  pts = new TSeries(Time(0),Interval(str),nBuffer);
       phase_tseries.push_back(pts);    
       
       pc = new char[100];
       if(n) sprintf(pc,"%s:DMT-LINE_%s_A%zd%s",ifo.c_str(),pch.c_str(),n,freqTag[i]);
       else  sprintf(pc,"%s:DMT-LINE_%s_AT%s",ifo.c_str(),pch.c_str(),freqTag[i]);
       amplitude_name.push_back(pc);    
       
       pc = new char[100];
       if(n) sprintf(pc,"%s:DMT-LINE_%s_F%zd%s",ifo.c_str(),pch.c_str(),n,freqTag[i]);
       else  sprintf(pc,"%s:DMT-LINE_%s_FB%s",ifo.c_str(),pch.c_str(),freqTag[i]);
       frequency_name.push_back(pc);
       
       pc = new char[100];
       if(n) sprintf(pc,"%s:DMT-LINE_%s_P%zd%s",ifo.c_str(),pch.c_str(),n,freqTag[i]);
       else  sprintf(pc,"%s:DMT-LINE_%s_SNR%s",ifo.c_str(),pch.c_str(),freqTag[i]);
       phase_name.push_back(pc);
       
       nV = amplitude_name.size()-1; 
       serveData(amplitude_name[nV],amplitude_tseries[nV]);
       serveData(frequency_name[nV],frequency_tseries[nV]);
       serveData(    phase_name[nV],    phase_tseries[nV]);

       cout<<"trend data channel name: "<<amplitude_name[nV]<<endl;
       cout<<"trend data channel name: "<<frequency_name[nV]<<endl;
       cout<<"trend data channel name: "<<phase_name[nV]<<endl;

       mTrend.addChannel(amplitude_name[nV]);
       mTrend.addChannel(frequency_name[nV]);
       mTrend.addChannel(phase_name[nV]);

     }
   }
	 
// set name for index file
   sprintf(indexFile,"%schannel-%s.cfg",trendDir,gpsTime);
   while((w = strchr(indexFile,':'))) w[0] = '-';
   mTrend.writeIndex(indexFile);
   sprintf(indexFile,"%schannel.cfg",trendDir);
//   mTrend.writeIndex(indexFile);

// Define an alarm
   if(!(al_E.empty()))
     {  
       if(strlen(suffix))
	 sprintf(a_namea,"LineMonitor_%s_%s",ifo.c_str(),suffix);
       else
	 sprintf(a_namea,"LineMonitor_%s",ifo.c_str());
       if(plow[0])
	 {
           const char* Desca="Line(s) at $1 Hz above the threshold";
	   AlarmData alm_sa(a_namea,"line_excited", Interval(3*timeStride*nDump), 4,Desca);
	   alm_sa.setWebFile("http://www.phys.ufl.edu/LIGO/LINE/lndt.html");
	   al_ca.defineAlarm(alm_sa);
	 }
       if(plow[1])
	 {
           const char* Descb="Line(s) at $1 Hz below the threshold";
	   AlarmData alm_sb(a_namea,"line_missing", Interval(3*timeStride*nDump), 4,Descb);
	   alm_sb.setWebFile("http://www.phys.ufl.edu/LIGO/LINE/lndt.html");
	   al_cb.defineAlarm(alm_sb);
	 }
       if(debug)
	 {
	   size_t ai = al_E.size();
	   size_t aj;
	   cout<<"These are the initialization parameters for the alarms."<<endl;
	   for(aj=0;aj<ai;aj++)
	     {
	       cout<<"Line Frequency = "      <<fabs(atof(freqTag[al_E[aj]->fid]))
		   <<", Signal to noise ratio = "    <<al_E[aj]->snr
		   <<", If alarm is on = "<<al_E[aj]->b  
		   <<", If there has been a change = "<<al_E[aj]->c
		   <<", alarm ID = "<<al_E[aj]->fid<<endl;
	     }
	 }
     }
   

}

//--------------------------------------  Skeleton object destructor.
LineMonitor::~LineMonitor() 
{
    size_t i;
    delete lmonTable;

    for(i=0; i<Filter.size(); i++){
	delete Filter[i];
	delete fileName[i];
	delete freqTag[i];
    }

    for(i=0; i<channelName.size(); i++){
	delete channelName[i];
	delete ws[i];
    }

    if( !(al_E.empty()) ){
      al_ca.cancelAlarm(al_ha);
      al_cb.cancelAlarm(al_hb);
	for(i=0;i<al_E.size();i++)
	    delete al_E[i];
    }
    for(i=0; i<amplitude_name.size(); i++){
	if(amplitude_tseries[i]) delete amplitude_tseries[i];
	if(phase_tseries[i])     delete phase_tseries[i];
	if(frequency_tseries[i]) delete frequency_tseries[i];
	if(amplitude_name[i])    delete amplitude_name[i];
	if(phase_name[i])        delete phase_name[i];
	if(frequency_name[i])    delete frequency_name[i];
    }
    
    mTrend.close();
    
    cout << "LineMonitor is finished" << endl;
}

//-- This skeleton function needed for writing to the dmt viewer

void LineMonitor::Attention(void) {
  MonServer::Attention();
  return;
}

/*  Frame processing function.      *
 *  do this step every timestride.  */
void LineMonitor::ProcessData(void) {
    
    wavearray<double> wa;
    wavearray<float> wdv;
    wavearray<double>* pwa = NULL;
    WSeries<double>* pws = NULL;
    TSeries spsd;
    TSeries npsd;
    double stride=0.;
    int k,al_len=0,al_p;
    LineFilter F(1.);
    size_t nV = 0, nH = 0, n, i, al_i;

    int nR;
    bool al_eb[2]={false,false};
    bool al_ec[2]={false,false};         
    float al_snr1=0.0,al_snr2=0.0,al_co1=0.0,al_co2=0.0;

    lmsg::error_t al_er =0;

    bool update = true, online = true;
    bool inlock = true;
    char name[256];
    int nn = 0;
    int nLPF = 0;

    if(!dumpLM) appendMode = 1;
    
    for(i=0; i<channelName.size(); i++){
      ts[i] = getDacc().refData(channelName[i]);
      if(!ts[i]) {cout<<"LineMonitor: Did not receive data!"<<endl; return;}
      pwa = (wavearray<double>*)(ws[i]);
      *pwa = *(ts[i]);
      ws[i]->pWavelet->reset();
    }
    
    currentTime = ts[0]->getStartTime();
    
    if(mDump == 0 && maxDump == 0) 
        frameStart = Time((long(currentTime.getS())/60+1)*60);
    
    if((mDump == 0 && maxDump == 0) || maxDump > 100000){
	sprintf(gpsTime,"%d",int(currentTime.getS()));
	maxDump = 0;
    }    
    
// print the timeseries gps time 
    if(debug>0) cout << "Start Time: " << (currentTime.getS());
    if(strlen(lockStatus)) 
      inlock = mOSC.satisfied(lockStatus);
    
    if(!inlock){
      ifolock = 0;
      cout<<endl;
      if(!al_E.empty()) {
	if(al_set[0]) { al_ca.cancelAlarm(al_ha); al_set[0]=false; }
	if(al_set[1]) { al_cb.cancelAlarm(al_hb); al_set[1]=false; }
      }
    }
    else if(debug>0) printf(" :  Processing data. \n");

    if(F.isDataValid(*ts[0]) == false) {cout<<"... data is not valid\n"; return;}

    if(debug>4) printf("lock status: %d \n",int(inlock));

    dmtDump++;

    if(appendMode && (dmtDump >= nDump)) {
      al_i=0;
      if(Filter.size()) stride = Filter[0]->Stride;

      for(i=0; i<Filter.size(); i++) {
	nH = Filter[i]->nLast - Filter[i]->nFirst;
	nH/= abs(Filter[i]->nStep);
	nH = (nH>0) ? nH+2 : 2;
	
	for(n=0; n<nH; n++) {
	  nn = (n<2) ? n : (n-1)*abs(Filter[i]->nStep)+1;
	  *(amplitude_tseries[nV])       = getTrend(Filter[i],nn,'a');
	  if(n) *(frequency_tseries[nV]) = getTrend(Filter[i],nn,'f');
	  else  *(frequency_tseries[nV]) = getTrend(Filter[i],nn,'F');
	  if(n){ 
	    if(refFilter[i]==0) *(phase_tseries[nV]) = getTrend(Filter[i],nn,'p');
	    else                *(phase_tseries[nV]) = getTrend(Filter[i],nn,'P');
	  } 
	  else  *(phase_tseries[nV])     = getTrend(Filter[i],nn,'K');
	  nV++;
	}
	
// This section calculates the SNR in order to cancel or set alarms
	if(!al_E.empty()) {

	  online = (!inlock || ifolock<4) ? false : true;

	  for(al_i=0; al_i<al_E.size(); al_i++) if(al_E[al_i]->fid==i) break;

	  if(al_i < al_E.size()) {
	    spsd = getTrend(Filter[i],0,'S'); 
	    npsd = getTrend(Filter[i],0,'N');
	    al_len =int(timeStride/Filter[i]->Stride + 0.5);
	    al_p   = spsd.getNSample(); 
	    al_snr1=0.0; 
	    al_snr2=0.0;
	    al_co1 =0.0; 
	    al_co2 =0.0;

	    if(debug>7) cout<<"al_len="<<al_len<<" al_p="<<al_p<<endl;

	    if(2*al_len > al_p) 
	      cout<<"LineMonitor error: buffer length = "<<al_p<<" subdivisions = "<<al_len<<endl;
	    
	    if( online && al_len > 0 ) {
	      
	      for(k=0;k<al_len;k++) {
		al_snr1 += float(npsd.getDouble(al_p - 1 - k));
		al_snr2 += float(npsd.getDouble(al_p - 1 - k - al_len));
		al_co1  += float(spsd.getDouble(al_p - 1 - k));
		al_co2  += float(spsd.getDouble(al_p - 1 - k - al_len));
	      }
	      
	      al_snr1/=float(al_len); 
	      al_snr2/=float(al_len);
	      
	      if(al_snr1 <= 0) al_snr1 = 0.;
	      else al_snr1= al_co1/al_snr1-float(al_len); 
	      
	      if(al_snr2 <= 0) al_snr2 = 0.;
	      else al_snr2= al_co2/al_snr2-float(al_len); 
	      
	    }
	    
	    if(debug>7) cout<<"Signal to noise ratios: 1) "<<al_snr1<<" 2) "<<al_snr2<<endl;

// This section sets the logic conditions if the signal to noise ratio given is an upper bound (ie positive)
		    
	    if(al_E[al_i]->snr >= 0 ) {
	      if(al_snr1 > al_E[al_i]->snr && al_snr2 > al_E[al_i]->snr && online) {
		if(al_E[al_i]->b) al_E[al_i]->c=false;
		else              al_E[al_i]->c=true;
		al_E[al_i]->b=true;
	      }
	      else {
		if(!(al_E[al_i]->b)) al_E[al_i]->c=false;
		else                 al_E[al_i]->c=true;
		al_E[al_i]->b=false;
	      } 
	    }

// This section sets the logic conditions if the signal to noise ratio given is a lower bound (ie negative)

	    else {
	      if(al_snr1 <  fabs(al_E[al_i]->snr) && al_snr2 <  fabs(al_E[al_i]->snr) && online) {
		if(al_E[al_i]->b) al_E[al_i]->c=false;
		else              al_E[al_i]->c=true;
		al_E[al_i]->b=true;
	      }
	      else {
		if(!(al_E[al_i]->b)) al_E[al_i]->c=false;
		else                 al_E[al_i]->c=true;
		al_E[al_i]->b=false;
	      } 
	    }
	  }           // end of if(al_E[al_i]->fid==i)
	}             // end of if(!al_E.empty() && (al_i < al_E.size())) 
      }               // end of for(i=0; i<Filter.size(); i++)

// SNR Calculation complete 	
// Here an Alarm is set if one line > set SNR or cancelled if otherwise

      if(!al_E.empty()) {       // should be here on every stride
	
	al_eb[0]=false;
	al_eb[1]=false;
	ostringstream prama;
	ostringstream pramb;
	
	for(i=0; i<al_E.size() ;i++) {
	  
	  if(al_E[i]->b && (al_E[i]->snr >=0) ) {
	    if(al_eb[0])prama<<","<<fabs(atof(freqTag[al_E[i]->fid]));
	    else prama<<fabs(atof(freqTag[al_E[i]->fid]));
	    al_eb[0]=true;    // got first alarm
	  }
	  if(al_E[i]->c && (al_E[i]->snr >= 0)) al_ec[0]=true;
	  
	  if(al_E[i]->b && (al_E[i]->snr <0) ) {
	    if(al_eb[1])pramb<<","<<fabs(atof(freqTag[al_E[i]->fid]));
	    else pramb<<fabs(atof(freqTag[al_E[i]->fid]));
	    al_eb[1]=true;    // got second alarm
	  }
	  if(al_E[i]->c && (al_E[i]->snr <0)) al_ec[1]=true;
	  
	}
	prama<<ends;
	pramb<<ends;
	
// handle first alarm

	if(al_eb[0]) {        // alarm is present
	  al_er = 1;
	  if(al_set[0]) {     // alarm was set
//	    al_ca.cancelAlarm(al_ha);
	    al_er = al_ca.prolongAlarm(al_ha,Interval(1.3*timeStride*nDump));
	    if(debug>7 && !al_er)
	      cout<<" Prolonging alarm 1 for lines "<<prama.str()<<endl;
	    if(al_er)
	      cout<<" Alarm 1 prolonging error "<<al_er<<" for lines "<<prama.str()<<endl;
	  }

	  if(debug>7 || al_er)
	    cout<<" Setting alarm 1 for lines "<<prama.str()<<endl;
	  
	  if(al_er){
	    AlarmData al_da(a_namea,"line_excited" ,Interval(2*timeStride*nDump), 4 , "",prama.str());
	    al_er=al_ca.setAlarm(al_da,al_ha);
	    al_set[0]=true;
	    if(al_er){
	      cout<< " [LineMonitor] alarm 1 setting error: "<<al_er<<endl;
	      al_ca.cancelAlarm(al_ha);
	      al_set[0]=false;
	    }
	  }
	}
	
	else {	       // alarm is not present
	  if(al_set[0]) {
	    al_er=al_ca.cancelAlarm(al_ha); 
	    al_set[0]=false; 
	    
	    if(debug>7)
	      cout<<" cancel alarm 1 for lines "<<prama.str()<<", error: "<<al_er<<endl;
	    
	    if(al_er) 
	      cout<< " [LineMonitor] alarm 1 cancel error: "<<al_er<<endl;	
	  }
	}
      
// handle second alarm

	if(al_eb[1]) {
	  al_er = 1;
	  if(al_set[1]) {     // alarm was set
	    //	    al_cb.cancelAlarm(al_hb);
	    al_er=al_cb.prolongAlarm(al_hb,Interval(1.3*timeStride*nDump));
	    if(debug>7 && !al_er)
	      cout<<" Prolonging alarm 2 for lines "<<pramb.str()<<endl;
	    if(al_er)
	      cout<<" Alarm 2 prolonging error "<<al_er<<" for lines "<<pramb.str()<<endl;
	  }

	  if(debug>7 || al_er)
	    cout<<" Setting alarm 2 for lines "<<pramb.str()<<endl;
	  
	  if(al_er){
	    AlarmData al_db( a_namea,"line_missing" ,Interval(3*timeStride*nDump), 4 , "",pramb.str());
	    al_er=al_cb.setAlarm(al_db,al_hb);
	    al_set[1]=true;
	    if(al_er) {
	      cout<< " [LineMonitor] alarm 2 setting error: "<<al_er<<endl;
	      al_cb.cancelAlarm(al_hb);
	      al_set[1]=false;
	    }	  
	  }
	}

	else {	       
	  if(al_set[1]) {
	    al_er=al_cb.cancelAlarm(al_hb); 
	    al_set[1]=false; 
	    
	    if(debug>7)
	      cout<<" Cancelling alarm 2 for lines "<<pramb.str()<<",  error:  "<<al_er<<endl;
	    
	    if(al_er) 
	      cout<< " [LineMonitor] alarm 2 cancel error: "<<al_er<<endl;	
	  }
	}

// debug 

	if(debug>7) {
	  size_t ai = al_E.size();
	  size_t aj;
	  const char *wu;
	  const char *wv;
	  float kl;
	  cout<<"These are the alarm settings."<<endl;
	  for(aj=0; aj<ai; aj++)
	    {
	      kl=fabs(atof(freqTag[al_E[aj]->fid]));
	      if(al_E[aj]->b)
		wv="on. ";
	      else 
		wv="off. ";
	      if(al_E[aj]->c)
		wu="";
	      else
		wu=" not";
	      cout<<" Line Frequency  = "<<kl
		  <<" Set signal to noise ratio = "
		  <<al_E[aj]->snr<<" The alarm is "<<wv
		  <<" There has"<<wu<<" been a change."<<endl;
	    }
	}   
      }
      
// The alarm is now set or canceled.
// Handle reference lines
      nV = 0;
      for(i=0; i<Filter.size(); i++){
	nH = Filter[i]->nLast - Filter[i]->nFirst;
	nH/= abs(Filter[i]->nStep);
	nH = (nH>0) ? nH+2 : 2;
	nR = refFilter[i]>0 ? getRefIndex(i) : -2;   // find the reference line
	
	nV++; nR++;
	for(n=1; n<nH; n++){
	  if(nR>=0){
	    *(phase_tseries[nV]) -= *(phase_tseries[nR++]);
	    wdv = *(phase_tseries[nV]);
	    for(size_t j=0; j<wdv.size(); j++){
	      if(wdv[j] > PI) wdv[j] -= 2.*PI;
	      if(wdv[j] <-PI) wdv[j] += 2.*PI;
	    }
	    phase_tseries[nV]->setData(phase_tseries[nV]->getStartTime(), 
				       Interval(phase_tseries[nV]->getTStep()), 
				       wdv.data, wdv.size());
	  }
	  nV++;
	}
      } 
      
// update trends
      update = true;
      while(update){
	nV = 0;
	for(i=0; i<Filter.size(); i++)
	  {
	    nH = Filter[i]->nLast - Filter[i]->nFirst;
	    nH/= abs(Filter[i]->nStep);
	    nH = (nH>0) ? nH+2 : 2;
	    for(n=0; n<nH; n++)
	      {
		update = setTrend(*(amplitude_tseries[nV]), amplitude_name[nV]);
		update = setTrend(*(frequency_tseries[nV]), frequency_name[nV]);
		update = setTrend(*(phase_tseries[nV]), phase_name[nV]);
		nV++;
	      }
	  }
	if(update) frameStart += Interval(60.);
	if(debug>5) printf("frame start: %li\n",frameStart.getS());
      }
      dmtDump = 0;
      
    }                 // end of if(appendMode && (dmtDump >= nDump))
    
    if(inlock) {      // execute only if in lock 
      
      ifolock++;
      
// decimate master channel
      nLPF = masterRate>0. ? getDeep(ws[0]->rate(),masterRate) : 0;
      if(nLPF>0) ws[0]->Forward(nLPF);
      ws[0]->getLayer(wa,0);
    
// run LineFinder
      lmonTable->lineFinder(wa,0.,0.);
      lmonTable->update(Filter, channelIndex);

// update lmonTable with current alarm status
      if(!al_E.empty())lmonTable->update(al_E); 

// apply line filter to data segment
      for(i=0; i<Filter.size(); i++){
	pws = ws[channelIndex[i]];
	pws->getLayer(wa,0);
	if(channelRate[i]<=0.){
	  nLPF = getDeep(pws->rate(), wa.rate());
	  if(nLPF>0) { pws->Inverse(nLPF); pws->getLayer(wa,0); }
	}
	else{
	  nLPF = getDeep(wa.rate(),channelRate[i]);
	  if(nLPF > 0) { pws->Forward( nLPF); pws->getLayer(wa,0); }
	  if(nLPF < 0) { pws->Inverse(-nLPF); pws->getLayer(wa,0); }
	}
	Filter[i]->apply(wa);
      }
      mDump++;
  
// append data from filter to binary file
 
      if(mDump == nDump){
	if(!maxDump && dumpLM) appendMode = 0;
	
	for(i=0; i<Filter.size(); i++){
	  sprintf(name,fileName[i],gpsTime);
	  if(dumpLM) Filter[i]->DumpTrend(name,appendMode); 
	  Filter[i]->resize(nBuffer);           // pop_front the lineList
	}
	appendMode = 1;
	maxDump += mDump;
	mDump = 0;
      }
    }

    if(debug>4) cout<<"ready to dump summary  ...";
      
    sprintf(name,"%sindex.html",indexHTML);
    lmonTable->DumpHTML(channelName, name);
    if(!al_E.empty())mktag(name);
    sprintf(name,"%sindex.revolver.html",indexHTML);
    lmonTable->DumpHTML(channelName, name,0);
    lmonTable->DumpHTML(channelName, htmlFile);
    if(!al_E.empty()) mktag(name);
    if(!al_E.empty()) mktag(htmlFile);
    if(debug>4) cout<<"done"<<endl;

    return;
}


// This function replaces "|-|" with a "<",in the file botch, so you can input html tags 
// when you do lmonTable->DumpHTML(,)
void    LineMonitor::mktag(const char *botch)
{
  fstream gdstf(botch,ios::in | ios::out);
  if (!gdstf) 
  {
      cerr<<"File can't be opened for make tag"<<endl;
      return;
  }
  int l=10000;
  char buf[l];
  const char *kk="|-|";
  char *bk,*ko;
  long koko,loko=0;
  while( !(gdstf.eof()) )
  {
      gdstf.getline(buf,sizeof(buf), '\n');
      ko=buf;
      koko=strlen(ko);
      if( !(gdstf.eof()) && !(strlen(buf) == 0) )
      { 
	  
	  bk =strstr(buf,kk);
	  if (!(bk == NULL))
	  {
	      loko=0;
	      while(loko < (koko-2) )
	      {
		  if(buf[loko]==           *("|"))
		      if( buf[loko+1] ==   *("-"))
			  if(buf[loko+2]== *("|"))
			  {
			      buf[loko  ] =*(" ");
			      buf[loko+1] =*(" ");
			      buf[loko+2] =*("<");
			  }
		  loko+=1;
	      }
	      gdstf.seekp(-(koko+1),ios::cur);
	      gdstf<<buf<<endl;
	  }
      }
  }
  gdstf.close();
}

TSeries LineMonitor::getTrend(LineFilter *inF, int n, char c)
{
   size_t i,k; 
   wavearray<float> xts;
   wavearray<float> t;
   wavearray<float> wts(nBuffer);
   LineFilter *F = inF;
   TSeries ts;
   wts = 0;
   t   = F->getTrend(0,'t');
   k = t.size();
   double sample = 1./t.rate();
   double start = t.start();
   double time = currentTime.getS();

   //   cout<<"data start="<<(long)t.start()<<"  current time="<<(long)time<<endl;
   //   cout<<"data   end="<<(long)(t.start()+t.data[k-1])<<" sample="<<sample<<endl;
    
   if(t.size()<3 || F->Stride <= 0.){
     ts.setData(Time(int(time-nBuffer*sample)), Interval(sample), wts.data, nBuffer);
     return ts;
   }
  
   xts = F->getTrend(n,c);

   if(c == 'F')
     for(i=0; i<k; i++)
       if(xts.data[i]<0) xts.data[i] *= -1.;

   if(c == 'f' || c == 'F')
     wts = xts.mean();
   else
     wts = 0.;

   wts.rate(xts.rate());

   i = nBuffer;

//   cout<<long(time)<<" "<<long(start+t.data[k-1])<<endl;

   while(i>0 && k>0){
     time -= sample;
     i--;
     if(time <= (start + (double)t.data[k-1]))
       wts.data[i] = xts.data[--k];
   }

//   cout<<long(time)<<" "<<long(time-i*sample)<<endl;
 
   ts.setData(Time(int(time-i*sample)), Interval(sample), wts.data, nBuffer);
   return ts;
}

bool LineMonitor::setTrend(const TSeries& ts, char* pc)
{
  Trend::math_t Data;

  size_t N=ts.getNSample();
  size_t n=ts.getNSample();
  
  double sample = ts.getTStep();
  double time = double(ts.getStartTime().totalS())+N*sample;

  double frameTime = frameStart.getS();

  float* vec_ref = (float*)(ts.refData());
   
  int k = 0;
  Data = 0.;

  while(time >= frameTime) { time -= sample; n--; }

  if(n<0) { cout<<"LineMonitor::setTrend error: n="<<n<<endl; exit(0); }

  for (size_t i=n+1; i<N; i++){
    time += sample;
    if(time < frameTime+60.) { Data += vec_ref[i]; k++; }
    else{
      Data = (k) ? Data/float(k) : vec_ref[i-1];
      mTrend.trendData(pc, Time(int(frameTime)), Data);
      return true;
    }
  }

  return false;

}

int LineMonitor::getRefIndex(int index){

  int n;
  int k = refFilter[index]-1; // count from 0 
  int refIndex = 0;
  if(k < 0 ) return -2;
  if(Filter[index]->nFirst != Filter[k]->nFirst) return -2;
  if(Filter[index]->nLast  != Filter[k]->nLast)  return -2;
  if(Filter[index]->nStep  != Filter[k]->nStep)  return -2;

  for(int i=0; i<k; i++){
    n  = Filter[i]->nLast - Filter[i]->nFirst;
    n /= abs(Filter[i]->nStep);
    refIndex += (n>0) ? n+2 : 2;
  }
  return refIndex;
}
	

char *LineMonitor::getToken(pchar &s, bool ignore)
// ignore = true - ignore punctuation except space
{
   char *w = 0;
   char *eos = s+strlen(s); 

   s--;
   while(++s < eos){

      if(ispunct(*s) && !ignore && *s != '-' && *s != '.' ){
	*s = '\0'; break;
      }

      if(!isspace(*s) && !w) w = s;  // first character
      if( isspace(*s) &&  w){ *(s++) = '\0'; break; }

   }
//   cout <<strlen(w)<<"  token  "<< w << endl;
   return (w) ? w : eos;
}


void LineMonitor::DumpHelp(void) {
   cout << "                   LINE MONITOR                          " << endl
        << "                                                         " << endl  
        << "reference: http://www.phys.ufl.edu/LIGO/LINE/linemon.html" << endl
        << "           S.Klimenko, University of Florida             " << endl
        << "                                                         " << endl  
        << "Perpose: LineMonitor is used to track narrow lines in the" << endl
        << "         data (typically line width should less then 1Hz)" << endl  
        << "         We define a line as a single narrow line or a   " << endl  
        << "         group of harmonics with the same base frequency." << endl  
        << "                                                         " << endl  
        << "Method: LineMonitor is based on the Quasi-Monochromatic  " << endl  
        << "        Line Removal method developed at UF. It allows to" << endl  
        << "        find strength, frequency and phase of specified  " << endl  
        << "        lines (including harmonics). It uses a LineFilter" << endl  
        << "        object to track harmonics with specific seed freq" << endl  
        << "        To monitor several lines, one LF object per line " << endl  
        << "        is created by LineMonitor.                       " << endl  
        << "                                                         " << endl  
        << "Output: The LM can produce several trend files, one per  " << endl
        << "        line. The trend data includes line frequency,    " << endl
        << "        phase, amplitude, spectral density of noise in   " << endl
        << "        the line vicinity, SNR, GPS time, etc. For output" << endl
        << "        should be specified in global variable DMTOUTPUT." << endl
        << "        There is no need to specify file names, since    " << endl
        << "        they are constructed by the LM.                  " << endl
        << "                                                         " << endl  
        << "command line :                                           " << endl  
        << "                                                         " << endl  
        << "  -c <channel name>: Channel name is mandatory. LM can be" << endl
        << "     run on one channel only. However, several LMs can be" << endl  
        << "     launched to monitor different channels.             " << endl  
        << "                                                         " << endl  
        << "  -infile <input frame file>: If  -infile is specified,  " << endl
        << "     the LM will read data from the input frame file.    " << endl
        << "                                                         " << endl  
        << "  -i <input config file>: The LM configuration file to   " << endl
        << "     track several lines. All options except -i & -infile" << endl
        << "     can be used to build an entry (string) in the file. " << endl            
        << "     Several parameters can be specified in the string.  " << endl            
        << "     If a string starts with -f F,then a LineFilter entry" << endl
	<< "     for line with frequency F is created.There should be" << endl
        << "     only one entry per line. All other LineFilter param." << endl
        << "     (options: -t, -n, -L, -F, -S, -s, -R, -I, -W) should" << endl
        << "     be specified in the same string. It's recommended to" << endl
        << "     put LineFilter entries in the end of the config.file" << endl
        << "     For example,                                        " << endl            
        << "        -c L0:PEM-LVEA_V1  // track LLO power monitor    " << endl            
        << "        -l L1:X_arm_loced  // when x arm locked          " << endl            
        << "        -t 4 -d 10  // stride 4sec, dump every 10 strides" << endl            
        << "        -f  60. -L 8    // track 8 harm of 60Hz line     " << endl            
        << "        -f 272. -L 1 -s // calibration line at 272Hz     " << endl            
        << "        -f  35. -t 1    // monitor 35Hz with 1sec stride " << endl            
        << "                                                         " << endl  
        << "  -l <lock condition> Track lines if some lock condition " << endl
        << "     is specified. For example:                          " << endl
        << "           H2:Both_arms_locked                           " << endl
        << "           H1:Mode_cleaner_locked                        " << endl
        << "           L1:X_arm_locked                               " << endl
        << "                                                         " << endl  
        << "  -t <time stride (sec)>: Length of time series requested" << endl
        << "     by LineMonitor. It sets the trend data sampling rate" << endl
        << "                                                         " << endl  
        << "  -n <number of stride subdivisions>: This parameter is  " << endl
        << "     needed to track weak and wide lines. Its recommended" << endl
        << "     to keep n/t > line_width.                           " << endl
        << "                                                         " << endl  
        << "  -f <seed frequency (Hz)>: Currently the LM tracks lines" << endl
        << "     with approximately known frequency. Typically the   " << endl
        << "     seed frequency error should be order of n/t, where  " << endl
        << "     t is a time stride and n is the number of stride    " << endl
        << "     subdivisions.                                       " << endl
        << "                                                         " << endl  
        << "  -I <filter ID (0/1)>: Line Filter ID                   " << endl
        << "     0 - find LM filter ignoring noise around the line   " << endl
        << "     1 - find LM filter taken into account the noise     " << endl
        << "                                                         " << endl  
        << "  -s <no frequency scan>: Do not perform frequency scan  " << endl
        << "     for lines with exactly known frequency (caliblines) " << endl
        << "                                                         " << endl  
        << "  -lm <>                 output the LineFilter data file." << endl
        << "                                                         " << endl  
        << "  -F <first harmonic>: Specify first harmonic            " << endl
        << "                                                         " << endl  
        << "  -L <last harmonic>: Specify last harmonic. If L=0,track" << endl
        << "     track all harmonics  available for channelName      " << endl
        << "                                                         " << endl  
        << "  -S <skip harmonics>: Along with -F allows to select    " << endl
        << "     subset of harmonics. Foe example,                   " << endl
        << "      -F 1 -S 1   - select odd harmonics                 " << endl
        << "                                                         " << endl  
        << "  -K <limit on signal to noise ratio> [2.]               " << endl
        << "                                                         " << endl  
        << "  -R <decimated sampling rate> used for data decimation  " << endl
        << "                                                         " << endl  
	<< "  -r <reference channel index> index of line description " << endl
	<< "     field (begins with -f option) in the config file.   " << endl
        << "                                                         " << endl  
        << "  -d m <dump trend data every m strides> [1]             " << endl
        << "                                                         " << endl  
        << "  -H name <file name to dump html table> [construct name]" << endl
        << "     output directory can be specified in global variable" << endl
        << "     DMTHTMLOUT. If DMTHTMLOUT is not defind, then use a " << endl
        << "     local directory.                                    " << endl
	<< "  -a <signal to noise ratio(SNR) to set alarm>:This will " << endl
	<< "     set an alarm for the corresponding line when the SNR" << endl
	<< "     of the line is >= to this value. If there is a - in " << endl
	<< "     front of this value, the alarm will be set when the " << endl
	<< "     line drops below this value.This option can be      " << endl
	<< "     specified in either the configuration file or on the" << endl
        << "     command line, but the configuration file  outranks  " << endl
	<< "     the command line.                                   " << endl
	<< "  -debug Output debugging information for the Alarms     " << endl
        << "                                                         " << endl  
        << "useage: LineMonitor                                      " << endl  
        << "  -c <channel name> [required]                           " << endl
        << "  -H <output html file name> [synthesized]               " << endl
        << "  -i <input config file> []                              " << endl
        << "  -l <lock condition> (*:Both_arms_locked for IFO chan.) " << endl
        << "  -f <seed frequency (Hz)> [60.]                         " << endl
        << "  -t <time stride (s)> [1.]                              " << endl
        << "  -I <filter ID (-1/0/1)> [1]                            " << endl
        << "  -n <number of stride subdivisions> [1]                 " << endl
        << "  -s <no frequency scan>                                 " << endl
        << "  -F <first harmonic> [1]                                " << endl
        << "  -L <last harmonic> [1]                                 " << endl
        << "  -S <skip harmonics> [0]                                " << endl
        << "  -K <limit on signal to noise ratio> [2.]               " << endl
        << "  -d <dump trend data every n strides> n=[1]             " << endl
        << "  -b <length of the DMTVIEWER buffer> length=[1024]      " << endl
        << "  -R <decimated sampling rate>                           " << endl
	<< "  -r <reference channel index>                           " << endl
	<< "  -lrf <length of resampling filter>[8]                  " << endl
	<< "  -tag <name tag for trend channels>                     " << endl
	<< "  -a <SNR to trigger alarm>[0]                           " << endl
	<< "  -lm output LineFilter data                             " << endl
	<< "  -debug - print out the debug information               " << endl
        << "  -N <number of frequency bins to estimate noise SD>     " << endl;
/*      << "  -m <max number of iterations during frequency scan>    " << endl  */
/*      << "  -B <bandwidth in units of fft bins> [0.45]             " << endl  */
  exit(0);
  return;
}





























