/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef METIOWRAP_HH
#define METIOWRAP_HH

struct MetaioParseEnvironment;

#include <string>

namespace xsil {

    /**  MetaIO is a wrapper for the LigoTools metaio package. The package is
      *  used by various tools to read (or write?) Ligo light-weight tables.
      *  @memo    I/O to/from a Ligo lightweight table.
      *  @version 1.1; Last modified August 2, 2010
      *  @author  John Zweizig
      *  @ingroup IO_xsil
      */
    class MetaIO {
    public:
	/**  Construct a MetaIO reader.
	  *  @memo Default constructor.
	  */
	MetaIO(void);

	/**  Construct a Ligo-lightweight table reader, open a specified
	  *  file and optionally position the file to the named table.
	  *  @memo Construct a reader and open an xml file.
	  *  @param file Constant pointer to a name string.
	  *  @param table Table name string.
	  */
	MetaIO(const char* file, const char* table=0);

	/**  Destroy a Ligo-lightweight table reader.
	  */
	~MetaIO(void);

	/**  Close a Ligo-lightweight reader.
	  *  @return zero if successful.
	  */
	int close(void);

	/**  Get the name of the specified column.
	  *  @brief Get a column name
	  *  @param icol Number of column.
	  *  @return Pointer to name character string.
	  */
	const char* getColumnName(int icol) const;

	/**  Get a floating point number from the specified column of the 
	  *  current row. If the column data type differs from the requested 
	  *  data type GetFloat attempts to convert the data to a double.
	  *  @memo Get a float table entry.
	  *  @param cname Column name string pointer.
	  *  @param def   Default value if the column isn't defined.
	  *  @return Float contents of specified column.
	  */
	double getFloat(const char* cname, double def=0.0) const;

	/**  Get a long integer number from the specified column of the current
	  *  row.If the column data type differs from the requested data type
	  *  GetInt attempts to convert the data to a long integer.
	  *  @memo Get an integer table entry
	  *  @param cname Column name string pointer.
	  *  @param def   Default value if the column isn't defined.
	  *  @return Integer contents of specified column.
	  */
	long getInt(const char* cname, long def=0) const;

	/**  Get the current number of columns in the table.
	  *  @brief Number of columns
	  *  @return Number of columns.
	  */
	int getNColumn(void) const;

	/**  Read the next row of the table from the opened file.
	  *  @memo read on row of the table.
	  *  @return Zero if successful.
	  */
	int getRow(void);

	/**  Get a string value from the specified column of the current
	  *  row. A default value is returned if the column is not found.
	  *  If the column data type differs from the requested data type
	  *  GetString attempts to convert the data to a string.
	  *  @memo Get a table data string.
	  *  @param cname Column name string pointer.
	  *  @param def   Default value if the column isn't defined.
	  *  @return string contents of specified column.
	  */
	std::string getString(const char* cname, const char* def="") const;

	/**  Get the table name from the header information.
	  *  @memo Get table name.
	  *  @return Constant pointer to table name string.
	  */
	const char* getTableName(void) const;

	/**  Get the column type name.
	  *  @memo Get column type.
	  *  @param icol Column number.
	  *  @return Pointer to column type string.
	  */
	const char* getTypeName(int icol) const;

	/**  Test whether the parse environment is open. 
	  *  @memo Test if file is open.
	  *  @return true if the input file is open.
	  */
	bool is_open(void) const;

	/**  The specified file is opened and the header information is read.
	  *  @memo   Open an xml table file.
	  *  @param  file  Name of xml file to be processed.
	  *  @param  table Name of table to be processed.
	  *  @return Zero  indicates success.
	  */
	int open(const char* file, const char* table=0);

	/**  Select a named table.
	  *  @memo Select named table.
	  *  @param table Name of table to process.
	  *  @return Zero  indicates success.
	  */
	int selectTable(const char* table);

    private:
	MetaioParseEnvironment* mParseEnv;
    };

    //==================================  inline methods
    inline bool
    MetaIO::is_open(void) const {
	return mParseEnv != 0;
    }

} // namespace xsil

#endif // METTAIOWRAP_HH
