/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//   Shared memory stream class
//
#include <time.h>
#include "oSMbuf.hh"
#include "lsmp_prod.hh"
using namespace std;

#ifdef OSMBUF_TRACE
#define TRACEPOINT(a,b,c) mTrace.tracepoint(a, b, c);
#define RETURNS(a) mTrace.returns(a)
#else
#define TRACEPOINT(a,b,c) 
#define RETURNS(a) (a)
#endif

//======================================  Default constructor.
oSMbuf::oSMbuf(void) 
    : mProducer(0), mBuffer(0), mLength(0)
{}

//======================================  Construct a buffer an open it
oSMbuf::oSMbuf(const char *partition, openmode_type mode)
    : mProducer(0), mBuffer(0), mLength(0)
{
    TRACEPOINT("oSMbuf", 0, mode);    
    open(partition, mode);
}

//======================================  open a producer
oSMbuf*
oSMbuf::open(const char *partition, openmode_type mode) {
    TRACEPOINT("open", 0, mode);
    if ((mode & (ios::in | ios::trunc))) return (oSMbuf*) RETURNS(0);

    //---------------------------------  Create/open a new partition.
    mProducer = new LSMP_PROD(partition);
    if (!mProducer || !mProducer->valid()) return (oSMbuf*)RETURNS(0);
    this->setp(0, 0);
    RETURNS((long)this);
    return this;
}

//======================================  Destructor
oSMbuf::~oSMbuf() {
    TRACEPOINT("~oSMbuf", 0, 0);    
    close();
#ifdef OSMBUF_TRACE
    mTrace.pTrace();
#endif
}

//======================================  Close the output buffer.
void
oSMbuf::close(void) {
    TRACEPOINT("close", 0, 0);    
    if (mProducer) {
	sync();
        delete mProducer;
	mProducer = 0;
    }
}

//======================================  Allocate a buffer
int oSMbuf::doallocate() {
    TRACEPOINT("doallocate", 0, 0);
    //----------------------------------  Make sure a partition is attached.
    if (!mProducer)          return streambuf::traits_type::eof();
    if (!mProducer->valid()) return streambuf::traits_type::eof();

    //----------------------------------  Release a buffer if one is allocated.
    if (mBuffer) {
        mProducer->release(0);
	mBuffer = (char*) 0;
    }

    //---------------------------------  Get a new SM buffer.
    mBuffer = mProducer->get_buffer();
    mLength = mProducer->getBufferLength();
    setb(mBuffer, mBuffer+mLength, 1);
    return 1;
}

//======================================  Release current buffer to consumers.
int 
oSMbuf::overflow(int c) {
    TRACEPOINT("overflow", c, 0);

    //----------------------------------  Make sure a partition is attached.
    if (!mProducer)          return RETURNS(streambuf::traits_type::eof());
    if (!mProducer->valid()) return RETURNS(streambuf::traits_type::eof());

    //----------------------------------  Release a buffer if one is allocated.
    sync();

    //----------------------------------  Get a new SM buffer.
    if (!mBuffer) {
        mBuffer = mProducer->get_buffer();
	mLength = mProducer->getBufferLength();
    }
    this->setp(mBuffer, mBuffer+mLength);
    if (c != streambuf::traits_type::eof()) sputc(c);
    return RETURNS(1);
}

//======================================  Set the event id.
void
oSMbuf::set_id(int evtid) {
    if (mProducer) mProducer->SetID(evtid);
}

//======================================  Synchronize buffer (flush the output)
int oSMbuf::sync(void) {
    TRACEPOINT("sync", 0, 0);
    int Length = out_waiting();
    if (mBuffer && (Length > 0)) {
        mProducer->release(Length);
	mBuffer = (char*) 0;
	mLength = 0;
	this->setp(mBuffer, mBuffer);
    }
    return 0;
}

//======================================  Seek (but ye shant find)
std::streampos 
oSMbuf::seekoff(std::streamoff off, seekdir_type dir, openmode_type mode) {
    TRACEPOINT("seekoff", off, dir);
    if ((mode & ios::in) != 0) return streambuf::traits_type::eof();
    std::streampos inx;
    switch (dir) {
    case ios::beg:
	if (off < 0) return streambuf::traits_type::eof();
	inx = off;
	break;
    case ios::cur:
	if (this->pptr() + off < mBuffer) return streambuf::traits_type::eof();
	inx = (this->pptr() - mBuffer) + off;
	if (inx >= mLength) return streambuf::traits_type::eof();
	break;
    case ios::end:
	if (this->epptr() + off < this->pbase()) return streambuf::traits_type::eof();
	inx = (this->pptr() - mBuffer) + off;	
    default:
	return streambuf::traits_type::eof();
    }
    return seekpos(inx, mode);
}

std::streampos 
oSMbuf::seekpos(std::streampos pos, openmode_type mode) {
    TRACEPOINT("seekpos", pos, mode);
    pbump( long(pos) - long(this->pptr() - this->pbase()));
    return RETURNS(pos);
}

//======================================  Trace point function
#ifdef OSMBUF_TRACE
void 
oSMbuf::trace::pTrace(void) const {
    cout << "Number of traced function calls: " << nTrace << endl;
    int npr = (nTrace < maxTrace) ? nTrace : maxTrace;
    for (int i=0 ; i < npr ; i++) {
        int inx = (nTrace - i - 1) % maxTrace;
	cout << mFuncName[inx] << " " << mArg1[inx] << " " << mArg2[inx] 
	     << endl;
    }
}
#endif
