//
//    Application:  smraw
//
//    Function:     Dump raw from the shared memory.
//
//    Syntax:
//
//       smraw [partition <pname>]
//
//       pname  shared memory partition name
//
//    Revision History:
//
//       5-Jan-1999  J. Zweizig
//          Preliminary draft version.
//
#include <time.h>
#include <iostream>
#include <iomanip>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "lsmp_con.hh"
#include "SigFlag.hh"

//
//    Shared memory consumer
//
LSMP_CON *cons;
int bufid = 0;
int nwmax = 64;

//--> Termination signal flag pointer
SigFlag *term;

//--> Internal entry points.
//
//    smraw_Init     Set up the input and output functions
//    smraw_Halt     Set down.
//    smraw_Read     Read routine.
static bool smraw_Init(int argc, const char **argv);
static void smraw_Halt(void);
static void smraw_Read(void);

//
//    main function. 
//      * Set up. This includes initializing the input and output 
//        files and the exception handling.
//      * Loop over records until the ceiling falls in on us
//      * Wait for notification of a new frame.
//      * Get a shared memory buffer (wait if non available.
//      * Read an input frame from the NDS into the buffer.
//      * When the roof caves in set-down the appropriate resources.
//
int main(int argc, const char **argv) {

    if (smraw_Init(argc, argv)) return 1;
    while (!term->test()) {
        smraw_Read();
    }
    smraw_Halt();
    return 0;}

//
//    smraw_Init - set up the internal environment for pushing data.
//       * Parse any command line parameters.
//       * Create a shared memory partition for data distribution.
//       * Make a connection to the CDS Network Data Server. Register 
//         notification of all frames.
//       * Set up the termination signal handling.
//
static bool smraw_Init(int argc, const char **argv) {
    const char*   parname = getenv("LIGOSMPART");
    if (!parname) parname = "LIGO_Online";

    //---------------------------------  Parse the arguments
    for (int i=1 ; i<argc ; i++) {
        if (!strcmp("partition", argv[i])) {
	    parname = argv[++i];
	} else if (!strcmp("-nwmax", argv[i])) {
	    nwmax = strtol(argv[++i], 0, 0);
	} else {
	    std::cerr << "smraw: Invalid argument " << argv[i] << std::endl;
	    std::cerr << "Syntax:" << std::endl;
	    std::cerr << "smraw [partition <part-name>] [-nwmax <max-wds>]" 
		      << std::endl;
	    return true;
	}
    }

    //---------------------------------  Create a shared memory consumer
    cons = new LSMP_CON(parname);

    if (!cons) return true;
    if (!cons->valid()) {
        std::cout << "Can't open partition " << parname << ": " 
		  << cons->Error() << std::endl;
	return true;}

    //---------------------------------  Set up the signal flags
    term = new SigFlag(SIGINT);
    term->add(SIGTERM);
    term->add(SIGHUP);

    //---------------------------------  Done, quit.
    return false;}

//
//    smraw_Halt - set down.
//       * Release the shared memory partition.
//       * Cancel the termination signal handling.
//
static void 
smraw_Halt(void) {

    //---------------------------------  Disable the signal
    delete term;

    //---------------------------------  Delete the partition
    delete cons;
}

//
//    smraw_Read - read the data.
//      *  Get a buffer
//      *  Dump the contents to stdout.
//      *  Release the buffer.
//
#define TEXTWID 16
static void 
smraw_Read(void) {
    char string[TEXTWID+1];

    //-----------------------------  Read in a frame
    const int *buffer =  (const int*) cons->get_buffer();
    if (!buffer) return;
    int nwd = (cons->getLength() + sizeof(int) - 1)/sizeof(int);
    if (nwd > nwmax) nwd = nwmax;

    //-----------------------------  Dump the frame to stdout.
    std::cout << "Buffer number " << bufid <<  "    Length " 
	      << cons->getLength() << std::endl;
    for (int i=0 ; i<nwd ; i+=TEXTWID/sizeof(int)) {
        for (int j=0 ; j<TEXTWID ; j++) {
	    char tchar = *((char*) (buffer + i) +j);
	    if      (tchar >= 'a' && tchar <= 'z') string[j] = tchar;
	    else if (tchar >= 'A' && tchar <= 'Z') string[j] = tchar;
	    else if (tchar >= '0' && tchar <= '9') string[j] = tchar;
	    else if (tchar == ' ' || tchar == '_') string[j] = tchar;
	    else if (tchar == ':' || tchar == '-') string[j] = tchar;
	    else                                   string[j] = '.';
	}
	string[TEXTWID] = 0;
        std::cout << std::hex 
		  << std::setw(8) << i*sizeof(int) 
		  << std::setfill('0')
		  << "  " << std::setw(8) << buffer[i] 
		  << " "  << std::setw(8) << buffer[i+1] 
		  << " "  << std::setw(8) << buffer[i+2] 
		  << " "  << std::setw(8) << buffer[i+3]
		  << " |" << string << "|" 
		  << std::setfill(' ') << std::endl;
    }
    bufid++;

    //-----------------------------  Delete the frame/release the buffer.
    cons->free_buffer();
    return;
}
