/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "SigFlag.hh"
#include "Interval.hh"
#include "Time.hh"
#include "lsmp_con.hh"
#include <iostream>
#include <iomanip>
#include <string>
#include <cmath>
#include <cstdlib>

using namespace std;

//======================================  Command line syntax
static void
syntax(void) {
    cout << "smlatency measures the latency of frames arriving in a" << endl;
    cout << "specified partition. The time is measured from the gps" << endl;
    cout << "time stamp carried in the partition to the current gps" << endl;
    cout << "time inferred from the system clock. Note that a misset" << endl;
    cout << "system clock will add an unknown offset to the latency" << endl;
    cout << " " << endl;
    cout << "The command line syntax is:" << endl;
    cout << " " << endl;
    cout << "   smlatency [--help] [--average <n> | --noaverage] "
	 << "[--length <len>] " << endl;
    cout << "             [--missing] [--log] <partition>" << endl;
}


class stats {
public:
    stats(int n) : nAvg(n) {
	clear();
    }
    void clear(void) {
	iAvg=0; Avg=0.0; Sig=0.0; Max=0.0;
    }
 
   void print(const string& hdr) {
	double N = iAvg;
	if (!N) N = 1.0;
	double sAvg = Avg / N;
	double sSig = Sig/N - sAvg * sAvg;
	if (sSig < 0) sSig = 0;
	sSig = sqrt(sSig);
	cout << hdr << "n=" << iAvg << "  avg=" << sAvg 
	     << "  sig=" << sSig << "  max=" << Max << endl;
    }

    bool ready( void) const {return iAvg >= nAvg;}

    void add(double x) {
	Avg += x;
	Sig += x*x;
	if (!iAvg) Max = x;
	else if (Max < x) Max = x;
	iAvg++;
    }
private:
    long nAvg, iAvg;
    double Avg, Sig, Max;
};

//======================================  Main program.
int
main(int argc, const char* argv[]) {
    SigFlag mTerm(SIGTERM);
    mTerm.add(SIGINT);

    //----------------------------------  Parse the command args
    int nAvg = 100;
    int len  = 4;
    bool avgopt(true), logopt(false), missopt(false);
    const char* part=getenv("LIGOSMPART");
    for (int i=1; i<argc; i++) {
	string argi = argv[i];
	if (argi == "--average") {
	    nAvg = strtol(argv[++i], 0, 0);
	    avgopt = true;
	}
	else if (argi == "--help") {
	    syntax();
	    return 0;
	}
	else if (argi == "--length") {
	    len = strtol(argv[++i], 0 ,0);
	}
	else if (argi == "--log") {
	    logopt = true;
	}
	else if (argi == "--missing") {
	    missopt = true;
	}
	else if (argi == "--noaverage") {
	    avgopt = true;
	}
	else if (i != argc-1) {
	    cerr << "Unrecognized command argument: " << argi << endl;
	    syntax();
	    return 1;
	}
    }

    //----------------------------------  Check partition name.
    if (argc > 1) part = argv[argc-1];
    if (!part) {
	cerr << "Partition name not specified." << endl;
	syntax();
	return 2;
    }

    //----------------------------------  Connect to the partition.
    LSMP_CON cons(part);
    if (!cons.attached()) {
	cerr << "Unable to open partition: " << part << endl;
	return 3;
    }

    stats  status(nAvg);
    string hdr = "Partition ";
    Time::ulong_t tLast = 0;
    hdr += part;
    hdr += " latency, ";
    while (!mTerm) {
	if (!cons.get_buffer()) continue;
	Time::ulong_t Id = cons.getEvtID();
	cons.free_buffer();
	double dt = double(Now() - Time(Id));

	//------------------------------  Check for missing Id's
	if (missopt && tLast != 0 && Id > tLast+len) {
	    cout << "Missing records: " << tLast+len << "-" << Id << endl;
	}

	//------------------------------  Log if requested
	if (logopt) {
	    cout << Id << " " << fixed << setprecision(4) << dt << endl;
	}

	//------------------------------  Calculate the average
	tLast = Id;
	if (avgopt) {
	    status.add(dt);
	    if (status.ready()) {
		status.print(hdr);
		status.clear();
	    }
	}
    }
    if (avgopt) status.print(hdr);
    return 0;
}
