/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//   LSMP is the Ligo Shared Memory Partition object class
//
#ifndef LSMP_CON_HH
#define LSMP_CON_HH

#include "lsmp.hh"

// #define COLDREAD

/**  The shared memory consumer class allows processes to read data buffers 
  *  as they are made available by the partition manager. At initialization,
  *  each consumer specifies the maximum number of buffers to be reserved
  *  at one time for it to read. Consumers access data by using get_buffer 
  *  and free_buffer.
  *  @memo Shared Memory Consumer Class.
  *  @author John G. Zweizig
  *  @version 1.3; Modified: October 28, 1999 
  *  @ingroup IO_lsmp
  */
class LSMP_CON : public LSMP {

  //------------------------------------  Public data and methods
  public:
    /**  The default constructor creates the consumer data structure without
      *  Attaching it to a partition or allocating a consumer slot.
      *  @memo Default (null consumer) constructor.
      */
    LSMP_CON(void);

    /**  The consumer access constructor finds the specified partition and 
      *  allocates a consumer slot in the partition. If no partition exists
      *  with the given name, a new partition will be created with default
      *  size parameters .he nbuf and mask arguments specify respectively 
      *  the number of buffers to be reserved automatically for the consumer 
      *  and the requisite type-flags for data to be read by the consumer.
      *  @memo Consumer access constructor.
      *  @param part Pointer to character string with the partition name.
      *  @param nbuf Number of buffers to reserve for the new consumer.
      *  @param mask Bit mask indicating event types to be accepted by the
      *              consumer.
      */
    LSMP_CON(const char *part, int nbuf=0, int mask=-1);

    /**  The consumer destructor releases the consumer slot if one was 
      *  allocated. It also releases the partition if no other processes 
      *  are using the partition and the Keep flag is not set.
      *  @memo Consumer destructor.
      */
    ~LSMP_CON(void);

    /**  Get the current buffer address.
      *  @memo Buffer address.
      *  @return Address of the currently allocated buffer.
      */
    const char* getBuffAddr(void) const;

    /**  Get the record ID of the consumer data.
      *  @memo Get current record ID.
      *  @return Identifier of the current data record.
      */
    eventid_type getEvtID(void) const;

    /**  Get the length in bytes of the data in the current consumer buffer.
      *  @memo Get current record length.
      *  @return Number of bytes of data in current buffer.
      */
    int getLength(void) const;

    /**  Get the mask of trigger types requested by this consumer.
      *  \brief Trigger mask.
      *  \return Trigger type mask.
      */
    int getMask(void) const;

    /**  Get the maximum number of buffers to be reserved for the consumer.
      *  \brief Number of buffers to reserve.
      *  \return Number of buffers to reserve.
      */
    int getNBuffer(void) const;

    /**  Get the number of buffers to skip between buffers allocated.
      *  @memo Skip count.
      *  @return Number of buffers to be skipped.
      */
    int getNSkip(void) const;

    /**  Test whether the consumer is connected to the partition i.e.
      *  whether a consumer id has been allocated.
      *  @memo Test for connected consumer
      *  @return true if consumer is connected to the partition.
      */
    bool isConnected(void) const;

    /**  Test whether the consumer's buffer is the latest in the free 
      *  queue
      *  @memo Test for latest buffer
      *  @return true if buffer is latest in free queue
      */
    bool latest_buffer(void) const;

    /**  Get the event ID of the latest buffer in the full queue.
      *  @memo Latest event ID
      *  @return Event ID of latest buffer.
      */
    eventid_type latest_evtID(void) const;

    /**  Test for errors in construction or connection of consumer.
      *  @memo Test for invalid consumer
      *  @return true if consumer is not connected to the partition.
      */
    bool operator!(void) const;

    /**  Sets the maximum number of full buffers that will be reserved for 
      *  the consumer. This overrides the \c nbuf argument of the constructor. 
      *  If "nbuf<0" all buffers will be reserved. If "nbuf=0"
      *  no buffers will be reserved and data will received as it becomes
      *  available. Reserved buffers will not be released until they have 
      *  been read, even if \c nbuf is set to a value less than the current 
      *  number of reserved buffers.
      *  @memo Specify the number of buffers to reserve.
      *  @param nbuf The number of buffers to reserve or -1 for all buffers.
      */
    void setNBuffer(int nbuf);

    /**  Set the number of buffers to skip between reading buffers. The
      *  consumer skips the specified number of otherwise qualifying 
      *  buffers between reading buffers. By default, the skip count is
      *  set to zero so all buffers matching the trigger criteria will
      *  be read. 
      *  @memo Set skip count.
      *  @param nskip Number of buffers to be skipped.
      */
    void setNSkip(int nskip);

    /**  Set the timeout value when waiting for data. By default, the buffer
      *  request will wait until data arrive. If the timeout is set to a
      *  value that is greater than zero, the data request will fail (i.e.
      *  get-buffer will return a NULL pointer) if no data have arrived in 
      *  the specified time limit. The behavior when the NOWAIT flag is set 
      *  is not affected by this value.
      *  @memo Set data timeout time.
      *  @param timeout Timeout value in seconds.
      */
    void setTimeout(double timeout);

    /**  Get a buffer of data and associate it with the 
      *  consumer. The buffer address is returned. By default, the consumer 
      *  will wait for new data if no full and unread buffers are available. 
      *  get_buffer will fail (\e i.e. it will return NULL) in the following
      *  cases:
      *  <ul>
      *   <li> "(flags & NOWAIT) != 0" and no buffers are immediately  
      *        available (errno = EAGAIN) </li>
      *   <li> A system call was interrupted by a signal (errno = EINTR) </li>
      *   <li> The consumer is not attached/registered in a partition </li>
      *   <li> The consumer already has a buffer allocated </li>
      *  </ul>
      *  @memo Allocate a full buffer.
      *  @return Pointer to the start of the allocated data buffer.
      *  @param flags Function modifier flag word.
      */
    const char* get_buffer(int flags=0);

    /**  find_dataID() finds a specific data record identified by the data 
      *  ID and returns the address of the buffer it is in. If the requested 
      *  record isn't currently in the partition, find_dataID() returns NULL.
      *  @memo Find a specific data buffer.
      *  @return pointer to the buffer containing the requested data.
      *  @param ID identified of data record to be found
      */
    const char* find_dataID(eventid_type ID);

    /**  free_buffer() instructs the partition manager to release the data 
      *  buffer accessed by the consumer.
      *  \brief Release a buffer.
      *  \return true if successful
      */
    bool free_buffer(void);

  private:
    //------------------------------------  Private methods
    /**  A consumer slot is allocated and the identifier is stored in icon.
      *  Up to 'nbuf' buffers will be reserved for the consumer at once. If 
      *  nbuf == 0, no buffer will be reserved and buffers will be read in 
      *  an on-demand basis. if nbuf < 0, the consumer is placed in read-all
      *  state whereby all data will be reserved for the consumer.
      *  @memo Allocate a consumer slot. 
      */
    void get_consumer(int nbuf, int mask);

    /**  Look for a specific data block.
      *  @memo Get a specific buffer.
      */
    int  get_by_ID(eventid_type dataID);

    //------------------------------------  Private data members.
    /**  Identifier of this consumer. This is the index into the shared
      *  memory consumer database or -1 if undefined.
      *  @memo Consumer number.
      */
    int icon;

    /**  ID of buffer currently being processed by the consumer.
      *  @memo Current buffer number.
      */
    int ibuf;

    /**  Timeout delay in seconds
      */
    double maxtime;
};

//--------------------------------------  Inline functions
inline const char* 
LSMP_CON::getBuffAddr(void) const {
    return buffer_addr(ibuf);
}

inline int 
LSMP_CON::getLength(void) const {
    return buffer_length(ibuf);
}

inline bool
LSMP_CON::isConnected(void) const {
    return (valid() && icon >= 0);
}

inline bool
LSMP_CON::operator!(void) const {
    return (!valid() || icon < 0);
}

#endif  // LSMP_CON_HH
