#ifndef _BSD_SOURCE
#define _BSD_SOURCE 1
#endif
#include "lmsg/Buffer.hh"
#include "lmsg/BufferPool.hh"
#include "lmsg/MsgAddr.hh"
#include "lmsg/Socket.hh"
#include "lmsg/SocketPool.hh"
#include "lmsg/ErrorList.hh"
#include <cerrno>
#include <fcntl.h>
#include <signal.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <unistd.h>
#include <cstdio> // only for perror
#include <string>
#include <iostream>
#ifdef hpux
#include <sys/ioctl.h>
#endif

using namespace std;

//======================================  Solaris posix sucks!
#include "sockfix.hh"

using namespace lmsg;

//======================================  Socket destructor
Socket::~Socket(void) {
}

//======================================  Constructor
Socket::Socket(void) 
  : mBPool(0), mConnected(false), mSocket(-1), mSPool(0)
{}

//======================================  Bind the socket.
error_type
Socket::bind(const MsgAddr* addr) {
    if (!isOpen()) return NotOpen;

    if (addr) myAddr = *addr;
    else      myAddr = MsgAddr();

    ipsock_t tAddr;
    myAddr.getAddr(tAddr);

    socklen_t len = 16;
    int rc = ::bind(mSocket, (struct sockaddr*)&tAddr, len);
    if (rc < 0) {
	perror("Error in bind");
	return SystemError;
    }	

    rc=getsockname(mSocket, (struct sockaddr*)&tAddr, &len);
    if (rc < 0) {
	perror("Error in getsockname");
	return SystemError;
    }
    myAddr = MsgAddr(tAddr);
    return OK;
}

//======================================  Get address
MsgAddr 
Socket::getAddr(void) const {
    return myAddr;
}

//======================================  Get partner address
MsgAddr
Socket::getPartner(void) const {
    return mPartner;
}

//======================================  Get receive buffer length
int
Socket::getRecvBuf(void) const {
    int len = -1;
    socklen_t vlen = sizeof(len);
    getsockopt(mSocket, SOL_SOCKET, SO_RCVBUF, (char*)&len, &vlen);
    return len;
}

//======================================  Get receive buffer length
int
Socket::getSendBuf(void) const {
    int len = -1;
    socklen_t vlen = sizeof(len);
    getsockopt(mSocket, SOL_SOCKET, SO_SNDBUF, (char*)&len, &vlen);
    return len;
}

//======================================  Enable/disable synchronous I/O
error_type 
Socket::async(bool enable) {
    if (!isOpen()) return NotOpen;

    //----------------------------------  Don't let the SIGIO kill us
    // struct sigaction tmp;
    // tmp.sa_flags   = 0;
    // tmp.sa_handler = SIG_IGN;
    // sigemptyset(&tmp.sa_mask);
    // sigaction(SIGIO, &tmp, NULL);

#ifdef hpux
    if (ioctl(mSocket, FIOASYNC, enable) < 0) return SystemError;
#else
    long fflags = fcntl(mSocket, F_GETFL);
    if (fflags < 0) {
	perror("Error reading flags in async");
	return SystemError;
    }
#ifdef __linux
    if (fcntl(mSocket, F_SETSIG, SIGIO) < 0) {
	perror("Error setting signal in async");
    }
    
    if (enable) fflags |= O_ASYNC;
    else        fflags &= ~O_ASYNC;
#else
    if (enable) fflags |= FASYNC;
    else        fflags &= ~FASYNC;
#endif
    if (fcntl(mSocket, F_SETFL, fflags) < 0) {
	perror("Error setting flags in async");
	return SystemError;
    }
    if (getDebug()) {
      cout << "Socket::async: Asynchronous socket ";
      if (enable) cout << "enabled";
      else        cout << "disabled";
      cout << endl;
    }
#endif
    if (enable) {
        if (fcntl(mSocket, F_SETOWN, getpid()) < 0) {
	    perror("Error setting socket owner ID");
	    return SystemError;
	}
    }
    return OK;
}

//======================================  Connect to a partner
error_type 
Socket::connect(const MsgAddr& addr) {
    ipsock_t tAddr;
    addr.getAddr(tAddr);
    int rc=::connect(mSocket, (struct sockaddr*)&tAddr, sizeof(tAddr));
    if (rc < 0) {
	perror("Error in connect");
	return SystemError;
    }
    if (getDebug()) cout << "Socket: connected to " << addr << endl;
    mConnected = true;
    mPartner = addr;
    return OK;
}

//======================================  Disconnect
error_type 
Socket::disconnect(void) {
    ipsock_t socknam;

    if (!mConnected) return OK;
    socknam.sin_family = AF_UNSPEC;
    int rc=::connect(mSocket, (struct sockaddr *) &socknam, sizeof(socknam));
    if (rc < 0) {
        perror("Error in disconnect");
	return SystemError;
    }
    mConnected = false;
    return OK;
}

//=======================================  Receive to buffer from pool
error_type 
Socket::read(char* data, size_type len) {
    char* ptr  = data;
    int out=100;
    for (int left=len, rc=0 ; left > 0 ; left-=rc) {
        rc = ::read(mSocket, ptr, left);
	if (rc < 0) {
	    perror("Error in read");
	    return SystemError;
	}
	if (rc == 0 && !out--) return SizeError;
	// if (rc < left) cout << left-rc << " Bytes left" << endl;
	ptr  += rc;
    }
    return OK;
}

//=======================================  Receive to buffer from pool
error_type 
Socket::receive(Buffer** buf) {
    if (!mBPool) return NoPool;
    Buffer* b = mBPool->getBuffer();
    if (!b) return NoBuffer;
    index_type rc = receive(b);
    if (rc == OK) *buf = b;
    else          b->Return();
    return rc;
}

//=======================================  Receive to buffer
error_type 
Socket::receive(Buffer* buf) {
    index_type flags=0;
    ipsock_t tAddr;
    socklen_t adlen = sizeof(tAddr);
    int rc = recvfrom(mSocket, buf->getBuffer(), buf->getSize(), flags, 
		      (struct sockaddr*) &tAddr, &adlen);
    if (rc>=0) {
        buf->setLength(rc);
	mPartner = MsgAddr(tAddr);
	rc = OK;
    } else {
	if      (errno == ECONNREFUSED) rc = BadAddress;
	else if (errno == EINTR)        rc = Continue;
	else {
	    perror("Error in recvfrom");
	    rc = SystemError;
	}
    }
    return rc;
}

//========================================  Send to a buffer
error_type 
Socket::send(const MsgAddr& to, Buffer* b) {
    int rc = 0;
    if (getDebug() > 1) cout << "Sending message to " << to << endl;
    if (mConnected && to == mPartner) {
        rc=send(b);
    } else {
	int flags=0;
	int rtry = 2;
	do {
	    mPartner = to;
	    ipsock_t tAddr;
	    mPartner.getAddr(tAddr);
	    rc = ::sendto(mSocket, b->getBuffer(), b->getLength(), flags,
			(struct sockaddr *) &tAddr, sizeof(tAddr));
	} while ((rc < 0) && (errno == ECONNREFUSED) && (--rtry > 0));
	if (rc >= 0) {
	    b->Return();
	    rc = OK;
	} else if (errno == EINTR) {
	    rc = Continue;
	} else {
	    perror("System error in Socket::send");
	    rc = SystemError;
	}
    }
    return rc;
}

//========================================  Send to a buffer
error_type 
Socket::send(Buffer* b) {
    if (!isConnected()) return BadAddress;
    if (getDebug() > 1) cout << "Sending message of length " << b->getLength() 
			     << endl;
    int rc=0;
    int flags=0;
    int rtry = 2;
    do {
        rc=::send(mSocket, b->getBuffer(), b->getLength(), flags);
	// cout << "Sending " << b->getLength() << " bytes." << endl;
    } while ((rc < 0) && (errno == ECONNREFUSED) && (--rtry > 0));
    if (rc >= 0) {
        b->Return();
	rc = OK;
    } else if (errno == EINTR) {
        rc = Continue;
    } else {
        perror("System error in Socket::send");
        rc = SystemError;
    }
    return rc;
}

//======================================  Wait for input.  
error_type 
Socket::wait(wtime_type maxtime) const {
    const Socket* list(this);
    return waitAny(maxtime, 1, &list);
}

error_type 
Socket::waitAny(wtime_type maxtime, int N, const Socket* list[]) {
    if (N <= 0) return NotOpen;

    //----------------------------------  Set up the timer
    struct timeval tspec, *tptr;
    if (maxtime >= 0) {
        tspec.tv_sec  = int(maxtime);
        tspec.tv_usec = int((maxtime-double(tspec.tv_sec))*1000000.0);
        tptr = &tspec;
    } else {
        tptr = reinterpret_cast<struct timeval*>(0);
    }

    //----------------------------------  Set up the socket mask
    fd_set readfds;
    FD_ZERO(&readfds);
    int maxSocket = -1;
    for (int i=0 ; i<N ; i++) {
        const Socket* p = list[i];
        if (!p || !p->isOpen()) continue;
        FD_SET(p->mSocket, &readfds);
	if (p->mSocket > maxSocket) maxSocket = p->mSocket;
    }
    if (maxSocket < 0) return NotOpen;

    //----------------------------------  Wait for something to happen
    int nrdy = select(maxSocket+1, &readfds, 0, 0, tptr);
    if (nrdy == 0) {
        return TimeOut;
    } else if (nrdy < 0) {
        if (errno == EINTR) return Continue;
        else {
	    perror("Error in select");
	    return SystemError;
        }
    }
    return OK;
}

//======================================  Assign a debug level
void
Socket::setDebug(int level) {
    mDebug = level;
}

//======================================  Assign a buffer pool
void
Socket::setPool(BufferPool* pool) {
    mBPool = pool;
}

//======================================  Assign a socket pool
void
Socket::setPool(SocketPool* pool) {
    mSPool = pool;
}

//======================================  Set receive buffer length
error_type
Socket::setRecvBuf(int nBytes) {
    int rc = setsockopt(mSocket, SOL_SOCKET, SO_RCVBUF, &nBytes, 
			sizeof(nBytes));
    if (rc) {
	perror("Error setting socket receive buffer size");
	return SystemError;
    }
    return OK;
}

//======================================  Set send buffer length
error_type
Socket::setSendBuf(int nBytes) {
    int rc = setsockopt(mSocket, SOL_SOCKET, SO_SNDBUF, &nBytes, 
			sizeof(nBytes));
    if (rc) {
	perror("Error setting socket receive buffer size");
	return SystemError;
    }
    return OK;
}

//======================================  Return socket to pool
void
Socket::Return(void) {
    if (mSPool) mSPool->returnSocket(this);
    else        delete this;
}
