/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "html/align.hh"
#include "html/boolAttr.hh"
#include "html/size.hh"
#include "html/table.hh"
#include "html/text.hh"
#include "html/writer.hh"
#include <stdexcept>
#include "PConfig.h"
#include <sstream>

using namespace std;
using namespace html;

//======================================  Construct an empty cell
cell::cell(void) {}

//======================================  Destroy a cell
cell::~cell(void) {}

//======================================  Clone a cell
cell*
cell::clone(void) const {
   return new cell(*this);
}

//======================================  Set the alignment attribute
void
cell::setAlign(const std::string& a) {
   addAttr("align", align(a));
}


//======================================  set the Save attribute
void
cell::setSave(bool a) {
   addAttr("save", boolAttr(a));
}

//======================================  Header cell constructor
header_cell::header_cell(const std::string& title) {
   if (!title.empty()) addObject(text(title));
}

//======================================  Header cell destructor.
header_cell::~header_cell(void) {
}

//======================================  Clone a header cell
header_cell* 
header_cell::clone(void) const {
   return new header_cell(*this);
}

//======================================  Construct an empty table
table::table(void)
   : mRow(0)
{}

//======================================  Construct an empty table
table::table(const string& title) 
   : mTitle(title), mRow(0)
{}

//======================================  Copy a table
table::table(const table& x) 
   : mTitle(x.mTitle), mRow(x.mRow), mCTitle(x.mCTitle), mAttrib(x.mAttrib)
{
   long N = x.mData.size();
   mData.reserve(N);
   for (long i=0 ; i<N; ++i) {
      if (x.mData[i]) mData.push_back(x.mData[i]->clone());
      else            mData.push_back(0);
   }
}

//======================================  Destroy a table
table::~table(void) {
   for (unsigned int i=0 ; i<mData.size() ; i++) {
      if (mData[i]) delete mData[i];
      mData[i] = 0;
   }
}

//======================================  Copy a table
table&
table::operator=(const table& x) {
   mTitle  = x.mTitle;
   mRow    = x.mRow;
   mCTitle = x.mCTitle;
   mData   = x.mData;
   mAttrib = x.mAttrib;
   for (unsigned int i=0 ; i<mData.size() ; i++) {
      if (mData[i]) mData[i] = new cell(*mData[i]);
   }
   return *this;
}


//======================================  Clone a table
table* 
table::clone(void) const {
   return new table(*this);
}

//======================================  Write out the html
void 
table::write(writer& out) const {
   out.tag("blockquote");
   out.tag("center");
   if (!mTitle.empty()) {
      html::text(mTitle).setSize(size(1)).write(out);
      out.endLine();
   }
   out.tag("table", mAttrib);
   out.endLine();

   //----------------------------------  Print out the column headers
   int nCol = getNColumn();
   out.tag("tr");
   for (int i=0 ; i<nCol ; i++) mCTitle[i].write(out);
   out.endTag("tr");
   out.endLine();

   //----------------------------------  Print the column data
   for (int j=0 ; j<mRow ; j++) {
      out.tag("tr");
      int row0 = j*nCol;
      for (int i=0 ; i<nCol ; i++) {
	 if (mData[i+row0]) {
	    mData[i+row0]->write(out);
	 } else {
	    out.tag("td");
	    out.endTag("td");
	 }
      }
      out.endTag("tr");
      out.endLine();
   }

   out.endTag("table");
   out.endTag("center");
   out.endTag("blockquote");
}

//======================================  Add a column
int
table::addColumn(const string& title) {
   mCTitle.push_back(header_cell(title));
   return mCTitle.size()-1;
}

//======================================  Add a row
int
table::addRow(void) {
   for (int i=0 ; i<getNColumn() ; i++) mData.push_back(0);
   return mRow++;
}

//======================================  Erase one or more rows
void 
table::eraseRow(int row, int count) throw(runtime_error)
{
   if (row < 0 || row >= mRow) throw runtime_error("Invalid row number");
   int nCol = getNColumn();
   int maxRow = row + count;
   if (maxRow > mRow) maxRow = mRow;
   int xErase = maxRow*nCol;

   //----------------------------------  Delete existing cells.
   for (int inx=row*nCol ; inx < xErase ; inx++) {
      if (mData[inx]) {
	 delete mData[inx];
	 mData[inx] = 0;
      }
   }

   //----------------------------------  Copy data forward
   int insPtr = row*nCol;
   for (unsigned int inx=xErase ; inx < mData.size() ; inx++) {
      mData[insPtr++] = mData[inx]; 
   }

   //----------------------------------  Wipe out unused entries.
   mData.erase(mData.begin()+insPtr, mData.end());
   mRow -= maxRow - row;
}


//======================================  Insert a number into a cell
void 
table::insertData(int row, int col, double x, int wid, int prec) 
   throw(runtime_error)
{
   ostringstream ss;
   if (wid>0)  ss.width(wid);
   if (prec>0) ss.precision(prec);
   ss << x;
#ifdef __GNU_STDC_OLD
   ss << ends;
#endif
   insertData(row, col, text(ss.str()));
   refCell(row, col).setAlign("right");
}

//======================================  Insert a number into a cell
void 
table::insertData(int row, int col, long x, int wid, int radix) 
   throw(runtime_error)
{
   ostringstream ss;
   if (wid>0)  ss.width(wid);
   switch (radix) {
   case 10:
      break;
   case 16:
      ss << hex;
      break;
   case 8:
      ss << dec;
      break;
   default:
      throw runtime_error("Invalid radix");
   }
   ss << x;
#ifdef __GNU_STDC_OLD
   ss << ends;
#endif
   insertData(row, col, text(ss.str()));
   refCell(row, col).setAlign("right");
}

//======================================  Insert text into a cell
void 
table::insertData(int row, int col, const string& txt) 
   throw(runtime_error)
{
   insertData(row, col, text(txt));
}

//======================================  Insert data into a cell
void 
table::insertData(int row, int col, const object& obj) 
   throw(runtime_error)
{
   int inx = inxrc(row, col);
   if ( mData[inx] == 0 ) mData[inx] = new cell();
   mData[inx]->addObject(obj);
}

//======================================  Insert a row in front of "row"
void 
table::insertRow(int row) throw(runtime_error)
{
   int svEnd = mData.size();
   addRow();
   int insPtr = inxrc(row, 0);
   if (insPtr >= svEnd) return;
   for (int from=svEnd, to=mData.size() ; from>insPtr ; ) {
      mData[--to] = mData[--from];
   }
   for (int inx=0 ; inx<getNColumn() ; inx++) mData[insPtr++] = 0;
}

//======================================  Add/Remove the border attribute
void
table::setBorder(bool On) {
   if (On) mAttrib.addAttr("border", boolAttr());
   else    mAttrib.remAttr("border");
}
