/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef HTML_ALIGN_HH
#define HTML_ALIGN_HH

#include "html/Attrib.hh"
#include <string>

namespace html {
  class writer;

  /**  Text alignment attribute class.
    *  @brief Text alignment attribute class.
    *  @author John Zweizig
    *  @version 1.1; Last modified March 4, 2008
    *  @ingroup IO_html
    */
  class align : public attrib {
  public:
    /**  Construct an alignment attribute object that specifies a default 
      *  alignment.
      *  @brief Default constructor.
      */
    align(void);

    /**  Construct an alignment atrribute object with specified alignment.
      *  Valid arguments and error recovery are as listed under \c setAlign().
      *  @brief Default constructor.
      *  @param clr Alignment specifier.
      */
    align(const std::string& clr);

    /**  Destroy an alignment object.
      *  @brief Destructor.
      */
    ~align(void);

    /**  Make an identical copy of the alignment object.
      *  @brief Clone alignment.
      *  @return Pointer to the cloned alignment object.
      */
    align* clone(void) const;

    /**  Test whether the alignment is the default value.
      *  @return true if default alignment.
      */
    bool isDefault(void) const;

    /**  Write the alignment attribute to the specified html writer.
      *  @brief Write the alignment.
      *  @param w Html writer reference.
      */
    void putAttr(writer& w) const;

    /**  Set the alignment to the default.
      *  @brief Set default alignment.
      */
    void setDefault(void);

    /**  Set the alignment to the specified value. Valid alignment strings 
      *  are \c "center", \c "right", \c "left" and \c "default". If an 
      *  invalid alignment string is specified, \c setAlign() prints an 
      *  error message to \c cerr and sets the alignment to "default".
      *  @brief Set the aligment.
      *  @param name Alignment value string.
      */
    void setAlign(const std::string& name);

    /**  Test whether this alignnment is equal to the specified.
      *  @brief Compare operator.
      *  @param x Alignment object to be compared
      *  @return True if this alignment is equal to the argument.
      */
    bool operator==(const align& x) const;

  private:
    enum AlignMode {
      k_default,
      k_left,
      k_right,
      k_center
    };

  private:
    AlignMode mMode;
  };

}

#endif //  HTML_ALIGN_HH
