#include "memmap/mmap.hh"
#include "framedir.hh"
#include "FrameF.hh"
#include <dirent.h>
#include <iostream>
#include <fstream>
#include <string.h>
#include <fnmatch.h>
#include <stdio.h>
#include <cstdlib>

   using namespace std;

   static string trim (const char* p)
           
   {
      while (isspace (*p)) p++;
      string s = p;
      while ((s.size() > 0) && isspace (s[s.size()-1])) {
         s.erase (s.size() - 1);
      }
      return s;
   }

   string ffData::getFile(count_t N) const
   {
      string s = mFilePrefix;
      if (isFollowed()) {
         char buf[128];
	 int dt = getDt();
	 sprintf (buf, "%lu-%i", getStartGPS() + N * dt, dt);
	 s += buf;
	 s += mFileSuffix;
      }
      else if (N != 0) {
         s = "";
      }
      return s;
   }


   bool ffDataSeries::joinable (const ffDataSeries& series) const
   {
      return isValid() && series.isValid() &&
             isFollowed() && series.isFollowed() &&
             (strcmp (getPrefix(), series.getPrefix()) == 0) &&
             (strcmp (getSuffix(), series.getSuffix()) == 0) &&
             Almost (getEndTime(), series.getStartTime()) &&
             getDt() == series.getDt();
   }
   
   bool ffDataSeries::join (const ffDataSeries& series)
   {
      if (!joinable (series)) {
         return false;
      }
      mNFiles += series.getNFiles();
      return true;
   }
   
   bool ffDataSeries::join (const char* Prefix, const char* Suffix, 
            const Time& time, Interval Dt)
   {
      ffDataSeries series (Prefix, Suffix, time, Dt);
      return join (series);
   }


   void ffDataConstIter::add (int delta)
   {
      if (!mList || (delta == 0)) {
         return;
      }
      if (delta > 0) {
         mNFiles += delta;
	 // check if we need to change series
	 if (mNFiles < mIndex->second.getNFiles()) {
	    mData.advance (delta);
	    return;
	 }
	 // advance mIndex
	 while((mIndex != mList->end()) &&
	       (mNFiles >= mIndex->second.getNFiles())) {
	    mNFiles -= mIndex->second.getNFiles();
	    ++mIndex;
	 }
      }
      else {
         delta = -delta; // don't go crazy
	 // check if we need to change series
	 if (delta <= (int)mNFiles) {
            mNFiles -= delta;
	    mData.advance (-delta);	 
	    return;
	 }
	 delta -= mNFiles;
	 mNFiles = 0;
	 // back on mIndex
	 while (mIndex != mList->begin()) {
	    --mIndex;
	    if (delta <= (int)mIndex->second.getNFiles()) {
	       mNFiles = mIndex->second.getNFiles() - delta;
	       break;
	    }
	    delta -= mIndex->second.getNFiles();
	 }
      }
      set();
   }
   
   void ffDataConstIter::set (void)
   {
      if (mIndex != mList->end()) {
         mData = mIndex->second;
	 mData.advance (mNFiles);
      }
      else {
         mData = ffData();
	 mNFiles = 0;
      }
   }
   
   
   FrameDir::~FrameDir(void) {
      clear();
   }

   FrameDir::FrameDir(void) 
   : mDebug(0), mDirty (false)
   {
      last_insert = mList.end();
   }

   FrameDir::FrameDir(const char* dir, bool delayed) 
   : mDebug(0), mDirty (false)
   {
      last_insert = mList.end();
      add(dir, delayed);
   }

   FrameDir::file_iterator
   FrameDir::find(const Time& t) const throw(NoData) {
      if (mDirty) checkData();
      series_iterator iter = mList.upper_bound(t.getS());
      if (iter == mList.begin()) throw NoData("Specified data not available");
      --iter;
      ffDataConstIter::count_t n = ffDataConstIter::count_t
         ((double(t.getS() - iter->first) + 0.5) / iter->second.getDt());
      if (n >= iter->second.getNFiles()) {
         throw NoData("Specified data not available");
      }		  
      return ffDataConstIter (mList, iter, n);
   }

   FrameDir::file_iterator 
   FrameDir::getStart(gps_t time) const {
      if (mDirty) checkData();
      series_iterator iter = mList.lower_bound(time);
      if (iter == mList.end()) {
         return end();
      }
      ffDataConstIter::count_t n = 0;
      if (iter != mList.begin()) {
         series_iterator prev = iter;
	 --prev;
	 if (time < prev->second.getEndGPS()) {
	    iter = prev;
            n = ffDataConstIter::count_t((double(time - prev->first) + 0.5) / 
                                          prev->second.getDt());
	 }
      }
      return file_iterator (mList, iter, n);
   }

   FrameDir::file_iterator
   FrameDir::getLast(gps_t time) const {
      if (mDirty) checkData();
      series_iterator iter = mList.upper_bound(time);
      if (iter == mList.begin()) 
         return begin();
      ffDataConstIter::count_t n = 0;
      series_iterator prev = iter;
      --prev;
      if (time < prev->second.getEndGPS()) {
         n = ffDataConstIter::count_t((double(time - prev->first) + 0.5) / 
	                               prev->second.getDt());
         if (time > prev->second.getGPS (n)) ++n; 
         if (n < prev->second.getNFiles()) {
	    iter = prev;
	 }
	 else {
	    n = 0;
	 }
      }
      return file_iterator (mList, iter, n);
   }

   FrameDir::file_iterator
   FrameDir::begin(void) const {
      if (mDirty) checkData();
      return ffDataConstIter (mList, mList.begin(), 0);
   }

   FrameDir::file_iterator
   FrameDir::end(void) const {
      if (mDirty) checkData();
      return ffDataConstIter (mList, mList.end(), 0);
   }

   FrameDir::series_iterator
   FrameDir::beginSeries(void) const {
      if (mDirty) checkData();
      return mList.begin();
   }

   FrameDir::series_iterator
   FrameDir::endSeries(void) const {
      if (mDirty) checkData();
      return mList.end();
   }

   int 
   FrameDir::sizeSeries() const 
   {
      if (mDirty) checkData();
      return mList.size();
   }

   int 
   FrameDir::size() const {
      int sz= 0;
      for (series_iterator iter = mList.begin(); iter != mList.end(); ++iter) {
         sz += iter->second.getNFiles();
      }
      return sz;
   }
	 
   void
   FrameDir::add(const char* dir, bool delayed) {
      if (!dir || !*dir) 
         return;
      string entry(dir);
   
    //----------------------------------  Add File if no wildcards.
    // cerr << "Directory: " << entry << endl;
      string::size_type wcpos = entry.find_first_of("*[?");
      if (wcpos == string::npos) {
         addFile(entry.c_str());
         if (mDirty && !delayed) checkData();
      } 
      else {
      
         //------------------------------  Find the first name level with a *
	 string::size_type last=0, next=0;
         while (next <= wcpos) {
            last = next;
            next = entry.substr(last).find("/");
            if (next == string::npos) next = entry.length()-last;
            next += last + 1;
         }
      
         string direc(entry.substr(0,last));
         if (direc.empty()) direc = ".";
      
      //----------------------------------  Open the directory.
         DIR* dd = opendir(direc.c_str());
         if (!dd) {
            cerr << "Directory " << direc << " is unknown" << endl;
            return;
         }
      
      //------------------------------  Scan for files matching the pattern
         string pattern=entry.substr(last, next-1-last);
      // cerr << "  Test against: " << pattern << endl;
         bool atomic;
         if (next >= entry.length()) {
            atomic = true;
            wcpos  = next;
         } 
         else {
            wcpos  = entry.substr(next).find_first_of("*[?");
            if (wcpos != string::npos) wcpos += next;
            atomic = (wcpos == string::npos);
         }
         for (dirent* dirt=readdir(dd) ; dirt ; dirt = readdir(dd)) {
            if (!fnmatch(pattern.c_str(), dirt->d_name, 0)) {
               string path(entry);
               path.replace(last, next-last-1, dirt->d_name);
            // cerr << "  testing: " << path << endl;
               if (atomic)  addFile(path.c_str());
               else         add(path.c_str(), true);
            }
         }
         closedir(dd);
         if (mDirty && !delayed) checkData();
      }
   }

//=======================================  Add another frame file
   void
   FrameDir::addFile(const char* File, unsigned int Cont) throw(BadFile) {
      if (!File || !*File) 
         return;
      if (getDebug()) cerr << "Adding file: " << File << endl;
   
    //----------------------------------  Check if list of frame names
      
      if ((strlen (File) >= 4) &&
	  (strcmp (File + strlen (File) - 4, ".udn") == 0)) {
          if (!read (File)) {
 	     if (getDebug()) cerr << "File name not standard: " << File << endl;
	  }
          return;
      }

    //----------------------------------  Parse the file name (time and length)

      gps_t time;
      gps_t tlen = 0;
      char prefix[16*1048];
      char suffix[16*1048];
      if (!parseName (File, time, tlen, prefix, suffix)) {
          if (getDebug()) cerr << "File name not standard: " << File << endl;
          return;
      }
      if (tlen != 0) {
         ffDataSeries data (prefix, suffix, Time(time, 0), Interval(tlen), 
	                    Cont + 1);
         if ((last_insert == mList.end()) || 
	     !last_insert->second.join (data)) {
	    last_insert = mList.insert (last_insert, 
	          dmap_t::value_type (time, data));
	 }
      }
      else {
         mList[time] = ffData(File, Time(time, 0), Interval(tlen), false);
      }
      if (getDebug()) cerr << "Found time: " << time 
			   << " nSec: " << tlen << endl;
      mDirty = true;
   }

/*   void
   FrameDir::remove(const char* dir) {
      for (file_iterator iter=begin() ; iter != end() ; ) {
         const ffData& d = iter->second;
         file_iterator next = iter;
         next++;
         if (!fnmatch(dir, d.getFile().c_str(), 0)) {
            erase(iter->first);
         }
         iter = next;
      }
   }
*/
   void
   FrameDir::clear(void) {
      mList.clear();
      last_insert = mList.end();
      mDirty = false;
   }

   void
   FrameDir::erase(gps_t time) {
      mList.erase(time);
      last_insert = mList.end();
   }

   bool 
   FrameDir::parseName (const char* File, 
                        gps_t& time, gps_t& tlen,
                        char* pre, char* post) {
      const char* ptr = File;
      for (const char* p=ptr ; *p ; p++) if (*p == '/') ptr = p+1;
      while (*ptr && *ptr != '-') ptr++;
      if (!*ptr++) {
	  return false;
      }
      // check for second non-numeric field
      const char* ptr2 = ptr;
      if (!isdigit (*ptr2)) {
         while (*ptr2 && *ptr2 != '-') ptr2++;
         if (!*ptr2++) {
	    return false;
         }
	 ptr = ptr2;
      }
      if (pre) {
         strncpy (pre, File, ptr - File);
	 pre[ptr - File] = 0;
      }
      time = strtol(ptr, (char**)&ptr, 10);
      tlen = 0;
      if (*ptr == '-') {
	  ptr++;
	  tlen = strtol(ptr, (char**)&ptr, 10);
      }
      if (!time || (*ptr && *ptr != '.')) {
	  return false;
      }
      if (post) {
         strcpy (post, ptr);
      }
      return true;
   }
			     
   bool 
   FrameDir::read (const char* File)
   {
      if (!File || !*File) {
         return false;
      }
      gdsbase::mmap mm (File, std::ios_base::in);
      if (!mm) {
         return false;
      }
      const char* p = (const char*) mm.get();
      // read file
      string n;
      n.reserve (1024);
      int errnum = 0;
      for (int i = 0; i < (int)mm.size() && errnum < 10; i++) {
         if (p[i] == '\n') {
            n = trim (n.c_str());
            if (!n.empty() && (n[0] != '#')) {
               // this is a new file
	       if (strncmp (n.c_str(), "file://", 7) == 0) {
	          n.erase (0, 7);
	       }
	       string::size_type pos = n.find ("-c");
	       int cont = 0;
	       if (pos != string::npos) {
	          cont = atoi (n.c_str() + pos + 2);
		  n.erase (pos);
		  n = trim (n.c_str());
	       }
	       if ((n.find_first_of("*[?") != string::npos) ||
	           (n.find (".udn") != string::npos)) {
                  add (n.c_str());
	       }
	       else {
	          dmap_t::iterator old_last = last_insert;
		  addFile (n.c_str());
		  if ((cont > 0) && (last_insert != old_last) &&
		      (last_insert != mList.end()) &&
		      (last_insert->second.getDt() > Interval (0))) {
		     last_insert->second.setNFiles (cont + 1);
		  }
	       }
            }
            n = "";
         }
         else if (isprint (p[i])) {
            n += p[i];
         }
         else if ((p[i] & 0x7F) != 0) {
            ++errnum;
         }
      }
      return (errnum == 0);
   }

   bool 
   FrameDir::write (const char* File, bool Longlist, gps_t Start, gps_t Stop,
                    bool SupHead)
   {
      if (((Stop > 0) && (Stop < Start)) || !File || !*File) {
         return false;
      }
      if (mDirty) checkData();
      ofstream fout (File);
      if (!fout) {
         return false;
      }
      return write (fout, Longlist, Start, Stop, SupHead);
   }

   bool 
   FrameDir::write (std::ostream& fout, bool Longlist, gps_t Start, gps_t Stop,
                    bool SupHead)
   {
      if ((Stop > 0) && (Stop < Start)) {
         return false;
      }
      if (Longlist) {
         FrameDir::file_iterator last = (Stop == 0) ? end() : getLast (Stop);
         for (FrameDir::file_iterator i = getStart (Start); i != last; ++i) {
            if (!SupHead) {
               fout << "file://";
            }
            fout << i->getFile() << endl;
         }
      }
      else {
         for (FrameDir::series_iterator i = beginSeries();
              i != endSeries(); ++i) {
            if ((i->second.getEndGPS() <= Start) || 
	        ((Stop > 0)  && (i->second.getStartGPS() >= Stop))) {
               continue;
            }
            ffData::count_t m = 0;
            int n = i->second.getNFiles();
            if (i->second.getStartGPS() < Start) {
               m = ffData::count_t(
	          (double (Start - i->second.getStartGPS()) + 0.5) / 
	           i->second.getDt());
               n -= m;
            }
            if ((Stop > 0) && (i->second.getEndGPS() > Stop)) {
               int d = int((double (Stop - i->second.getStartGPS()) - 0.5) / 
                  i->second.getDt() + 1);
               n -= (int)i->second.getNFiles() - d;
            }
            if (n == 0) {
               continue;
            }
            if (!SupHead) {
               fout << "file://";
            }
            fout << i->second.getFile (m);
            if (n > 1) {
               fout << " -c " << n - 1;
            }
            fout << endl;
         }
      }
      return true;
   }

   std::ostream& 
   FrameDir::operator<< (std::ostream& Os)
   {
      write (Os);
      return Os;
   }

   void
   FrameDir::checkData(CheckLevel lvl) {
      mDirty = false;
      if (getDebug()) cerr << "check Data" << endl;

    //----------------------------------  Make sure there's data to process.
      if (begin() == end()) 
         return;
   
    //----------------------------------  Swich on mode.
      /*ffData::count_t lastNFrame(1);*/
      Interval        lastDt(1.0);
      switch(lvl) {
         case none:
            break;
      
      //----------------------------------  Try to fill time gaps;
         case gapsOnly:
            for (series_iterator iter=mList.begin() ; iter != mList.end() ; ) {
            
            //--------------------------  Access current and next entries
               const ffDataSeries& d = iter->second;
               series_iterator next = iter;
               ++next;
               if (next == mList.end()) 
                  break;
	       if (d.getDt() > Interval (0.0)) {
	          iter = next;
	          continue;
	       }
            //--------------------------  Fill in info if there's a gap
               if (!Almost(d.getEndTime(), next->second.getStartTime())) {
               
               //----------------------  Is gap same as last?
                  Interval Delta=next->second.getStartTime() - d.getStartTime();
                  if (Delta == lastDt) {
                     if (getDebug()) {
                        cerr << "Inferring file: " << d.getFile()
                           << " parameters from previous file." << endl;
                     }
                     mList[iter->first]=ffData(d.getFile().c_str(), d.getStartTime(),
                                           lastDt, true);
                  
                  //----------------------  Read nFrames, length from file.
                  } 
                  else {
                     if (getDebug()) {
                        cerr << "Getting parameters from file: " 
                           << d.getFile() << "." << endl;
                     }
                     try {
                        ffData newFdata = getFileData(d.getFile().c_str());
                        mList[iter->first] = newFdata;
                        lastDt     = newFdata.getDt();
                        if (getDebug()) {
                           cerr << "lastDt = " << lastDt << endl;
                        }
                     } 
                        catch (BadFile& b) {
                           if (getDebug()) {
                              cerr << "Exception: " << b.what() 
                                 << " caught while processing file " 
                                 << d.getFile() << endl;
                           }
                           mList.erase(iter->first);
                        }
                  }
               }
               iter = next;
            }
            break;
      
         case allData:
            for (series_iterator iter=mList.begin() ; iter != mList.end() ; ) {
               const ffData& d = iter->second;
               series_iterator next = iter;
               ++next;
	       if (d.getDt() > Interval (0.0)) {
	          iter = next;
	          continue;
	       }
               if (!d.isValid()) {
                  try {
                     mList[iter->first] = getFileData(d.getFile().c_str());
                  } 
                     catch (BadFile& b) {
                        if (getDebug()) {
                           cerr << "Exception: " << b.what() 
                              << " caught while processing file " 
                              << d.getFile() << endl;
                        }
                        mList.erase(iter->first);
                     }
               }
               iter = next;
            }
            break;
      }
      join();
   }

   void
   FrameDir::checkData(CheckLevel lvl) const {
      const_cast<FrameDir*>(this)->checkData();
   }

   void
   FrameDir::join() {
      for (dmap_t::iterator iter=mList.begin() ; iter != mList.end() ; ) {
         dmap_t::iterator next = iter;
         ++next;
	 if (next == mList.end()) break;
	 if (iter->second.join (next->second)) {
	    mList.erase(next);
	 }
	 else {
  	    iter = next;
	 }
      }
      last_insert = mList.end();
   }
   
   ffData
   FrameDir::getFileData(const char* File) throw (BadFile) {
      //ffData::count_t NFrame(1);
      Time Start(0);
      Interval Dt(0.0);
   
    //----------------------------------  Read in the file header
      ifstream s(File, ios::in);
      FrameF f(s);
      if (!f.isOK()) throw BadFile("Unable to open File");
   
    //----------------------------------  Find the FrHeader structure type
      short IDFrHeader(0);
      while (!IDFrHeader && f.NxStruct()) {
         if (f.getID() == 1) {
            string StructName = f.getString();
            if (StructName == "FrameH") IDFrHeader = f.getShort();
         }
      }
      if (!IDFrHeader) throw BadFile("No FrameH definition");
   
    //----------------------------------  Look for the first header
      while (f.NxStruct() && f.getID() != IDFrHeader);
      if (f.getID() != IDFrHeader) throw BadFile("Can't find a frame header");
   
    //----------------------------------  Get the Start time and length
      f.getString();
      if (f.getVersion() >= 4) f.Skip(3*sizeof(FrameF::int4_t));
      else                     f.Skip(2*sizeof(FrameF::int4_t));
      Start.setS(f.getInt());
      Start.setN(f.getInt());
      f.Skip(sizeof(FrameF::int2_t));
      if (f.getVersion() < 6) f.Skip(sizeof(FrameF::int4_t));
      Dt = Interval(f.getDouble());
   
    //----------------------------------  Get the number of frames from FrEOF
//       int eofsize = 0;
//       if (f.getVersion() >= 6) { 
// 	  eofsize = 3 * 8 + // int_8 length, nBytes, offset
// 	            4 * 4 + // int_4 instance, nFrames, chkFlag, chkSum
// 	            1 * 2;  // int_2 structID
//       } else if (f.getVersion() >= 4) {
// 	  eofsize = 6 * 4 + // int_4 length, nBytes, offset, nFrames, 
// 	                    //       chkFlag, chkSum
// 	            2 * 2;  // int_2 structID, instance
//       } else {
// 	  eofsize = 5 * 4 + // int_4 length, nBytes, nFrames, chkFlag, chkSum
// 	            2 * 2;  // int_2 structID, instance
//       }
//       f.Seek(-eofsize, ios::end);
//       if (f.NxStruct() && f.getLength() == eofsize) {
//          NFrame = f.getInt();
//          if (!NFrame) NFrame = 1;
//       } 
//       else {
//          cerr << "End of File structure not found!" << endl;
//       }
   
      s.close();
      return ffData(File, Start, Dt, true);
   }
