/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef FILE_STREAM_HH
#define FILE_STREAM_HH

#include <list>
#include <vector>
#include <string>

/**  The file stream class maintains a list of files that match a given 
  *  pattern and uses iNotify to update the list with newly arriving frames.
  *  \brief Keep a fifo list of file paths matchin a pattern.
  */ 
class file_stream {
public:
   typedef unsigned long gps_type;
public:
   /**  Default constructor for inactive file stream.
     *  \brief Default constructor.
     */
   file_stream(void);

   /**  Construct a file_stream list.
     */
   explicit file_stream(const std::string& pattern, gps_type tStart=0);

   /**  Destructor for inactive file stream.
     *  \brief Destroy a %file_stream.
     */
   ~file_stream(void);

   /** Loop over file directoies. Add those that match the pattern to the list.
    */
   void add_files(gps_type tStart=0);

   /**  Open an inotify unit and specify the inotify watch values. The 
     *  pattern must be setbefore calling iwatch. 
     *  \brief set up the inotify unit
     *  \returns true if inotify was successfully opened.
     */
   bool add_watch(void);

   /**  Test whether the inotify file device is open.
     *  \returns True if inotify is running
     */
   bool is_open(void) const;

   /**  Pop a file path off of the front of the file path list.
     *  \brief Get the next file path.
     *  \returns File path string.
     */
   std::string next_file(void);

   /**  Create the notify stack from the file path pattern.
     *  \brief Parse the file path pattern.
     *  \param path The file path template used to spcify the notify elements
     */ 
   void parse_pattern(const std::string& path);

   /**  Set the debug print level.
     *  \brief Set the debug print level.
     *  \param lvl Debug level to be set.
     */
   void set_debug(int lvl);

   /**  Test whether the fifo list contains one or more file paths.
     *  \brief test test for an available file.
     *  \return True if there is at least one file available.
     */
   bool test_file(void);

   /**  Wait until a file path is available in the list ot an error is 
     *  discovered in the inotify device file.
     *  \brief Wait for an input file.
     *  \returns true if a file is available in the list. false on error. 
     */
   bool wait_file(void);
private:
   void insert_file(const std::string& pattern);
   void fetch_files(size_t pos, const std::string& path, gps_type t=0);
   bool match_pattern(const std::string& name, const std::string& pattern, 
		      gps_type t=0) const;
   bool open_notify(void);
   void pop_event(void);
   void push_event(const std::string& pattern);
   void replace_event(const std::string& pattern);
   bool read_event(void);
   bool test_event(void) const;
   /**  Wait up to \a tWait seconds for an inotify event to be available. 
     *  wait_event returns 1 if an event is available, zero if the time 
     *  elapses without an event arriving, and \<0 if an error occurs while 
     *  while waiting for the event. On error, \c errno will be set with the 
     *  appropriate error code.
     *  \brief wait for an inotify event.
     *  \param tWait maximum wait time in seconds.
     *  \returns status return.
     */
   int  wait_event(double tWait) const;
private:
   int  mNotifyFd;
   int  debug;
   typedef std::pair<std::string, std::string::size_type> file_list_node;
   typedef std::list<file_list_node> file_list_type;
   typedef file_list_type::iterator file_list_iter;
   file_list_type file_list;
   
   std::vector<std::string> pattern_list;

   typedef std::pair<std::string, int>  event_list_node;
   typedef std::vector<event_list_node> event_list_type;
   typedef event_list_type::iterator    event_list_iter;
   event_list_type event_list;
};

//======================================  Inline methods
inline bool
file_stream::is_open(void) const {
   return mNotifyFd >= 0;
}
#endif // !defined(FILE_STREAM_HH)
